package org.bidib.wizard.mvc.main.controller;

import java.awt.Point;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.AddressTypeEnum;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.EnrailmentDirectionEnum;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.PomProgrammerControllerFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.main.controller.listener.GlobalDetectorPanelListener;
import org.bidib.wizard.mvc.main.model.GlobalDetectorModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.GlobalDetectorPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.pom.controller.PomProgrammerController;
import org.bidib.wizard.mvc.pom.controller.listener.PomProgrammerControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.vlsolutions.swing.docking.DockingDesktop;

public class GlobalDetectorPanelController implements GlobalDetectorPanelListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalDetectorPanelController.class);

    private final MainModel mainModel;

    private final GlobalDetectorModel globalDetectorModel;

    private final NodeListListener nodeListListener;

    private GlobalDetectorPanel globalDetectorPanel;

    @Autowired
    private CommandStationService commandStationService;

    private final TabVisibilityListener tabVisibilityListener;

    public GlobalDetectorPanelController(final MainModel mainModel, final TabVisibilityListener tabVisibilityListener) {
        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;

        globalDetectorModel = new GlobalDetectorModel();

        nodeListListener = new DefaultNodeListListener() {
            @Override
            public void nodeChanged(NodeInterface node) {
                LOGGER.debug("The node has changed.");

                NodeInterface selectedNode = mainModel.getSelectedNode();
                // update the selected node
                globalDetectorModel.setSelectedNode(selectedNode);

                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        LOGGER.debug("Notify that the selected node was changed.");
                        if (globalDetectorPanel != null) {
                            globalDetectorPanel.listChanged();
                        }
                        else {
                            LOGGER.warn("The globalDetectorPanel is not available.");
                        }
                    }
                });
            }
        };
        this.mainModel.addNodeListListener(nodeListListener);

    }

    @Override
    public NodeInterface getSelectedNode() {
        return globalDetectorModel.getSelectedNode();
    }

    public GlobalDetectorPanel createGlobalDetectorPanel() {
        LOGGER.debug("Create new GlobalDetectorPanel.");
        globalDetectorPanel = new GlobalDetectorPanel(this, globalDetectorModel, mainModel, tabVisibilityListener);

        return globalDetectorPanel;
    }

    @Override
    public void openLocoDialog(FeedbackAddressData addressData) {
        LOGGER.info("Open the loco controller dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node = findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {
            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final LocoControllerFactory locoControllerFactory = ctx.getBean(LocoControllerFactory.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node,
                        (JFrame) JOptionPane.getFrameForComponent(globalDetectorPanel.getComponent()),
                        mainModel.getNodeProvider());

            AddressData initialAddress = null;
            if (addressData != null) {
                final EnrailmentDirectionEnum enrailmentDirection = addressData.getType();
                AddressTypeEnum addressType = null;
                switch (enrailmentDirection) {
                    case LOCOMOTIVE_LEFT:
                    case LOCOMOTIVE_RIGHT:
                        addressType = AddressTypeEnum.LOCOMOTIVE_FORWARD;
                        break;
                    case BASIC_ACCESSORY:
                        addressType = AddressTypeEnum.ACCESSORY;
                        break;
                    case EXTENDED_ACCESSORY:
                        addressType = AddressTypeEnum.EXTENDED_ACCESSORY;
                        break;
                    default:
                        break;
                }
                initialAddress = new AddressData(addressData.getAddress(), addressType);
            }
            locoController.start(initialAddress, null);
        }
    }

    @Override
    public void openPomDialog(FeedbackAddressData addressData) {
        LOGGER.info("Open the POM dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node = findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final PomProgrammerControllerFactory pomProgrammerControllerFactory =
                ctx.getBean(PomProgrammerControllerFactory.class);
            final PomProgrammerController pomProgrammerController =
                pomProgrammerControllerFactory
                    .createPomProgrammerController(node,
                        (JFrame) JOptionPane.getFrameForComponent(globalDetectorPanel.getComponent()), new Point(0, 0));

            pomProgrammerController.addPomProgrammerControllerListener(new PomProgrammerControllerListener() {

                @Override
                public void sendRequest(
                    CommandStationNodeInterface node, PomAddressData locoAddress, CommandStationPom opCode,
                    int cvNumber, int cvValue) {
                    LOGGER.info("Send POM request.");

                    PomAcknowledge pomAck =
                        commandStationService
                            .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node, locoAddress, opCode,
                                cvNumber, cvValue);
                    LOGGER.info("Received pomAck: {}", pomAck);
                }

                @Override
                public void close() {
                }
            });

            AddressData initialAddress = null;
            if (addressData != null) {
                final EnrailmentDirectionEnum enrailmentDirection = addressData.getType();
                AddressTypeEnum addressType = null;
                switch (enrailmentDirection) {
                    case LOCOMOTIVE_LEFT:
                    case LOCOMOTIVE_RIGHT:
                        addressType = AddressTypeEnum.LOCOMOTIVE_FORWARD;
                        break;
                    case BASIC_ACCESSORY:
                        addressType = AddressTypeEnum.ACCESSORY;
                        break;
                    case EXTENDED_ACCESSORY:
                        addressType = AddressTypeEnum.EXTENDED_ACCESSORY;
                        break;
                    default:
                        break;
                }
                initialAddress = new AddressData(addressData.getAddress(), addressType);
            }

            final DockingDesktop desktop =
                DefaultApplicationContext
                    .getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);
            // open the dialog
            pomProgrammerController.start(desktop, initialAddress);
        }
    }

    private CommandStationNodeInterface findFirstCommandStationNode(Iterable<NodeInterface> nodes) {
        NodeInterface node = IterableUtils.find(nodes, new Predicate<NodeInterface>() {

            @Override
            public boolean evaluate(final NodeInterface node) {
                if (node.getCommandStationNode() != null && NodeUtils.hasCommandStationFunctions(node.getUniqueId())
                    && !(ProductUtils.isRFBasisNode(node.getUniqueId())
                        || ProductUtils.isSpeedometer(node.getUniqueId()))) {
                    LOGGER.debug("Found command station node: {}", node);
                    return true;
                }
                return false;
            }
        });
        return node != null ? node.getCommandStationNode() : null;
    }
}
