package org.bidib.wizard.mvc.main.controller;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.MapUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.SwitchingNodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.SoundPortListener;
import org.bidib.wizard.api.script.ScriptCommand;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.script.DefaultScriptContext;
import org.bidib.wizard.common.script.common.WaitCommand;
import org.bidib.wizard.common.script.engine.ScriptEngine;
import org.bidib.wizard.common.script.switching.PortScripting;
import org.bidib.wizard.common.script.switching.SoundPortCommand;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.PortTypeAware;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.SoundPortStatus;
import org.bidib.wizard.mvc.main.controller.wrapper.NodePortWrapper;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SoundPortTableModel;
import org.bidib.wizard.mvc.main.model.listener.SoundPortModelListener;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.SoundPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class SoundPortPanelController implements PortScripting {

    private static final Logger LOGGER = LoggerFactory.getLogger(SoundPortPanelController.class);

    private final MainModel mainModel;

    private SoundPortListPanel soundPortListPanel;

    private final Map<NodeInterface, NodePortWrapper> testToggleRegistry = new LinkedHashMap<>();

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private ConsoleService consoleService;

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject = PublishSubject.create();

    private CompositeDisposable compDisp;

    public SoundPortPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        this.compDisp = new CompositeDisposable();
    }

    public SoundPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final SoundPortTableModel tableModel = new SoundPortTableModel();

        tableModel.setPortListener(new SoundPortModelListener() {

            @Override
            public void labelChanged(SoundPort port, String label) {
                port.setLabel(label);

                try {
                    final NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.soundPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save switch port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void configChanged(SoundPort port, PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port value are changed for port: {}, portConfigKeys: {}", port, portConfigKeys);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                // we must get the configured port
                final List<SoundPort> soundPorts = mainModel.getSelectedNode().getSoundPorts();
                final SoundPort soundPort = PortListUtils.findPortByPortNumber(soundPorts, port.getId());

                for (PortConfigKeys key : portConfigKeys) {

                    if (!soundPort.isPortConfigKeySupported(key)) {
                        LOGGER.info("Unsupported port config key detected: {}", key);
                        continue;
                    }

                    switch (key) {
                        case BIDIB_PCFG_TICKS:
                            int pulseTime = port.getPulseTime();
                            values
                                .put(BidibLibrary.BIDIB_PCFG_TICKS,
                                    new BytePortConfigValue(ByteUtils.getLowByte(pulseTime)));
                            break;
                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        LOGGER.info("Set the port params: {}", values);
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the soundPort parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }
            }

            @Override
            public void changePortType(LcOutputType portType, SoundPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);
            }

            @Override
            public void testButtonPressed(SoundPort port, SoundPortStatus newStatus) {
                LOGGER.info("The test button was pressed for port: {}", port);

                NodeInterface node = mainModel.getSelectedNode();
                if (SoundPortStatus.TEST != port.getStatus()) {
                    stopTestToggleTask(node, port);

                    SoundPort soundPort = port;
                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), soundPort);
                }
                else {
                    addTestToggleTask(node, port);
                }
            }

        });

        final SoundPortListPanel soundPortListPanel =
            new SoundPortListPanel(this, tableModel, mainModel, tabVisibilityListener, portConfigChangeEventSubject);

        soundPortListPanel.setPortListener(new SoundPortListener() {
            @Override
            public Class<?> getPortClass() {
                return SoundPort.class;
            }

            @Override
            public void labelChanged(final SoundPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                final NodeLabels nodeLabels = getNodeLabels();
                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.soundPort, port.getId(), port.getLabel());
                saveLabels();

                soundPortListPanel.repaint();
            }

            @Override
            public void statusChanged(final NodeInterface node, final SoundPort port) {
                LOGGER.debug("Status of sound port has changed, port: {}", port);
            }

            @Override
            public void valuesChanged(final SoundPort port, final PortConfigKeys... portConfigKeys) {

                LOGGER.info("The port value are changed for port: {}", port);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                for (PortConfigKeys key : portConfigKeys) {
                    switch (key) {
                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, null, values);

            }

            @Override
            public void testButtonPressed(
                final NodeInterface node, final SoundPort port, final SoundPortStatus newStatus) {
                LOGGER.info("The test button was pressed for port: {}, newStatus: {}", port, newStatus);

                if (SoundPortStatus.TEST != port.getStatus()) {
                    stopTestToggleTask(node, port);

                    // create a new instance of the port
                    SoundPort soundPort = new SoundPort();
                    soundPort.setId(port.getId());
                    soundPort.setStatus(PortUtils.getOppositeStatus(newStatus));

                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), soundPort);
                }
                else {
                    addTestToggleTask(node, port);
                }
            }

            @Override
            public void configChanged(final NodeInterface node, final SoundPort port) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                SwingUtilities.invokeLater(() -> tableModel.notifyPortConfigChanged(port));
            }

            @Override
            public void changePortType(LcOutputType portType, SoundPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);

            }
        });

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeWillChange(final NodeInterface node) {
                LOGGER.info("The selected node will change!");
                try {
                    List<NodeInterface> nodes = new LinkedList<>();
                    for (NodeInterface currentNode : testToggleRegistry.keySet()) {
                        nodes.add(currentNode);
                    }
                    LOGGER.info("Found nodes to stop the test toggle task: {}", nodes);
                    for (NodeInterface currentNode : nodes) {
                        stopTestToggleTask(currentNode, null);
                    }
                    LOGGER.info("Stop the test toggle task passed for nodes: {}", nodes);
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop test toggle tasks failed.", ex);
                }
            }

            @Override
            public void nodeChanged(NodeInterface node) {
                super.nodeChanged(node);

                LOGGER.info("The selected node has been changed: {}", node);

                compDisp.dispose();
                compDisp.clear();

                compDisp = new CompositeDisposable();

                if (node != null) {
                    addSoundPortModelListener(node);
                }
            }
        });

        // keep the reference
        this.soundPortListPanel = soundPortListPanel;

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null) {
            addSoundPortModelListener(selectedNode);
        }

        return soundPortListPanel;
    }

    public void addTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Add test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.remove(node);
        ScriptEngine<PortScripting> scriptEngine = null;
        if (nodePortWrapper != null) {
            scriptEngine = nodePortWrapper.removePort(port);
        }

        if (scriptEngine != null) {
            LOGGER.info("Found a node scripting engine in the registry: {}", scriptEngine);
            try {
                scriptEngine.stopScript(Long.valueOf(2000));
            }
            catch (Exception ex) {
                LOGGER.warn("Stop script failed.", ex);
            }
        }

        ApplicationContext context = new DefaultScriptContext();
        context.register(DefaultScriptContext.KEY_SELECTED_NODE, node);
        context.register(DefaultScriptContext.KEY_MAIN_MODEL, mainModel);

        scriptEngine = new ScriptEngine<PortScripting>(this, context);

        List<ScriptCommand<PortScripting>> scriptCommands = new LinkedList<ScriptCommand<PortScripting>>();
        SoundPortCommand<PortScripting> spc = new SoundPortCommand<>(this.consoleService);
        spc.parse(SoundPortCommand.KEY + " " + port.getId() + " PLAY");
        scriptCommands.add(spc);
        WaitCommand<PortScripting> wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);
        spc = new SoundPortCommand<>(this.consoleService);
        spc.parse(SoundPortCommand.KEY + " " + port.getId() + " STOP");
        scriptCommands.add(spc);
        wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);

        LOGGER.info("Prepared list of commands: {}", scriptCommands);

        scriptEngine.setScriptCommands(scriptCommands);
        // repeating
        scriptEngine.setScriptRepeating(true);

        if (nodePortWrapper == null) {
            LOGGER.info("Create new NodePortWrapper for node: {}", node);
            nodePortWrapper = new NodePortWrapper(node);
        }

        LOGGER.info("Put script engine in registry for node: {}", node);
        nodePortWrapper.addPort(port, scriptEngine);

        testToggleRegistry.put(node, nodePortWrapper);

        scriptEngine.startScript();
    }

    public void stopTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Stop test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.get(node);

        if (nodePortWrapper != null) {
            Set<Port<?>> toRemove = new HashSet<>();
            if (port != null) {
                toRemove.add(port);
            }
            else {
                toRemove.addAll(nodePortWrapper.getKeySet());
            }

            for (Port<?> removePort : toRemove) {
                ScriptEngine<PortScripting> engine = nodePortWrapper.removePort(removePort);

                if (engine != null) {
                    LOGGER.info("Found a node scripting engine in the registry: {}", engine);
                    try {
                        engine.stopScript(Long.valueOf(2000));
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Stop script failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No scripting engine found for node: {}", node);
                }
            }

            if (nodePortWrapper.isEmpty()) {
                LOGGER.info("No more ports registered for node: {}", node);
                testToggleRegistry.remove(node);
            }
        }
    }

    @Override
    public void sendPortStatusAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, BidibStatus portStatus) {

        LOGGER.info("Sound action on the port: {}, portStatus: {}", port, portStatus);
        try {
            final SwitchingNodeInterface selectedNode =
                node != null ? node : mainModel.getSelectedNode().getSwitchingNode();
            SoundPortStatus soundPortStatus = (SoundPortStatus) portStatus;

            SoundPort soundPortPort = new SoundPort();
            soundPortPort.setId(port);
            soundPortPort.setStatus(soundPortStatus);
            switchingNodeService.setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, soundPortPort);

        }
        catch (Exception ex) {
            LOGGER.warn("Activate sound port failed.", ex);
        }
    }

    @Override
    public void sendPortValueAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, int portValue) {

    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save backlight labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.soundPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.soundPortListPanel.refreshView());
        }
    }

    private void addSoundPortModelListener(final NodeInterface selectedNode) {

        LOGGER.info("Add sound port model listener for node: {}", selectedNode);

        final Disposable disp = this.portConfigChangeEventSubject.subscribe(evt -> {
            LOGGER.info("Received event: {}", evt);

            final PortTypeAware port = evt.getPort();

            // update port config
            try {
                final SoundPort soundPort = new SoundPort();
                soundPort.setId(port.getPortNumber());

                LOGGER.info("Prepared sound port: {}", soundPort);

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(), soundPort,
                        null, evt.getPortConfig());
            }
            catch (Exception ex) {
                LOGGER.warn("Set the soundport config failed.", ex);
                selectedNode.setNodeHasError(true);
                selectedNode.setReasonData("Set the soundport config failed.");
            }
        });

        compDisp.add(disp);
    }
}
