package org.bidib.wizard.mvc.main.model;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidibbase.BaseLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.exchange.lcmacro.LcAccessoryExporter;
import org.bidib.jbidibc.exchange.lcmacro.LcAccessorySwitchTime;
import org.bidib.jbidibc.exchange.lcmacro.LcAccessoryType;
import org.bidib.jbidibc.exchange.lcmacro.LcAspectPointType;
import org.bidib.jbidibc.exchange.lcmacro.TimeBaseUnitType;
import org.bidib.jbidibc.messages.enums.TimeBaseUnitEnum;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroRef;
import org.bidib.wizard.api.model.PortsProvider;
import org.bidib.wizard.core.labels.AccessoryLabelUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AccessoryFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(AccessoryFactory.class);

    private AccessoryFactory() {
    }

    public static void saveAccessory(String fileName, Accessory accessory, final ApplicationContext context)
        throws FileNotFoundException, IOException {
        saveAccessory(fileName, accessory, true, context);
    }

    public static void saveAccessory(
        String fileName, Accessory accessory, boolean gzip, final ApplicationContext context)
        throws FileNotFoundException, IOException {

        new AccessoryFactory().saveAccessoryWithJaxb(fileName, accessory, /* gzip */false, context);
    }

    public static Accessory loadAccessory(
        String fileName, PortsProvider portsProvider, final ApplicationContext context) {

        Accessory accessory = new AccessoryFactory().loadAccessoryWithJaxb(fileName, portsProvider, context);
        return accessory;
    }

    protected void saveAccessoryWithJaxb(
        String fileName, Accessory accessory, boolean gzip, final ApplicationContext context) {

        LcAccessoryType lcAccessory = new LcAccessoryType();
        int accessoryId = accessory.getId();
        lcAccessory.setAccessoryId(accessoryId);
        lcAccessory.setAccessoryName(accessory.toString());

        NodeLabels accessoryLabels = context.get("labels", NodeLabels.class);
        AccessoryLabel accessoryAspectLabel = AccessoryLabelUtils.getAccessoryLabel(accessoryLabels, accessoryId);

        lcAccessory.setStartupState(accessory.getStartupState());

        Integer switchTime = accessory.getSwitchTime();
        if (switchTime != null) {
            LcAccessorySwitchTime lcAccessorySwitchTime = new LcAccessorySwitchTime();
            lcAccessorySwitchTime.setSwitchTime(switchTime);
            TimeBaseUnitEnum timeBaseUnit = accessory.getTimeBaseUnit();
            switch (timeBaseUnit) {
                case UNIT_100MS:
                    lcAccessorySwitchTime.setTimeBaseUnit(TimeBaseUnitType.UNIT_100_MS);
                    break;
                case UNIT_1S:
                    lcAccessorySwitchTime.setTimeBaseUnit(TimeBaseUnitType.UNIT_1_S);
                    break;
                default:
                    LOGGER.warn("Unsupported timebase unit detected for conversion to xml: {}", timeBaseUnit);
                    throw new RuntimeException(
                        "Unsupported timebase unit detected for conversion to xml: " + timeBaseUnit);
            }
        }

        int aspectIndex = 0;
        for (MacroRef aspect : accessory.getAspects()) {
            Integer macroId = aspect.getId();
            LcAspectPointType lcAspectPoint = new LcAspectPointType();

            BaseLabel aspectLabel = AccessoryLabelUtils.getAccessoryAspectLabel(accessoryAspectLabel, aspectIndex);

            if (aspectLabel != null && StringUtils.isNotBlank(aspectLabel.getLabel())) {

                lcAspectPoint.setAspectName(aspectLabel.getLabel());
            }

            lcAspectPoint.setMacroId(macroId);
            lcAccessory.getLcAspectPoint().add(lcAspectPoint);
            aspectIndex++;
        }

        new LcAccessoryExporter().saveAccessory(lcAccessory, fileName, gzip);
    }

    protected Accessory loadAccessoryWithJaxb(
        String fileName, PortsProvider portsProvider, final ApplicationContext context) {
        // load accessories from file
        LcAccessoryType lcAccessory = new LcAccessoryExporter().loadAccessory(fileName);
        Accessory accessory = null;
        if (lcAccessory != null) {
            accessory = new Accessory();

            int accessoryId = lcAccessory.getAccessoryId();
            accessory.setId(accessoryId);

            NodeLabels nodeLabels = context.get("labels", NodeLabels.class);

            // import name of accessory

            String label = lcAccessory.getAccessoryName();
            accessory.setLabel(label);

            AccessoryLabelUtils.replaceAccessoryLabel(nodeLabels, accessory.getId(), label);

            // startup state
            accessory.setStartupState(lcAccessory.getStartupState());
            if (lcAccessory.getAccessorySwitchTime() != null) {
                accessory.setSwitchTime(lcAccessory.getAccessorySwitchTime().getSwitchTime());
                switch (lcAccessory.getAccessorySwitchTime().getTimeBaseUnit()) {
                    case UNIT_100_MS:
                        accessory.setTimeBaseUnit(TimeBaseUnitEnum.UNIT_100MS);
                        break;
                    case UNIT_1_S:
                        accessory.setTimeBaseUnit(TimeBaseUnitEnum.UNIT_1S);
                        break;
                    default:
                        break;
                }
            }

            List<MacroRef> macroIds = new ArrayList<>();
            int aspectId = 0;
            for (LcAspectPointType lcAspectPoint : lcAccessory.getLcAspectPoint()) {
                Macro macro = null;
                // get the assigned macro
                for (Macro currentMacro : portsProvider.getMacros()) {
                    if (currentMacro.getId() == lcAspectPoint.getMacroId()) {
                        macro = currentMacro;

                        // update the label
                        String aspectName = lcAspectPoint.getAspectName();
                        LOGGER.info("Found new aspect name to set: {}", aspectName);

                        AccessoryLabelUtils.replaceAspectLabel(nodeLabels, accessoryId, aspectId, aspectName);
                        break;
                    }
                }
                if (macro != null) {
                    LOGGER.info("Add current macro reference as aspect to accessory: {}", macro);

                    // TODO set the label of MacroRef
                    MacroRef aspect = new MacroRef(macro.getId());
                    macroIds.add(aspect);
                }
                else {
                    LOGGER.warn("No macro available to add to accessory with id: {}", lcAspectPoint.getMacroId());
                }

                // next aspect
                aspectId++;
            }
            accessory.setAspects(macroIds);
            accessory.setTotalAspects(macroIds.size());

        }
        LOGGER.info("return accessory: {}", accessory);
        return accessory;
    }
}
