package org.bidib.wizard.mvc.main.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.FeedbackPosition;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.mvc.main.controller.FeedbackPositionStatusChangeProvider;
import org.bidib.wizard.utils.NodeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.EventList;

public class FeedbackPositionModel extends Model implements FeedbackPositionStatusChangeProvider {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPositionModel.class);

    private final int MAX_SIZE = 3000;

    private NodeInterface selectedNode;

    private final PropertyChangeListener pclFeedbackPositions;

    private EventList<FeedbackPosition> positionsEventList = new BasicEventList<FeedbackPosition>(500);

    private boolean appendToEventList;

    public FeedbackPositionModel(final WizardSettingsInterface wizardSettings) {

        appendToEventList = wizardSettings.isAppendToPositionTable();

        pclFeedbackPositions = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.debug("Property has changed, evt: {}", evt);
                final FeedbackPosition position = (FeedbackPosition) evt.getNewValue();
                NodeInterface node = (NodeInterface) evt.getSource();

                LOGGER.info("Received feedback position from node: {}", NodeUtils.prepareLabel(node));

                SwingUtilities.invokeLater(() -> {
                    positionsEventList.getReadWriteLock().writeLock().lock();
                    try {

                        // allow to support append or replace new positions
                        if (!appendToEventList) {
                            int index = positionsEventList.indexOf(position);
                            if (index > -1) {
                                positionsEventList.remove(index);
                                positionsEventList.add(index, position);
                            }
                            else {
                                positionsEventList.add(position);
                            }
                        }
                        else {
                            if (positionsEventList.size() > MAX_SIZE) {
                                // remove the first 10 items
                                for (int index = 9; index >= 0; index--) {
                                    positionsEventList.remove(index);
                                }
                            }
                            positionsEventList.add(position);
                        }

                    }
                    finally {
                        positionsEventList.getReadWriteLock().writeLock().unlock();
                    }
                });
            }
        };

        wizardSettings.addPropertyChangeListener(WizardSettingsInterface.PROPERTY_APPEND_TO_POSITION_TABLE, evt -> {

            appendToEventList = wizardSettings.isAppendToPositionTable();
        });

    }

    @Override
    public NodeInterface getSelectedNode() {
        return selectedNode;
    }

    /**
     * @param selectedNode
     *            the selectedNode to set
     */
    public void setSelectedNode(final NodeInterface selectedNode) {

        if (this.selectedNode != null) {
            // unregister node listener
            LOGGER.info("Remove pcl from previous selected node: {}", this.selectedNode);
            this.selectedNode
                .removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPOSITIONS, pclFeedbackPositions);
        }

        this.selectedNode = selectedNode;

        if (this.selectedNode != null) {
            // register node listener
            LOGGER.info("Add pcl to currently selected node: {}", this.selectedNode);
            this.selectedNode.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPOSITIONS, pclFeedbackPositions);
        }

    }

    /**
     * @return the positionsEventList
     */
    public EventList<FeedbackPosition> getPositionsEventList() {
        return positionsEventList;
    }
}
