package org.bidib.wizard.mvc.main.view.statusbar;

import java.awt.Font;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import javax.swing.SwingUtilities;

import org.bidib.wizard.api.model.common.PreferencesPortType;
import org.bidib.wizard.api.notification.TimeEvent;
import org.bidib.wizard.common.model.settings.ConnectionConfiguration;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.core.utils.AopUtils;
import org.bidib.wizard.mvc.main.view.panel.listener.ModelClockStatusListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.status.LabelStatusBarItem;

public class DigitalClockStatusBarItem extends LabelStatusBarItem implements ModelClockStatusListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(DigitalClockStatusBarItem.class);

    private static final long serialVersionUID = 1L;

    private final DateTimeFormatter format = DateTimeFormatter.ofPattern("ccc HH:mm:ss");

    private boolean startEnabled;

    private final SettingsService settingsService;

    public DigitalClockStatusBarItem(final SettingsService settingsService) {
        this.settingsService = settingsService;

        // enable start by default
        startEnabled = true;

        checkSelectedPortType();

        GlobalSettingsInterface gs = settingsService.getGlobalSettings();
        try {
            GlobalSettings globalSettings = AopUtils.getTargetObject(gs);
            globalSettings
                .addPropertyChangeListener(GlobalSettings.PROPERTY_SELECTED_PORTTYPE, new PropertyChangeListener() {

                    @Override
                    public void propertyChange(PropertyChangeEvent evt) {
                        checkSelectedPortType();
                    }
                });
        }
        catch (Exception ex) {
            LOGGER.warn("Add property change listeners failed.", ex);
        }
    }

    @Override
    public String getItemName() {
        return "DigitalClock";
    }

    public void setBoldFont() {
        setFont(new Font("SansSerif", Font.BOLD, 20));
    }

    /**
     * @return the startEnabled
     */
    public boolean isStartEnabled() {
        return startEnabled;
    }

    /**
     * @param startEnabled
     *            the startEnabled to set
     */
    public void setStartEnabled(boolean startEnabled) {
        this.startEnabled = startEnabled;
    }

    public void start() {
        LOGGER.info("Start the digital clock.");
    }

    public void stop() {
        LOGGER.info("Stop the digital clock.");

        LOGGER.info("Stop the clock passed.");

        SwingUtilities.invokeLater(() -> setText(null));
    }

    public void setModelTime(final TimeEvent timeEvent) {

        SwingUtilities.invokeLater(() -> {

            updateText(timeEvent.getTimestamp());
        });
    }

    private void updateText(final LocalDateTime date) {
        if (SwingUtilities.isEventDispatchThread()) {

            if (date != null) {
                setText(format.format(date));
            }
            else {
                setText(null);
            }
        }
        else {
            SwingUtilities.invokeLater(() -> {
                if (date != null) {
                    setText(format.format(date));
                }
                else {
                    setText(null);
                }
            });
        }
    }

    @Override
    public void setModelTimeStartStatus(boolean enabled) {
        LOGGER.info("Set the start enabled flag: {}", enabled);
        startEnabled = enabled;
    }

    private void checkSelectedPortType() {
        // must get the correct connection from the globalSettings
        PreferencesPortType selectedPortType =
            ConnectionConfiguration
                .toPreferencesPortType(settingsService.getGlobalSettings().getConnectionConfigurations(),
                    ConnectionRegistry.CONNECTION_ID_MAIN);

        if (selectedPortType != null
            && PreferencesPortType.ConnectionPortType.SerialOverTcp.equals(selectedPortType.getConnectionPortType())) {
            LOGGER.info("The current selected port type is: {}. Do not start the clock.", selectedPortType);
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        if (!enabled) {
            stop();
        }
    }

}
