package org.bidib.wizard.mvc.ping.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.bidib.wizard.api.model.NodeInterface;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class PingTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(PingTableModel.class);

    public static final String PROPERTY_NODES = "nodes";

    public static final String PROPERTY_NODE_PING_STATUS = "nodePingStatus";

    private ArrayListModel<NodePingModel> nodeList = new ArrayListModel<>();

    private final PropertyChangeListener pclNodePingModel;

    private int pingInterval = 200;

    public PingTableModel() {

        pclNodePingModel = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("The property of the node ping model has changed: {}", evt.getSource());

                if (evt.getSource() instanceof NodePingModel) {
                    NodePingModel nodePingModel = (NodePingModel) evt.getSource();

                    firePropertyChange(PROPERTY_NODE_PING_STATUS, null, nodePingModel.getNode());
                }
            }
        };
    }

    /**
     * Set the default ping interval.
     * 
     * @param pingInterval
     *            the default ping interval
     */
    public void setPingInterval(int pingInterval) {
        LOGGER.info("Set the pingInterval: {}", pingInterval);
        this.pingInterval = pingInterval;
    }

    public void addNode(final NodeInterface node) {
        synchronized (nodeList) {
            NodePingModel nodePingModel = new NodePingModel(node);
            if (!nodeList.contains(nodePingModel)) {
                LOGGER.info("Add node to ping node list: {}", node);
                nodePingModel.registerNode();

                // set the default interval
                nodePingModel.setPingInterval(this.pingInterval);
                nodePingModel.setLastPing(System.currentTimeMillis());
                nodePingModel.setNodePingState(NodePingState.OFF);
                String nodeLabel = nodePingModel.prepareNodeLabel();
                nodePingModel.setNodeLabel(nodeLabel);

                List<NodePingModel> oldValue = new LinkedList<>(nodeList);
                nodeList.add(nodePingModel);

                firePropertyChange(PROPERTY_NODES, oldValue, nodeList);

                nodePingModel.addPropertyChangeListener(NodePingModel.PROPERTY_NODE_PING_STATUS, pclNodePingModel);
            }
            else {
                LOGGER.warn("Node is already in ping node list: {}", node);
            }
        }
    }

    public void removeNode(final NodeInterface node) {
        synchronized (nodeList) {
            LOGGER.info("Remove node from ping node list: {}", node);

            List<NodePingModel> oldValue = new LinkedList<>(nodeList);
            int index = nodeList.indexOf(new NodePingModel(node));
            if (index > -1) {
                NodePingModel removed = nodeList.remove(index);
                LOGGER.info("Removed node: {}", removed);

                removed.removePropertyChangeListener(NodePingModel.PROPERTY_NODE_PING_STATUS, pclNodePingModel);

                if (removed != null) {
                    removed.freeNode();
                }

                firePropertyChange(PROPERTY_NODES, oldValue, nodeList);
            }
        }
    }

    public ArrayListModel<NodePingModel> getNodeListModel() {
        return nodeList;
    }

    public List<NodePingModel> getNodes() {
        return Collections.unmodifiableList(nodeList);
    }

    public void setNodePingState(final NodeInterface node, NodePingState nodePingState) {
        synchronized (nodeList) {
            int index = nodeList.indexOf(new NodePingModel(node));
            if (index > -1) {
                NodePingModel nodePingModel = nodeList.get(index);
                if (nodePingModel != null) {
                    nodePingModel.setNodePingState(nodePingState);

                    // firePropertyChange(PROPERTY_NODE_PING_STATUS, null, node);
                }
            }
        }
    }

    public void setPongMarker(byte[] address, int marker) {
        // TODO implement
    }
}
