package org.bidib.wizard.mvc.position.model;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.IntConsumer;

import org.bidib.wizard.api.model.PositionFeedbackPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.FeedbackPortStatus;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FeedbackPositionTableModel
    extends SimplePortTableModel<FeedbackPortStatus, PositionFeedbackPort, PortModelListener<PositionFeedbackPort>> {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPositionTableModel.class);

    private static final int COLUMNS = 8;

    // map from port to table position
    private final Map<Port<?>, Integer> portIndexes = new HashMap<Port<?>, Integer>();

    public FeedbackPositionTableModel() {

        setColumnIdentifiers(new String[COLUMNS /* + 1 */]);
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        return PositionFeedbackPort.class;
    }

    @Override
    public void setRowCount(int rowCount) {
        if (rowCount == 0) {
            portIndexes.clear();
        }
        super.setRowCount(rowCount);
    }

    /**
     * Add ports to table model.
     * 
     * @param ports
     *            the ports to add to the table model
     */
    public void addRows(List<PositionFeedbackPort> ports) {
        LOGGER.trace("Current ports: {}", ports);
        if (ports != null) {
            List<Object> rowData = new LinkedList<Object>();
            int index = 0;

            for (Port<?> port : ports) {
                portIndexes.put(port, index++);
                rowData.add(port);
                if (index > 0 && index % COLUMNS == 0) {
                    // dummy column which will be removed from SimplePortListPanel
                    rowData.add("");
                    addRow(rowData.toArray());
                    rowData.clear();
                }
            }

            // check if a row is not fully filled ...
            if (rowData.size() > 0) {
                for (index = rowData.size(); index < COLUMNS + 1; index++) {
                    if (index > 0 && index % COLUMNS == 0) {
                        // dummy column which will be removed from SimplePortListPanel
                        rowData.add("");
                        addRow(rowData.toArray());
                        rowData.clear();

                        break;
                    }
                    rowData.add(null);
                }
            }
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        // use the context menu to change the name
        return false;
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final Object o = getValueAt(row, column);

        if (o instanceof PositionFeedbackPort) {
            final PositionFeedbackPort port = (PositionFeedbackPort) o;
            LOGGER.debug("Set value, current port: {}, value: {}", port, value);

            if (value instanceof String) {
                port.setLabel((String) value);
                super.setValueAt(port, row, column);
                fireLabelChanged(port, port.getLabel());
            }
            else if (value instanceof PositionFeedbackPort) {
                super.setValueAt(value, row, column);
            }

        }
    }

    // TODO verify if this method is called or can be replaced by updatePortStatus
    public void updatePort(Port<?> port) {
        if (port != null) {
            Integer index = portIndexes.get(port);
            if (index != null) {
                int row = index / COLUMNS;
                int column = index % COLUMNS;

                setValueAt(port, row, column);
            }
            else {
                LOGGER.warn("Feedback port is not registered: {}", port);
            }
        }
    }

    @Override
    public void notifyPortStatusChanged(final PositionFeedbackPort port) {
        // TODO Auto-generated method stub

    }

    @Override
    public void notifyPortConfigChanged(PositionFeedbackPort port) {
        // TODO Auto-generated method stub
    }

    public void validatePositions(final List<PositionFeedbackPort> outdatedPorts) {
        for (PositionFeedbackPort port : outdatedPorts) {
            updatePort(port);
        }
    }

    @Override
    public void notifyPortLabelChanged(PositionFeedbackPort port) {
        // TODO Auto-generated method stub

    }

    @Override
    public void refreshRow(PositionFeedbackPort port, final IntConsumer rowConsumer) {
        // TODO Auto-generated method stub

    }
}
