package org.bidib.wizard.mvc.preferences.view;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.netbidib.pairingstore.PairingStore;
import org.bidib.jbidibc.netbidib.pairingstore.PairingStoreEntry;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.dialog.EscapeDialog;
import org.bidib.wizard.client.common.preferences.view.panel.SettingsPanelInterface;
import org.bidib.wizard.common.model.settings.ExperimentalSettingsInterface;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.common.model.settings.NetBidibSettingsInterface;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.mvc.preferences.model.PairingStoreEntryModel;
import org.bidib.wizard.mvc.preferences.model.PairingStoreModel;
import org.bidib.wizard.mvc.preferences.model.PreferencesModel;
import org.bidib.wizard.mvc.preferences.view.listener.PreferencesViewListener;
import org.bidib.wizard.mvc.preferences.view.panel.ConnectionPanel;
import org.bidib.wizard.mvc.preferences.view.panel.ExperimentalSettingsPanel;
import org.bidib.wizard.mvc.preferences.view.panel.MiscSettingsPanel;
import org.bidib.wizard.mvc.preferences.view.panel.NetBidibSettingsPanel;
import org.bidib.wizard.mvc.preferences.view.panel.PairingStoreSettingsPanel;
import org.bidib.wizard.mvc.preferences.view.panel.TimePanel;
import org.bidib.wizard.mvc.preferences.view.panel.WizardSettingsPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;

public class PreferencesView extends EscapeDialog {
    private static final long serialVersionUID = -1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(PreferencesView.class);

    private final Collection<PreferencesViewListener> listeners = new LinkedList<PreferencesViewListener>();

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:p, 3dlu, p, fill:3dlu:grow";

    private final ConnectionPanel connectionPanel;

    private final MiscSettingsPanel miscPanel;

    private final TimePanel timePanel;

    private final WizardSettingsPanel wizardSettingsPanel;

    private final ExperimentalSettingsPanel experimentalSettingsPanel;

    private final NetBidibSettingsPanel netBidibSettingsPanel;

    private final PairingStoreSettingsPanel pairingStoreSettingsPanel;

    private final PairingStoreModel pairingStoreModel;

    private final JTabbedPane tabbedPane;

    private final JButton saveButton;

    private final JButton cancelButton;

    private ValueModel saveButtonEnabled = new ValueHolder();

    private final PreferencesModel model;

    private final List<SettingsPanelInterface> settingsPanelInterfaces;

    private final JFrame parent;

    public PreferencesView(final JFrame parent, final PreferencesModel model,
        final List<SettingsPanelInterface> settingsPanelInterfaces) {
        super(parent, Resources.getString(PreferencesView.class, "title"), true);
        this.parent = parent;
        this.model = model;
        this.settingsPanelInterfaces = settingsPanelInterfaces;

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.DIALOG);

        // prepare buttons
        this.saveButton = new JButton(Resources.getString(getClass(), "save"));
        this.saveButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSave();

                setVisible(false);
            }
        });
        this.saveButton.setEnabled(false);

        this.cancelButton = new JButton(Resources.getString(getClass(), "cancel"));
        this.cancelButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireCancel();

                setVisible(false);
            }
        });

        // create the tabbed pane for the preferences panels
        tabbedPane = new JTabbedPane();

        final Consumer<Boolean> bufferingCallback = (isBuffering) -> {

            bufferingChanged();
        };

        final GlobalSettingsInterface globalSettings = model.getGlobalSettings();
        final WizardSettingsInterface wizardSettings = model.getWizardSettings();
        final MiscSettingsInterface miscSettings = model.getMiscSettings();
        final ExperimentalSettingsInterface experimentalSettings = model.getExperimentalSettings();
        final NetBidibSettingsInterface netBidibSettings = model.getNetBidibSettings();

        if (StringUtils.isBlank(netBidibSettings.getPairingStoreLocation())) {
            String pairingStoreLocation = new File(miscSettings.getBidibConfigDir(), "/data/netBiDiB").getPath();
            LOGGER.info("Set the default location of the pairing store: {}", pairingStoreLocation);
            netBidibSettings.setPairingStoreLocation(pairingStoreLocation);
        }

        connectionPanel = new ConnectionPanel(model, globalSettings, wizardSettings, bufferingCallback);
        miscPanel = new MiscSettingsPanel(model, miscSettings, bufferingCallback);
        wizardSettingsPanel = new WizardSettingsPanel(model, wizardSettings, bufferingCallback);
        experimentalSettingsPanel = new ExperimentalSettingsPanel(experimentalSettings, bufferingCallback);
        netBidibSettingsPanel = new NetBidibSettingsPanel(netBidibSettings, bufferingCallback);

        // provide the pairingStore
        final PairingStore pairingStore = model.getPairingStore();

        this.pairingStoreModel = new PairingStoreModel();

        for (PairingStoreEntry entry : pairingStore.getPairingStoreEntries()) {
            PairingStoreEntryModel entryModel = new PairingStoreEntryModel();
            entryModel.setUid(entry.getUid());
            entryModel.setRequestorName(entry.getRequestorName());
            entryModel.setProductName(entry.getProductName());
            entryModel.setUserName(entry.getUserName());
            if (entry.getProtocolVersion() != null) {
                entryModel.setProtocolVersion(entry.getProtocolVersion().toProtocolVersion());
            }
            entryModel.setLastSeen(entry.getLastSeen());
            entryModel.setPaired(entry.isPaired());

            pairingStoreModel.addPairing(entryModel);
        }

        pairingStoreSettingsPanel = new PairingStoreSettingsPanel(pairingStoreModel, bufferingCallback);

        timePanel = new TimePanel(model, globalSettings, bufferingCallback);

        // add tabs
        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-connection"), null/* icon */, connectionPanel.createPanel(),
                Resources.getString(getClass(), "tab-connection.tooltip"));
        tabbedPane.setMnemonicAt(0, KeyEvent.VK_1);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-wizardsettings"), null/* icon */,
                wizardSettingsPanel.createPanel(), Resources.getString(getClass(), "tab-wizardsettings.tooltip"));
        tabbedPane.setMnemonicAt(1, KeyEvent.VK_2);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-misc"), null/* icon */, miscPanel.createPanel(),
                Resources.getString(getClass(), "tab-misc.tooltip"));
        tabbedPane.setMnemonicAt(1, KeyEvent.VK_3);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-time"), null/* icon */, timePanel.createPanel(),
                Resources.getString(getClass(), "tab-time.tooltip"));
        tabbedPane.setMnemonicAt(2, KeyEvent.VK_4);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-experimentalsettings"), null/* icon */,
                experimentalSettingsPanel.createPanel(),
                Resources.getString(getClass(), "tab-experimentalsettings.tooltip"));
        tabbedPane.setMnemonicAt(1, KeyEvent.VK_5);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-netbidibsettings"), null/* icon */,
                netBidibSettingsPanel.createPanel(), Resources.getString(getClass(), "tab-netbidibsettings.tooltip"));
        tabbedPane.setMnemonicAt(1, KeyEvent.VK_6);

        tabbedPane
            .addTab(Resources.getString(getClass(), "tab-pairingstoresettings"), null/* icon */,
                pairingStoreSettingsPanel.createPanel(),
                Resources.getString(getClass(), "tab-pairingstoresettings.tooltip"));
        tabbedPane.setMnemonicAt(1, KeyEvent.VK_7);

        for (SettingsPanelInterface settingsPanelInterface : settingsPanelInterfaces) {
            LOGGER.info("Add settings panel: {}", settingsPanelInterface);
            tabbedPane
                .addTab(settingsPanelInterface.getTabTitle(), null,
                    settingsPanelInterface.createPanel(bufferingCallback), settingsPanelInterface.getTabTooltip());
        }

        dialogBuilder.add(tabbedPane).xyw(1, 1, 3);

        final JPanel buttons =
            new ButtonBarBuilder().addGlue().addButton(saveButton, cancelButton).border(Paddings.DLU4).build();

        dialogBuilder.add(buttons).xyw(1, 3, 3);

        JPanel contentPanel = dialogBuilder.build();

        PropertyConnector.connect(saveButtonEnabled, "value", saveButton, "enabled");

        getContentPane().add(contentPanel);

        netBidibSettingsPanel.flushBuffer();
        connectionPanel.flushBuffer();
        experimentalSettingsPanel.flushBuffer();
        miscPanel.flushBuffer();
        wizardSettingsPanel.flushBuffer();
        timePanel.flushBuffer();
        pairingStoreSettingsPanel.flushBuffer();

        for (SettingsPanelInterface settingsPanelInterface : this.settingsPanelInterfaces) {
            settingsPanelInterface.flushBuffer();
        }

    }

    public void showView() {

        pack();
        setLocationRelativeTo(parent);

        setMinimumSize(getSize());
        setVisible(true);
    }

    private void bufferingChanged() {
        LOGGER.info("bufferingChanged.");

        boolean isBuffering =
            netBidibSettingsPanel.isBuffering() || connectionPanel.isBuffering()
                || experimentalSettingsPanel.isBuffering() || miscPanel.isBuffering()
                || wizardSettingsPanel.isBuffering() || timePanel.isBuffering()
                || pairingStoreSettingsPanel.isBuffering();

        for (SettingsPanelInterface settingsPanelInterface : this.settingsPanelInterfaces) {
            isBuffering = isBuffering || settingsPanelInterface.isBuffering();
        }

        saveButtonEnabled.setValue(isBuffering);
    }

    @Override
    public Dimension getPreferredSize() {

        Dimension dim = super.getPreferredSize();
        if (dim.width < 720) {
            dim.width = 720;
        }
        return dim;
    }

    public void addPreferencesViewListener(PreferencesViewListener l) {
        listeners.add(l);
    }

    private void fireSave() {
        netBidibSettingsPanel.commitBuffer();
        connectionPanel.commitBuffer();
        experimentalSettingsPanel.commitBuffer();
        miscPanel.commitBuffer();
        wizardSettingsPanel.commitBuffer();
        timePanel.commitBuffer();
        pairingStoreSettingsPanel.commitBuffer();

        for (SettingsPanelInterface settingsPanelInterface : this.settingsPanelInterfaces) {
            settingsPanelInterface.commitBuffer();
        }

        // update the pairing store with the pairingStoreModel
        final PairingStore pairingStore = model.getPairingStore();
        final List<PairingStoreEntry> keepEntries = new LinkedList<>();
        for (PairingStoreEntryModel entry : this.pairingStoreModel.getPairings()) {
            String uid = entry.getUid();
            PairingStoreEntry existing =
                pairingStore
                    .getPairingStoreEntries().stream().filter(e -> e.getUid().equals(uid)).findFirst().orElse(null);
            if (existing != null) {
                keepEntries.add(existing);
            }
        }
        pairingStore.clear();
        pairingStore.setPairings(keepEntries);

        for (PreferencesViewListener l : listeners) {
            l.save();
        }
    }

    private void fireCancel() {
        netBidibSettingsPanel.flushBuffer();
        connectionPanel.flushBuffer();
        experimentalSettingsPanel.flushBuffer();
        miscPanel.flushBuffer();
        wizardSettingsPanel.flushBuffer();
        timePanel.flushBuffer();
        pairingStoreSettingsPanel.flushBuffer();

        for (SettingsPanelInterface settingsPanelInterface : this.settingsPanelInterfaces) {
            settingsPanelInterface.flushBuffer();
        }

        for (PreferencesViewListener l : listeners) {
            l.cancel();
        }
    }
}
