package org.bidib.wizard.mvc.stepcontrol.view.excel;

import java.awt.Color;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.List;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Row.MissingCellPolicy;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExcelAspectReader {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExcelAspectReader.class);

    private static final Color EXCEL_COLOR_GREEN = new Color(0x00, 0xB0, 0x50);

    private static final Color EXCEL_COLOR_RED = new Color(0xFF, 0x00, 0x00);

    public List<ImportAspect> readAspects(final InputStream excelFile) {

        final List<ImportAspect> importAspects = new LinkedList<>();
        Workbook wb = null;
        try {
            // create the workbook
            wb = new XSSFWorkbook(excelFile);

            final Sheet firstSheet = wb.getSheetAt(0);

            int rowNum = firstSheet.getFirstRowNum();
            LOGGER.info("First row num: {}", rowNum);

            // search the row with text 'Gleisanschluss'
            for (; rowNum < 20; rowNum++) {
                Row row = firstSheet.getRow(rowNum);
                if (row == null) {
                    LOGGER.info("Emtpy row: {}", rowNum);
                    continue;
                }
                int firstCellNum = row.getFirstCellNum();

                if (firstCellNum == -1) {
                    LOGGER.info("Emtpy row: {}", rowNum);
                    continue;
                }

                Cell firstCell = row.getCell(firstCellNum, MissingCellPolicy.RETURN_NULL_AND_BLANK);
                if (firstCell != null && "Gleisanschluss".equals(firstCell.getStringCellValue())) {
                    LOGGER.info("Found 'Gleisanschluss' in row: {}", rowNum);
                    rowNum++;
                    parseAspects(importAspects, firstSheet, rowNum, firstCellNum);
                    break;
                }
            }

        }
        catch (IOException ex) {
            LOGGER.warn("Import aspects from Excel file failed.", ex);

            throw new DataImportException("Import aspects from Excel file failed.", ex);
        }
        finally {
            if (wb != null) {
                try {
                    wb.close();
                }
                catch (Exception ex) {
                    LOGGER.warn("Close Excel workbook failed.", ex);
                }
            }
        }

        return importAspects;
    }

    private void parseAspects(
        final List<ImportAspect> importAspects, final Sheet firstSheet, int rowNumStart, int firstCellNum) {

        for (int rowNum = rowNumStart; rowNum < rowNumStart + 48; rowNum++) {

            Row row = firstSheet.getRow(rowNum);
            Cell indexCell = row.getCell(firstCellNum, MissingCellPolicy.RETURN_NULL_AND_BLANK);
            Cell positionCell = row.getCell(firstCellNum + 1, MissingCellPolicy.RETURN_NULL_AND_BLANK);

            LOGGER.debug("Current cell, index: {}, position: {}", indexCell, positionCell);

            double index = indexCell.getNumericCellValue();
            double position = positionCell.getNumericCellValue();

            int intIndex = (int) index;
            int roundedPos = (int) Math.round(position);

            LOGGER.info("Current cell values, index: {}, position: {}", intIndex, roundedPos);

            // get the color of the indexCell
            if (indexCell.getCellStyle() != null && indexCell.getCellStyle().getFillBackgroundColorColor() != null) {

                XSSFCellStyle cellStyle = (XSSFCellStyle) indexCell.getCellStyle();
                XSSFColor bgColor = cellStyle.getFillForegroundXSSFColor();

                LOGGER.debug("Current bgColor: {}, val: {}", bgColor, ByteUtils.bytesToHex(bgColor.getRGB()));

                if ((EXCEL_COLOR_GREEN.getRGB() & 0xFFFFFF) == ByteUtils.getRGB(bgColor.getRGB())) {
                    LOGGER.info("Current color is green.");

                    importAspects.add(new ImportAspect(intIndex, roundedPos, false));
                }
                else if ((EXCEL_COLOR_RED.getRGB() & 0xFFFFFF) == ByteUtils.getRGB(bgColor.getRGB())) {
                    LOGGER.info("Current color is red.");

                    importAspects.add(new ImportAspect(intIndex, roundedPos, true));
                }
            }

        }
    }
}
