package org.bidib.wizard.mvc.locolist.model;

import java.util.Collections;
import java.util.List;

import javax.swing.SwingUtilities;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.bidib.jbidibc.messages.DriveState;
import org.bidib.jbidibc.messages.enums.DirectionEnum;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.locolist.controller.listener.LocoTableControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class LocoTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoTableModel.class);

    public static final String PROPERTY_CS_NODE_SELECTED = "csNodeSelected";

    private final ArrayListModel<LocoListModel> locoArrayList;

    private boolean csNodeSelected;

    public LocoTableModel(final LocoTableControllerListener locoTableController,
        final ArrayListModel<LocoListModel> locoModelList) {
        this.locoArrayList = locoModelList;
    }

    public void addLoco(final DriveState driveState) {
        synchronized (locoArrayList) {
            int locoAddress = driveState.getAddress();
            final LocoListModel loco = new LocoListModel(locoAddress);
            if (!locoArrayList.contains(loco)) {
                LOGGER.info("Add loco to loco list: {}, driveState: {}", loco, driveState);
                toLocoModel(driveState, loco);
                locoArrayList.add(loco);
            }
            else {
                LOGGER.warn("Loco is already in loco list: {}", loco);
                LocoListModel existing = IterableUtils.find(locoArrayList, new Predicate<LocoListModel>() {

                    @Override
                    public boolean evaluate(LocoListModel model) {
                        return model.equals(loco);
                    }

                });
                toLocoModel(driveState, existing);

                fireLocoChanged(existing);
            }
        }
    }

    private void fireLocoChanged(final LocoListModel locoModel) {
        LOGGER.debug("The loco address has been changed.");
        int index = locoArrayList.indexOf(locoModel);
        locoArrayList.fireContentsChanged(index);
    }

    private void toLocoModel(final DriveState driveState, final LocoListModel loco) {
        // update members of loco
        loco.setSpeed(driveState.getSpeed() & 0x7F);

        loco
            .setDirection((driveState.getDirection() == DirectionEnum.FORWARD ? DirectionStatus.FORWARD
                : DirectionStatus.BACKWARD));

        loco.setSpeedSteps(SpeedSteps.fromBidibFormat(driveState.getAddressFormat()));

        loco.setFunctions(driveState.getFunctions());

        loco.setOutputActive(driveState.getOutputActive() > 0);
    }

    public void removeLoco(int locoAddress) {
        synchronized (locoArrayList) {
            LOGGER.info("Remove loco from loco list: {}", locoAddress);

            int index = locoArrayList.indexOf(new LocoListModel(locoAddress));
            if (index > -1) {
                LocoListModel loco = locoArrayList.remove(index);

                // TODO remove loco from service
            }
        }
    }

    public void removeAllLocos() {
        synchronized (locoArrayList) {
            LOGGER.info("Remove all locos from loco list.");

            if (SwingUtilities.isEventDispatchThread()) {

                internalRemoveAllLocos();
            }
            else {
                SwingUtilities.invokeLater(() -> internalRemoveAllLocos());
            }
        }
    }

    private void internalRemoveAllLocos() {
        locoArrayList.clear();
    }

    public void setDriveState(byte[] address, DriveState driveState) {
        LOGGER.info("Drive state was delivered: {}", driveState);

        // if (driveState.getOutputActive() > 0) {
        if (driveState.getAddress() > 0) {
            addLoco(driveState);
        }
        // TODO remove locos from list if no longer in stack?
        // else {
        // removeLoco(driveState.getAddress());
        // }

    }

    public ArrayListModel<LocoListModel> getLocoListModel() {
        return locoArrayList;
    }

    public List<LocoListModel> getLocos() {
        if (locoArrayList == null) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(locoArrayList);
    }

    public boolean isCsNodeSelected() {
        return csNodeSelected;
    }

    public void setCsNodeSelected(boolean csNodeSelected) {
        boolean oldValue = this.csNodeSelected;
        this.csNodeSelected = csNodeSelected;

        if (SwingUtilities.isEventDispatchThread()) {
            firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected);
        }
        else {
            SwingUtilities.invokeLater(() -> firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected));
        }
    }
}
