package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;

import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.event.MenuEvent;
import org.bidib.wizard.client.common.event.MenuEvent.Action;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.listener.NodeListActionListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.ComponentFactory;
import com.jgoodies.forms.factories.DefaultComponentFactory;
import com.jgoodies.forms.factories.Paddings;

public class BasicOperationsPanel implements TabPanelProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(BasicOperationsPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "fill:pref:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref, 3dlu, pref, 3dlu, pref";

    private final MainModel mainModel;

    private final JPanel contentPanel;

    private final JButton btnFirmwareUpdate;

    private final JButton btnNodeDocumentation;

    private final JButton btnNodeConfigurator;

    private ImageIcon firmwareUpdateIcon;

    private ImageIcon nodeConfiguratorIcon;

    private ImageIcon nodeDocumentationIcon;

    private final ApplicationContext context;

    public BasicOperationsPanel(final MainModel model, final ApplicationContext context) {
        this.mainModel = model;
        this.context = context;

        // create form builder
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(BasicOperationsPanel.class, "name");
                }
            };
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout()) {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(BasicOperationsPanel.class, "name");
                }
            };
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        final ComponentFactory componentFactory = DefaultComponentFactory.getInstance();

        firmwareUpdateIcon = ImageUtils.createImageIcon(BasicOperationsPanel.class, "/icons/64x64/firmware-update.png");

        btnFirmwareUpdate =
            componentFactory
                .createButton(new AbstractAction(
                    Resources.getString("org.bidib.wizard.client.common.mvc.firmware.view.FirmwareView", "title"),
                    firmwareUpdateIcon) {

                    private static final long serialVersionUID = 1L;

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        fireFirmwareUpdate();
                    }

                    @Override
                    public boolean isEnabled() {
                        return true;
                    }
                });

        dialogBuilder.add(btnFirmwareUpdate).xy(1, 1);

        nodeConfiguratorIcon =
            ImageUtils.createImageIcon(BasicOperationsPanel.class, "/icons/64x64/nodescript-wizard.png");

        btnNodeConfigurator =
            componentFactory
                .createButton(new AbstractAction(
                    Resources.getString("org.bidib.wizard.nodescript.client.view.wizard.NodeScriptWizard", "title"),
                    nodeConfiguratorIcon) {

                    private static final long serialVersionUID = 1L;

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        fireNodeScriptWizard();
                    }

                    @Override
                    public boolean isEnabled() {
                        return true;
                    }
                });

        dialogBuilder.add(btnNodeConfigurator).xy(1, 3);

        nodeDocumentationIcon =
            ImageUtils.createImageIcon(BasicOperationsPanel.class, "/icons/64x64/node-documentation.png");

        btnNodeDocumentation =
            componentFactory
                .createButton(new AbstractAction(
                    Resources.getString("org.bidib.wizard.mvc.main.view.menu.NodeListMenu", "documentation"),
                    nodeDocumentationIcon) {

                    private static final long serialVersionUID = 1L;

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        fireNodeDocumentation();
                    }

                    @Override
                    public boolean isEnabled() {
                        return true;
                    }
                });
        dialogBuilder.add(btnNodeDocumentation).xy(1, 5);

        contentPanel = dialogBuilder.build();
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    public void nodeChanged() {
        LOGGER.info("The selected node has changed.");

        updateComponentState();
    }

    private void updateComponentState() {

        final NodeInterface node = mainModel.getSelectedNode();

        if (node != null) {
            updateNodeInfo(node);
        }
    }

    private void updateNodeInfo(final NodeInterface node) {
        if (node == null) {
            // no node available
            return;
        }
        LOGGER.info("Update the basic operations panel for node: {}", node);

        btnFirmwareUpdate.setEnabled(node.isUpdatable());

        // only enable nodeConfigurator if node has accessory and switch class bit set
        if (NodeUtils.hasAccessoryFunctions(node.getUniqueId()) || NodeUtils.hasSwitchFunctions(node.getUniqueId())) {
            btnNodeConfigurator.setEnabled(true);
        }
        else {
            btnNodeConfigurator.setEnabled(false);
        }

    }

    private void fireFirmwareUpdate() {
        LOGGER.info("Trigger the firmware update.");

        NodeListActionListener nodeListActionListener =
            DefaultApplicationContext
                .getInstance()
                .get(DefaultApplicationContext.KEY_MAIN_NODELIST_ACTION_LISTENER, NodeListActionListener.class);

        if (nodeListActionListener != null) {
            nodeListActionListener.firmwareUpdate(mainModel.getSelectedNode());
        }
        else {
            LOGGER.warn("No nodeListAction listener available.");
        }
    }

    private void fireNodeScriptWizard() {
        LOGGER.info("Trigger the node script wizard.");

        this.context.publishEvent(new MenuEvent(Action.nodeScriptWizard));
    }

    private void fireNodeDocumentation() {
        LOGGER.info("Trigger the node documentation.");

        NodeListActionListener nodeListActionListener =
            DefaultApplicationContext
                .getInstance()
                .get(DefaultApplicationContext.KEY_MAIN_NODELIST_ACTION_LISTENER, NodeListActionListener.class);

        if (nodeListActionListener != null) {
            nodeListActionListener.generateDocumentation(mainModel.getSelectedNode());
        }
        else {
            LOGGER.warn("No nodeListAction listener available.");
        }
    }
}
