package org.bidib.wizard.mvc.main.view.table;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.Int16PortConfigValue;
import org.bidib.jbidibc.messages.port.RgbPortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.component.SliderAndValuePanel;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.ColorRGBToIntegerConverter;
import org.bidib.wizard.client.common.converter.StringToDmxChannelConverter;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.HintTextField;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardBindings;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.renderer.BidibStatusListRenderer;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.LightPortStatus;
import org.bidib.wizard.mvc.main.model.LightPortTableModel;
import org.bidib.wizard.mvc.main.view.table.converter.PortStatusToStringConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.Severity;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;
import com.jidesoft.combobox.ColorExComboBox;
import com.jidesoft.converter.ColorConverter;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class LightPortEditorPanel extends AbstractPortEditorPanel<LightPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(LightPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, max(60dlu;pref), 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_STATUS_EVENTS_MILLIS = 80;

    private static final int TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS = 100;

    private ValueModel portLevelOffConverterModel;

    private ValueModel portLevelOnConverterModel;

    private ValueModel dimmDownConverterModel;

    private ValueModel dimmUpConverterModel;

    private ValueModel rgbValueModel;

    private ValueModel transitionTimeConverterModel;

    private ValueModel dmxMappingConverterModel;

    private JTextField portName;

    private HintTextField portLevelOffText;

    private HintTextField portLevelOnText;

    private HintTextField dimmDownText;

    private HintTextField dimmUpText;

    private HintTextField transitionTimeText;

    private HintTextField dmxMappingText;

    private SliderAndValuePanel portLevelOffSlider;

    private SliderAndValuePanel portLevelOnSlider;

    private SliderAndValuePanel dimmDownSlider;

    private SliderAndValuePanel dimmUpSlider;

    private JComboBox<LightPortStatus> comboPortStatus;

    private JLabel statusLabel;

    private JButton btnPortStatus;

    private ColorExComboBox rgbColorComboBox;

    private byte dimmUpPcfg = BidibLibrary.BIDIB_PCFG_DIMM_UP;

    private byte dimmDownPcfg = BidibLibrary.BIDIB_PCFG_DIMM_DOWN;

    private PublishSubject<LightPortStatus> statusEventSubject = PublishSubject.create();

    public LightPortEditorPanel(LightPort port, Consumer<LightPort> saveCallback,
        final Consumer<LightPort> valueCallback, final Consumer<LightPort> refreshCallback,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject,
        final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, portConfigChangeEventSubject, nodeSelectionProvider);
    }

    @Override
    protected LightPort clonePort(final LightPort port) {
        // create a clone of the input port
        final LightPort clone =
            LightPort
                .builder() //
                .withDimMax(port.getDimMax()) //
                .withDimMin(port.getDimMin()) //
                .withPwmMax(port.getPwmMax()) //
                .withPwmMin(port.getPwmMin()) //
                .withRgbValue(port.getRgbValue()) //
                .withTransitionTime(port.getTransitionTime()) //
                .withDmxMapping(port.getDmxMapping()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();

        // security check
        if (clone.getDimMax() == 0) {
            clone.setDimMax(1);
        }
        if (clone.getDimMin() == 0) {
            clone.setDimMin(1);
        }

        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final LightPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        final List<Component> order = new ArrayList<>();

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(LightPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(LightPortTableModel.class, "label") + ":").xy(1, row);
        this.portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        this.portName.setEnabled(port.isEnabled());
        dialogBuilder.add(this.portName).xyw(3, row, 7);

        order.add(this.portName);

        row += 2;

        // if port level off is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_PWM_MIN);

            // port level off
            this.portLevelOffConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            this.portLevelOffText = new HintTextField(() -> triggerValidation());
            final IntegerInputValidationDocument portLevelOffDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);

            portLevelOffDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(0, 255, showHint -> portLevelOffText.showHint(showHint)));
            portLevelOffText.setDocument(portLevelOffDocument);
            portLevelOffText.setColumns(2);
            portLevelOffText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(portLevelOffText, portLevelOffConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "portLevelOff")).xy(1, row);
            dialogBuilder.add(portLevelOffText).xy(3, row);

            portLevelOffText.setEnabled(port.isEnabled());
            order.add(portLevelOffText);

            ValidationComponentUtils.setMandatory(portLevelOffText, true);
            ValidationComponentUtils.setMessageKeys(portLevelOffText, "validation.portLevelOff_key");

            // use the clone port to provide the values
            this.portLevelOffSlider = new SliderAndValuePanel(0, 255, 10);
            portLevelOffSlider.createComponent();
            WizardBindings.bind(portLevelOffSlider, valueValueModel);
            portLevelOffSlider.setEnabled(port.isEnabled());

            dialogBuilder.add(portLevelOffSlider).xyw(5, row, 5);
            order.add(portLevelOffSlider.getSlider());

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int portLevelOffValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The portLevelOff value has been changed: {}", portLevelOffValue);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF,
                            new BytePortConfigValue(ByteUtils.getLowByte(portLevelOffValue))));
                }
            });

            row += 2;
        }

        // if port level on is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_PWM_MAX);

            // PWM max
            this.portLevelOnConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            this.portLevelOnText = new HintTextField(() -> triggerValidation());
            final IntegerInputValidationDocument portLevelOnDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);

            portLevelOnDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(0, 255, showHint -> portLevelOnText.showHint(showHint)));

            portLevelOnText.setDocument(portLevelOnDocument);
            portLevelOnText.setColumns(2);
            portLevelOnText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(portLevelOnText, portLevelOnConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "portLevelOn")).xy(1, row);
            dialogBuilder.add(portLevelOnText).xy(3, row);

            portLevelOnText.setEnabled(port.isEnabled());
            order.add(portLevelOnText);

            ValidationComponentUtils.setMandatory(portLevelOnText, true);
            ValidationComponentUtils.setMessageKeys(portLevelOnText, "validation.portLevelOn_key");

            // use the clone port to provide the values
            this.portLevelOnSlider = new SliderAndValuePanel(0, 255, 10);
            portLevelOnSlider.createComponent();
            WizardBindings.bind(portLevelOnSlider, valueValueModel);
            portLevelOnSlider.setEnabled(port.isEnabled());

            dialogBuilder.add(portLevelOnSlider).xyw(5, row, 5);
            order.add(portLevelOnSlider.getSlider());

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int portLevelOnValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The portLevelOn value has been changed: {}", portLevelOnValue);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON,
                            new BytePortConfigValue(ByteUtils.getLowByte(portLevelOnValue))));
                }
            });

            row += 2;
        }

        // if dimm min is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DIM_MIN);

            // dimm min
            this.dimmDownConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8) ? 65535 : 255;

            // JTextField dimmDownText = new JTextField();
            this.dimmDownText = new HintTextField(() -> triggerValidation(), maxRange);
            final IntegerInputValidationDocument dimmDownDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmDownDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(1, maxRange, showHint -> dimmDownText.showHint(showHint)));
            dimmDownText.setDocument(dimmDownDocument);
            dimmDownText.setColumns(2);
            dimmDownText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(dimmDownText, dimmDownConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dimmDown")).xy(1, row);
            dialogBuilder.add(dimmDownText).xy(3, row);

            dimmDownText.setEnabled(port.isEnabled());
            order.add(dimmDownText);

            ValidationComponentUtils.setMandatory(dimmDownText, true);
            ValidationComponentUtils.setMessageKeys(dimmDownText, "validation.dimmDown_key");

            // use the clone port to provide the values
            this.dimmDownSlider = new SliderAndValuePanel(1, maxRange, 10);
            dimmDownSlider.createComponent();
            WizardBindings.bind(dimmDownSlider, valueValueModel);
            dimmDownSlider.setEnabled(port.isEnabled());

            dialogBuilder.add(dimmDownSlider).xyw(5, row, 5);
            order.add(dimmDownSlider.getSlider());

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8);
            dimmDownPcfg = dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_DOWN;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int dimmDownValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The dimmDown value has been changed: {}", dimmDownValue);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port,
                            dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_DOWN,
                            dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmDownValue))
                                : new BytePortConfigValue(ByteUtils.getLowByte(dimmDownValue))));
                }
            });

            row += 2;
        }

        // if dimm max is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DIM_MAX);

            // dimm max
            this.dimmUpConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8) ? 65535 : 255;

            this.dimmUpText = new HintTextField(() -> triggerValidation(), maxRange);
            final IntegerInputValidationDocument dimmUpDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmUpDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(1, maxRange, showHint -> dimmUpText.showHint(showHint)));
            dimmUpText.setDocument(dimmUpDocument);
            dimmUpText.setColumns(2);
            dimmUpText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(dimmUpText, dimmUpConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dimmUp")).xy(1, row);
            dialogBuilder.add(dimmUpText).xy(3, row);

            dimmUpText.setEnabled(port.isEnabled());
            order.add(dimmUpText);

            ValidationComponentUtils.setMandatory(dimmUpText, true);
            ValidationComponentUtils.setMessageKeys(dimmUpText, "validation.dimmUp_key");

            // use the clone port to provide the values
            this.dimmUpSlider = new SliderAndValuePanel(1, maxRange, 10);
            dimmUpSlider.createComponent();
            WizardBindings.bind(dimmUpSlider, valueValueModel);
            dimmUpSlider.setEnabled(port.isEnabled());

            dialogBuilder.add(dimmUpSlider).xyw(5, row, 5);
            order.add(dimmUpSlider.getSlider());

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8);
            dimmUpPcfg = dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_UP;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int dimmUpValue = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The dimmUp value has been changed: {}", dimmUpValue);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port,
                            dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_UP,
                            dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmUpValue))
                                : new BytePortConfigValue(ByteUtils.getLowByte(dimmUpValue))));
                }
            });

            row += 2;
        }

        // if RGB value is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_RGB)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_RGB_VALUE);

            this.rgbValueModel = new ConverterValueModel(valueValueModel, new ColorRGBToIntegerConverter());

            this.rgbColorComboBox = new ColorExComboBox();
            rgbColorComboBox.setConverterContext(ColorConverter.CONTEXT_RGB);
            rgbColorComboBox.setColorValueVisible(false);
            rgbColorComboBox.setEnabled(port.isEnabled());

            rgbColorComboBox.setSelectedColor((Color) rgbValueModel.getValue());

            rgbColorComboBox.addItemListener(new ItemListener() {
                @Override
                public void itemStateChanged(ItemEvent e) {
                    Color selectedColor = rgbColorComboBox.getSelectedColor();

                    rgbValueModel.setValue(selectedColor);
                }
            });

            this.rgbValueModel.addValueChangeListener((evt) -> {
                LOGGER.info("Set the color: {}", evt.getNewValue());

                rgbColorComboBox.setSelectedColor((Color) evt.getNewValue());
            });

            this.rgbColorComboBox.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "rgbValue")).xy(1, row);
            dialogBuilder.add(rgbColorComboBox).xy(3, row);

            order.add(rgbColorComboBox);

            ValidationComponentUtils.setMandatory(rgbColorComboBox, true);
            ValidationComponentUtils.setMessageKeys(rgbColorComboBox, "validation.rgbValue_key");

            row += 2;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int rgbValue = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The rgb value has been changed: {}", rgbValue);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_RGB,
                            new RgbPortConfigValue(ByteUtils.getRGB(rgbValue))));
                }
            });
        }

        // if transition time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TRANSITION_TIME)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_TRANSITION_TIME);

            this.transitionTimeConverterModel =
                new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            this.transitionTimeText = new HintTextField(() -> triggerValidation());
            final IntegerInputValidationDocument transitionTimeDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            transitionTimeDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(0, 255, showHint -> transitionTimeText.showHint(showHint)));
            transitionTimeText.setDocument(transitionTimeDocument);
            transitionTimeText.setColumns(2);
            transitionTimeText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(transitionTimeText, transitionTimeConverterModel, false);
            transitionTimeText.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "transitionTime")).xy(1, row);
            dialogBuilder.add(transitionTimeText).xy(3, row);

            order.add(transitionTimeText);

            ValidationComponentUtils.setMandatory(transitionTimeText, true);
            ValidationComponentUtils.setMessageKeys(transitionTimeText, "validation.transitionTime_key");

            row += 2;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int transitionTime = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The transitionTime value has been changed: {}", transitionTime);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_TRANSITION_TIME,
                            new Int16PortConfigValue(transitionTime)));
                }
            });
        }

        // if DMX mapping is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DMX_MAPPING);

            this.dmxMappingConverterModel = new ConverterValueModel(valueValueModel, new StringToDmxChannelConverter());

            this.dmxMappingText = new HintTextField(() -> triggerValidation());
            final IntegerInputValidationDocument dmxMappingDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            dmxMappingDocument
                .setRangeValidationCallback(new DefaultRangeValidationCallback(MIN_DMX_MAPPING_VALUE, 255,
                    showHint -> dmxMappingText.showHint(showHint)));
            dmxMappingText.setDocument(dmxMappingDocument);
            dmxMappingText.setColumns(2);
            dmxMappingText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(dmxMappingText, dmxMappingConverterModel, false);
            dmxMappingText.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dmxMapping")).xy(1, row);
            dialogBuilder.add(dmxMappingText).xy(3, row);

            order.add(dmxMappingText);

            ValidationComponentUtils.setMandatory(dmxMappingText, true);
            ValidationComponentUtils.setMessageKeys(dmxMappingText, "validation.dmxMapping_key");

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int outputMap = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The outputMap value has been changed: {}", outputMap);

                    getLocalPortConfigChangeEventSubject()
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_OUTPUT_MAP,
                            new BytePortConfigValue(ByteUtils.getLowByte(outputMap))));
                }
            });

            row += 2;
        }

        // change the status
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(LightPort.PROPERTY_STATUS);

        final ConverterValueModel portStatusConverterModel =
            new ConverterValueModel(selectionHolderPortStatus,
                new PortStatusToStringConverter(LightPortStatus.class, "status."));

        this.statusLabel = WizardComponentFactory.createLabel(portStatusConverterModel);
        this.statusLabel.setEnabled(port.isEnabled());

        final SelectionInList<LightPortStatus> portStatusSelection =
            new SelectionInList<>(LightPortStatus.ON.getValues());

        final ValueModel portStatusHolder = new ValueHolder(selectionHolderPortStatus.getValue());
        ComboBoxAdapter<LightPortStatus> comboBoxAdapterPortStatus =
            new ComboBoxAdapter<>(portStatusSelection, portStatusHolder);
        this.comboPortStatus = new JComboBox<>();
        this.comboPortStatus.setRenderer(new BidibStatusListRenderer<>(LightPortStatus.class));
        this.comboPortStatus.setModel(comboBoxAdapterPortStatus);

        LightPortStatus oppositeStatus =
            PortUtils.getOppositeStatus((LightPortStatus) selectionHolderPortStatus.getValue());
        LOGGER.debug("Set the opposite status in the port status selection: {}", oppositeStatus);
        this.comboPortStatus.setSelectedItem(oppositeStatus);

        this.btnPortStatus = new JButton(Resources.getString(LightPortTableModel.class, "test"));

        btnPortStatus.addActionListener(evt -> {
            LightPortStatus portStatus = (LightPortStatus) portStatusHolder.getValue();
            LOGGER.info("The port status has been changed: {}", portStatus);

            statusEventSubject.onNext(portStatus);
        });

        // throttle the status events
        final Disposable disp =
            this.statusEventSubject
                .throttleLatest(TIME_BETWEEN_STATUS_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(portStatus -> sendStatusToPort(port, portStatus));
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(LightPortTableModel.class, "status") + ":").xy(1, row);
        dialogBuilder.add(statusLabel).xy(3, row);
        dialogBuilder.add(this.comboPortStatus).xy(5, row);
        dialogBuilder.add(btnPortStatus).xy(7, row);

        order.add(this.comboPortStatus);
        order.add(btnPortStatus);

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 5);
        addButtons(buttonPanel, order);

        // combine the port config events
        final Disposable dispLocalPortConfigChange =
            getLocalPortConfigChangeEventSubject()
                .buffer(TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance())
                .subscribe(evts -> {
                    // combine the values
                    PortConfigChangeEvent event = null;
                    for (PortConfigChangeEvent evt : evts) {
                        LOGGER.info("Process event: {}", evt);
                        if (event == null) {
                            event = evt;
                            continue;
                        }
                        event.getPortConfig().putAll(evt.getPortConfig());
                    }

                    if (event != null) {
                        LOGGER.info("Publish the config change event to the node: {}", event);
                        getPortConfigChangeEventSubject().onNext(event);
                    }
                }, err -> {
                    LOGGER.warn("The localPortConfigChangeEventSubject has received an error.", err);
                }, () -> {
                    LOGGER.info("The localPortConfigChangeEventSubject has completed.");
                });
        getCompDisp().add(dispLocalPortConfigChange);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        if (this.portLevelOffConverterModel != null) {
            this.portLevelOffConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.portLevelOnConverterModel != null) {
            this.portLevelOnConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmDownConverterModel != null) {
            this.dimmDownConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmUpConverterModel != null) {
            this.dimmUpConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.rgbValueModel != null) {
            this.rgbValueModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.transitionTimeConverterModel != null) {
            this.transitionTimeConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dmxMappingConverterModel != null) {
            this.dmxMappingConverterModel.addValueChangeListener(evt -> triggerValidation());
        }

        enableComponents();

        triggerValidation();

        getPanel().setFocusCycleRoot(true);
        getPanel().setFocusTraversalPolicy(createFocusTransversalPolicy(order));

        return getPanel();
    }

    @Override
    public void requestDefaultFocus() {
        this.portName.requestFocusInWindow(); // default focus
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case LightPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
                        updatePortStatusFromOriginalPort();
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        if (getOriginalPort().getPortType() == LcOutputType.LIGHTPORT) {
                            LOGGER.info("Current port type is LIGHTPORT.");
                            updatePortStatusFromOriginalPort();
                            enableComponents();
                        }
                        break;

                    case LightPort.PROPERTY_DIM_MIN:
                    case LightPort.PROPERTY_DIM_MAX:
                    case LightPort.PROPERTY_PWM_MIN:
                    case LightPort.PROPERTY_PWM_MAX:
                        LOGGER.info("LightPort configuration value has been changed.");

                        LightPort clone = clonePort(getOriginalPort());
                        getPresentationModel().setBean(clone);
                        triggerValidation();
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    private void updatePortStatusFromOriginalPort() {
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(LightPort.PROPERTY_STATUS);
        LightPortStatus status = getOriginalPort().getStatus();
        LOGGER.info("Current status of original port: {}", status);
        selectionHolderPortStatus.setValue(status);

        if (LightPortEditorPanel.this.comboPortStatus != null) {
            LightPortStatus oppositeStatus =
                PortUtils.getOppositeStatus((LightPortStatus) selectionHolderPortStatus.getValue());
            LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
            LightPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
        }
    }

    @Override
    protected ValidationResult validate(final LightPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.portLevelOffConverterModel != null && this.portLevelOffConverterModel.getValue() == null) {
            support.addError("portLevelOff_key", "not_empty_for_write");
        }
        else if (this.portLevelOffText != null && this.portLevelOffText.isShowHint()) {
            support.add(Severity.INFO, "portLevelOff_key", "validRange;0..255");
        }

        if (this.portLevelOnConverterModel != null && this.portLevelOnConverterModel.getValue() == null) {
            support.addError("portLevelOn_key", "not_empty_for_write");
        }
        else if (this.portLevelOnText != null && this.portLevelOnText.isShowHint()) {
            support.add(Severity.INFO, "portLevelOn_key", "validRange;0..255");
        }

        if (this.dimmDownConverterModel != null && this.dimmDownConverterModel.getValue() == null) {
            support.addError("dimmDown_key", "not_empty_for_write");
        }
        else if (this.dimmDownText != null && this.dimmDownText.isShowHint()) {
            support.add(Severity.INFO, "dimmDown_key", "validRange;0.." + this.dimmDownText.getMaxRange());
        }

        if (this.dimmUpConverterModel != null && this.dimmUpConverterModel.getValue() == null) {
            support.addError("dimmUp_key", "not_empty_for_write");
        }
        else if (this.dimmUpText != null && this.dimmUpText.isShowHint()) {
            support.add(Severity.INFO, "dimmUp_key", "validRange;0.." + this.dimmUpText.getMaxRange());
        }

        if (this.rgbValueModel != null && this.rgbValueModel.getValue() == null) {
            support.addError("rgbValue_key", "not_empty_for_write");
        }

        if (this.transitionTimeConverterModel != null && this.transitionTimeConverterModel.getValue() == null) {
            support.addError("transitionTime_key", "not_empty_for_write");
        }
        else if (this.transitionTimeText != null && this.transitionTimeText.isShowHint()) {
            support.add(Severity.INFO, "transitionTime_key", "validRange;0..255");
        }

        if (this.dmxMappingConverterModel != null && this.dmxMappingConverterModel.getValue() == null) {
            support.addError("dmxMapping_key", "not_empty_for_write");
        }
        else if (this.dmxMappingText != null && this.dmxMappingText.isShowHint()) {
            support.add(Severity.INFO, "dmxMapping_key", "validRange;" + MIN_DMX_MAPPING_VALUE + "..255");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    private void sendStatusToPort(final LightPort port, LightPortStatus portStatus) {
        LOGGER.info("Send the new portStatus to the port: {}", portStatus);

        triggerValidation();

        final LightPort lightPort = LightPort.builder().withStatus(portStatus).withId(port.getId()).build();

        getValueCallback().accept(lightPort);
    }

    @Override
    protected void doEnableComponents(final LightPort port) {

        boolean enabled = port.isEnabled();
        this.portName.setEnabled(enabled);

        if (portLevelOffText != null) {
            portLevelOffText
                .setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF));
        }
        if (portLevelOffSlider != null) {
            portLevelOffSlider
                .setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF));
        }
        if (portLevelOnText != null) {
            portLevelOnText
                .setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON));
        }
        if (portLevelOnSlider != null) {
            portLevelOnSlider
                .setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON));
        }

        if (dimmDownSlider != null) {
            dimmDownSlider.setEnabled(port.isEnabled() && port.isPortConfigKeySupported(dimmDownPcfg));
        }
        if (dimmDownText != null) {
            dimmDownText.setEnabled(port.isEnabled() && port.isPortConfigKeySupported(dimmDownPcfg));
        }
        if (dimmUpSlider != null) {
            dimmUpSlider.setEnabled(port.isEnabled() && port.isPortConfigKeySupported(dimmUpPcfg));
        }
        if (dimmUpText != null) {
            dimmUpText.setEnabled(port.isEnabled() && port.isPortConfigKeySupported(dimmDownPcfg));
        }

        if (transitionTimeText != null) {
            transitionTimeText
                .setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TRANSITION_TIME));
        }

        if (statusLabel != null) {
            statusLabel.setEnabled(port.isEnabled());
        }
        if (comboPortStatus != null) {
            comboPortStatus.setEnabled(port.isEnabled());
        }
        if (btnPortStatus != null) {
            btnPortStatus.setEnabled(port.isEnabled());
        }

        if (rgbColorComboBox != null) {
            rgbColorComboBox.setEnabled(port.isEnabled() && port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_RGB));
        }

        super.doEnableComponents(port);
    }
}
