package org.bidib.wizard.mvc.console.controller;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.core.event.MessageTimeoutEvent;
import org.bidib.wizard.api.event.ErrorEvent;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.NodeListProvider;
import org.bidib.wizard.api.model.event.AccessoryErrorEvent;
import org.bidib.wizard.api.model.event.ConsoleMessageEvent;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.mvc.console.view.ConsoleView;
import org.bushe.swing.event.annotation.AnnotationProcessor;
import org.bushe.swing.event.annotation.EventSubscriber;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

public class ConsoleController {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConsoleController.class);

    private ConsoleView consoleView;

    private final DockingDesktop desktop;

    private final NodeListProvider nodeListProvider;

    private final ConsoleService consoleService;

    public ConsoleController(final DockingDesktop desktop, final NodeListProvider nodeListProvider,
        final ConsoleService consoleService) {
        this.desktop = desktop;
        this.nodeListProvider = nodeListProvider;
        this.consoleService = consoleService;
    }

    protected ConsoleService getConsoleService() {
        return this.consoleService;
    }

    public void start() {

        // check if the console is already opened
        String searchKey = DockKeys.CONSOLE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(view);
            return;
        }

        // if (view == null) {
        createDockable();
        // }

    }

    public Dockable createDockable() {
        LOGGER.info("Create new ConsoleView.");

        if (consoleView != null) {
            // LOGGER.warn("ConsoleView is created already!");
            // throw new IllegalArgumentException("ConsoleView is created already!");
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return consoleView;
        }

        // if (view == null) {
        LOGGER.info("Create the consoleView instance.");
        consoleView = new ConsoleView(this.consoleService);
        // }

        // add the log panel next to the booster panel
        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState boosterTableView = null;
            // search the booster table view
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_BOOSTER_TABLE_VIEW.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the booster table view dockable.");
                    boosterTableView = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (boosterTableView != null) {
                LOGGER.info("Add the console view to the booster table view panel.");
                dock = boosterTableView.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new log panel at order: {}", order);

                desktop.createTab(dock, consoleView, order, true);
            }
            else if (desktop.getDockables().length > 1) {

                desktop.split(dock, consoleView, DockingConstants.SPLIT_BOTTOM);
                desktop.setDockableHeight(consoleView, 0.2d);
            }
            else {
                desktop.split(dock, consoleView, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(consoleView, RelativeDockablePosition.RIGHT);
        }

        // create the nodeList listener
        final NodeListListener nodeListListener = new DefaultNodeListListener() {

            @Override
            public void nodeStateChanged(final NodeInterface node) {
                // if the node has an error --> show the console
                if (node.isNodeHasError() /* || node.getNode().isStall() */) {
                    LOGGER.info("Show the console because the node has an error.");
                    SwingUtilities.invokeLater(() -> {
                        try {

                            // ensure console is visible
                            ConsoleController.ensureConsoleVisible();
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Set console visible failed.", ex);
                        }
                    });
                }
            }

        };

        // add listener to detect when the view is closed
        desktop.addDockableStateChangeListener(new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(consoleView) && event.getNewState().isClosed()) {
                    LOGGER.info("ConsoleView was closed, free resources.");

                    try {
                        consoleView.close();
                    }
                    finally {

                        // // unregister from desktop
                        // desktop.unregisterDockable(consoleView);
                        consoleView = null;

                        // add the eventbus processing
                        AnnotationProcessor.unprocess(this);

                        // register as nodeList listener at the main controller
                        ConsoleController.this.nodeListProvider.removeNodeListListener(nodeListListener);
                    }
                }
            }
        });
        // register as nodeList listener at the main controller
        this.nodeListProvider.addNodeListListener(nodeListListener);

        // add the eventbus processing
        AnnotationProcessor.process(this);

        return consoleView;
    }

    public static synchronized void ensureConsoleVisible() {
        // check if the console is already opened
        String searchKey = DockKeys.CONSOLE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);

        DockingDesktop desktop =
            DefaultApplicationContext.getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);
        if (desktop == null) {
            LOGGER.info("The desktop is not available.");
            return;
        }

        Dockable consoleView = desktop.getContext().getDockableByKey(searchKey);
        if (consoleView != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return;
        }
        else {
            LOGGER.info("Create new controller to open the console.");

            final ApplicationContext applicationContext =
                DefaultApplicationContext
                    .getInstance().get(DefaultApplicationContext.KEY_SPRINGCONTEXT, ApplicationContext.class);

            // create the consoleController
            ConsoleController consoleController = applicationContext.getBean(ConsoleController.class);
            consoleController.start();
        }
    }

    public void ensureConsoleIsVisible() {
        LOGGER.info("Ensure that the console is visible. Current consoleView: {}", consoleView);

        // check if the console is already opened
        if (consoleView != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return;
        }
        else {
            LOGGER.info("Start the controller to open the console.");
            this.start();
        }
    }

    public void addConsoleLine(ConsoleColor consoleColor, String line) {
        this.consoleService.addConsoleLine(consoleColor, line);
    }

    @EventSubscriber(eventClass = MessageTimeoutEvent.class)
    public void messageTimeoutEvent(final MessageTimeoutEvent messageTimeoutEvent) {
        LOGGER.info("Received a messageTimeoutEvent: {}", messageTimeoutEvent);

        if (!SwingUtilities.isEventDispatchThread()) {
            SwingUtilities.invokeLater(() -> addConsoleMessage(messageTimeoutEvent));
        }
        else {
            addConsoleMessage(messageTimeoutEvent);
        }
    }

    private void addConsoleMessage(final MessageTimeoutEvent messageTimeoutEvent) {

        // ensure console is visible
        ensureConsoleVisible();

        try {
            // add line
            getConsoleService()
                .addConsoleLine(ConsoleColor.red,
                    String
                        .format("Message timeout from node: %s, %s", messageTimeoutEvent.getNode().toString(),
                            messageTimeoutEvent.getMessage()));
        }
        catch (Exception ex) {
            LOGGER.warn("Add messageTimeoutEvent to console model failed: {}", messageTimeoutEvent, ex);
        }
    }

    @EventListener
    public void errorEvent(final ErrorEvent errorEvent) {
        LOGGER.info("Received a errorEvent: {}", errorEvent);

        if (!SwingUtilities.isEventDispatchThread()) {
            SwingUtilities.invokeLater(() -> addConsoleMessage(errorEvent));
        }
        else {
            addConsoleMessage(errorEvent);
        }
    }

    private void addConsoleMessage(final ErrorEvent errorEvent) {

        // ensure console is visible
        ensureConsoleVisible();

        try {
            // add line
            getConsoleService()
                .addConsoleLine(ConsoleColor.red,
                    String
                        .format("Error event from connectionId: %s, message: %s", errorEvent.getConnectionId(),
                            errorEvent.getMessage()));
        }
        catch (Exception ex) {
            LOGGER.warn("Add errorEvent to console model failed: {}", errorEvent, ex);
        }
    }

    @EventListener
    public void handleApplicationEvent(final AccessoryErrorEvent accessoryErrorEvent) {
        LOGGER.info("Received a accessoryErrorEvent: {}", accessoryErrorEvent);

        SwingUtilities.invokeLater(() -> {
            // ensure console is visible
            ensureConsoleVisible();
            // add line
            getConsoleService().addConsoleLine(ConsoleColor.red, accessoryErrorEvent.getErrorMessage());
        });
    }

    @EventListener
    public void handleConsoleMessageEvent(final ConsoleMessageEvent consoleMessageEvent) {
        LOGGER.info("Received a consoleMessageEvent: {}", consoleMessageEvent);

        SwingUtilities.invokeLater(() -> {
            // ensure console is visible
            if (consoleMessageEvent.isEnsureConsoleVisible()) {
                ensureConsoleVisible();
            }
            // add line
            getConsoleService()
                .addConsoleLine(consoleMessageEvent.getConsoleColor(), consoleMessageEvent.getConsoleMessage());
        });
    }

}
