package org.bidib.wizard.mvc.preferences.view.panel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.io.File;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.filechooser.FileFilter;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.preferences.view.panel.AbstractSettingsPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.IntegerRangeFilter;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.settings.MiscSettings;
import org.bidib.wizard.mvc.preferences.model.PreferencesModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;

/**
 * This panel displays miscellaneous preferences like selected comm port, path to log files or label files.
 */
public class MiscSettingsPanel extends AbstractSettingsPanel<MiscSettingsInterface> {

    private static final Logger LOGGER = LoggerFactory.getLogger(MiscSettingsPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, fill:50dlu:grow, 3dlu, pref";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu";

    private final PreferencesModel model;

    private final MiscSettingsInterface miscSettings;

    private JPanel contentPanel;

    public MiscSettingsPanel(final PreferencesModel model, final MiscSettingsInterface miscSettings,
        final Consumer<Boolean> bufferingCallback) {
        super(bufferingCallback);
        this.model = model;
        this.miscSettings = miscSettings;
    }

    public JPanel createPanel() {

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // use value model here and only update settings on focus lost
        preparePresentationModel(this.miscSettings);

        // bidib config dir
        dialogBuilder.add(Resources.getString(getClass(), "bidibConfigDir") + ":").xy(1, 1);

        final BufferedValueModel bidibConfigDirModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_BIDIB_CONFIG_DIR);

        final JTextField bidibConfigDir = WizardComponentFactory.createTextField(bidibConfigDirModel, false);

        dialogBuilder.add(bidibConfigDir).xy(3, 1);

        final JButton selectDirectoryButton =
            new JButton(Resources.getString(MiscSettingsPanel.class, "select-config-directory"));
        selectDirectoryButton
            .setToolTipText(Resources.getString(MiscSettingsPanel.class, "select-config-directory.tooltip"));
        selectDirectoryButton.addActionListener(evt -> {
            // select the directory
            String bidibConfigDirectory = bidibConfigDirModel.getString();

            final FileFilter[] ff = null;
            FileDialog dialog =
                new FileDialog(this.contentPanel, FileDialog.SAVE, bidibConfigDirectory, (String) null, ff) {
                    @Override
                    public void approve(String selectedFile) {
                        File file = new File(selectedFile);
                        if (file != null && file.isDirectory()) {

                            String configDirLocation = file.getPath();
                            LOGGER.info("Set the bidib config directory location: {}", configDirLocation);
                            bidibConfigDir.setText(configDirLocation);
                        }
                    }
                };
            dialog.setApproveButtonText(Resources.getString(MiscSettingsPanel.class, "select-config-directory"));
            dialog.showDialog();

        });
        dialogBuilder.add(selectDirectoryButton).xy(5, 1);

        // logfile path
        dialogBuilder.add(Resources.getString(getClass(), "logFilePath") + ":").xy(1, 3);

        final BufferedValueModel logFilePathModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_LOGFILE_PATH);

        final JTextField logFilePath = WizardComponentFactory.createTextField(logFilePathModel, false);

        logFilePath.setPreferredSize(new Dimension(300, logFilePath.getPreferredSize().height));
        dialogBuilder.add(logFilePath).xy(3, 3);

        final JButton selectLogFilePathButton =
            new JButton(Resources.getString(MiscSettingsPanel.class, "select-logfile-directory"));
        selectLogFilePathButton
            .setToolTipText(Resources.getString(MiscSettingsPanel.class, "select-logfile-directory.tooltip"));
        selectLogFilePathButton.addActionListener(evt -> {
            // select the directory
            String logFileDirectory = logFilePathModel.getString();

            final FileFilter[] ff = null;
            FileDialog dialog =
                new FileDialog(this.contentPanel, FileDialog.SAVE, logFileDirectory, (String) null, ff) {
                    @Override
                    public void approve(String selectedFile) {
                        File file = new File(selectedFile);
                        if (file != null && file.isDirectory()) {

                            String logFileDirLocation = file.getPath();
                            LOGGER.info("Set the logfile directory location: {}", logFileDirLocation);
                            logFilePath.setText(logFileDirLocation);
                        }
                    }
                };
            dialog.setApproveButtonText(Resources.getString(MiscSettingsPanel.class, "select-logfile-directory"));
            dialog.showDialog();

        });
        dialogBuilder.add(selectLogFilePathButton).xy(5, 3);

        // logfile append
        final BufferedValueModel logFileAppendModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_LOGFILE_APPEND);

        JCheckBox logFileAppend =
            WizardComponentFactory.createCheckBox(logFileAppendModel, Resources.getString(getClass(), "logFileAppend"));
        logFileAppend.setToolTipText(Resources.getString(getClass(), "logFileAppend.tooltip"));

        dialogBuilder.add(logFileAppend).xyw(1, 5, 5);

        // workspace path
        dialogBuilder.add(Resources.getString(getClass(), "workspacePath") + ":").xy(1, 7);

        final BufferedValueModel workspacePathModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_WORKSPACE_PATH);

        final JTextField workspacePath = WizardComponentFactory.createTextField(workspacePathModel, false);

        workspacePath.setPreferredSize(new Dimension(300, workspacePath.getPreferredSize().height));
        dialogBuilder.add(workspacePath).xy(3, 7);

        final JButton selectWorkspaceDirectoryButton =
            new JButton(Resources.getString(MiscSettingsPanel.class, "select-workspace-directory"));
        selectWorkspaceDirectoryButton
            .setToolTipText(Resources.getString(MiscSettingsPanel.class, "select-workspace-directory.tooltip"));
        selectWorkspaceDirectoryButton.addActionListener(evt -> {
            // select the directory
            String workspaceDirectory = workspacePathModel.getString();

            final FileFilter[] ff = null;
            FileDialog dialog =
                new FileDialog(this.contentPanel, FileDialog.SAVE, workspaceDirectory, (String) null, ff) {
                    @Override
                    public void approve(String selectedFile) {
                        File file = new File(selectedFile);
                        if (file != null && file.isDirectory()) {

                            String workspaceLocation = file.getPath();
                            LOGGER.info("Set the workspace location: {}", workspaceLocation);
                            workspacePath.setText(workspaceLocation);
                        }
                    }
                };
            dialog.setApproveButtonText(Resources.getString(MiscSettingsPanel.class, "select-workspace-directory"));
            dialog.showDialog();

        });
        dialogBuilder.add(selectWorkspaceDirectoryButton).xy(5, 7);

        // load workspace at startup
        final BufferedValueModel loadWorkspaceAtStartupModel =
                presentationModel.getBufferedModel(MiscSettings.PROPERTY_LOAD_WORKSPACE_AT_STARTUP);

        JCheckBox loadWorkspaceAtStartup =
                WizardComponentFactory.createCheckBox(loadWorkspaceAtStartupModel, Resources.getString(getClass(), "loadWorkspaceAtStartup"));
        loadWorkspaceAtStartup.setToolTipText(Resources.getString(getClass(), "loadWorkspaceAtStartup.tooltip"));

        dialogBuilder.add(loadWorkspaceAtStartup).xyw(1, 9, 5);

        // ignore wait timeout
        final BufferedValueModel ignoreWaitTimeoutModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_IGNORE_WAIT_TIMEOUT);

        JCheckBox ignoreWaitTimeout =
            WizardComponentFactory
                .createCheckBox(ignoreWaitTimeoutModel, Resources.getString(getClass(), "ignoreWaitTimeout"));

        dialogBuilder.add(ignoreWaitTimeout).xyw(1, 11, 5);

        // ignore wrong receive message number
        final BufferedValueModel ignoreWrongReceiveMessageNumberModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_IGNORE_WRONG_RECEIVE_MESSAGE_NUMBER);

        JCheckBox ignoreWrongReceiveMessageNumber =
            WizardComponentFactory
                .createCheckBox(ignoreWrongReceiveMessageNumberModel,
                    Resources.getString(getClass(), "ignoreWrongReceiveMessageNumber"));

        dialogBuilder.add(ignoreWrongReceiveMessageNumber).xyw(1, 13, 5);

        // response timeout
        final ValueModel responseTimeoutConverterModel =
            new ConverterValueModel(presentationModel.getBufferedModel(MiscSettings.PROPERTY_RESPONSE_TIMEOUT),
                new StringToIntegerConverter(Integer.valueOf(400)));

        dialogBuilder.add(Resources.getString(getClass(), "responseTimeout") + ":").xy(1, 15);
        final JTextField responseTimeout = new JTextField();
        responseTimeout.setDocument(new InputValidationDocument(4, InputValidationDocument.NUMERIC));

        // bind manually because we changed the document of the textfield
        Bindings.bind(responseTimeout, responseTimeoutConverterModel, false);

        dialogBuilder.add(responseTimeout).xy(3, 15);

        // firmware packet timeout
        final ValueModel firmwarePacketTimeoutConverterModel =
            new ConverterValueModel(presentationModel.getBufferedModel(MiscSettings.PROPERTY_FIRMWARE_PACKET_TIMEOUT),
                new StringToIntegerConverter(Integer.valueOf(4500)));

        dialogBuilder.add(Resources.getString(getClass(), "firmwarePacketTimeout") + ":").xy(1, 17);
        final JTextField firmwarePacketTimeout = new JTextField();
        firmwarePacketTimeout.setDocument(new InputValidationDocument(4, InputValidationDocument.NUMERIC));

        // bind manually because we changed the document of the textfield
        Bindings.bind(firmwarePacketTimeout, firmwarePacketTimeoutConverterModel, false);

        dialogBuilder.add(firmwarePacketTimeout).xy(3, 17);

        // serial use hardware flow control
        final BufferedValueModel serialUseHardwareFlowControlModel =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_SERIAL_USE_HARDWARE_FLOWCONTROL);

        JCheckBox serialUseHardwareFlowControl =
            WizardComponentFactory
                .createCheckBox(serialUseHardwareFlowControlModel,
                    Resources.getString(getClass(), "serialUseHardwareFlowControl"));

        dialogBuilder.add(serialUseHardwareFlowControl).xyw(1, 19, 5);

        // serial port provider
        final BufferedValueModel selectionHolderSerialProvider =
            presentationModel.getBufferedModel(MiscSettings.PROPERTY_SELECTED_SERIALPORT_PROVIDER);

        SelectionInList<String> serialProviderSelection =
            new SelectionInList<>((ListModel<String>) model.getSerialProviderList());

        ComboBoxAdapter<String> comboBoxAdapterSerialProvider =
            new ComboBoxAdapter<>(serialProviderSelection, selectionHolderSerialProvider);
        final JComboBox<String> comboSerialProvider = new JComboBox<>();
        comboSerialProvider.setModel(comboBoxAdapterSerialProvider);

        dialogBuilder.add(Resources.getString(MiscSettingsPanel.class, "serialPortProvider") + ":").xy(1, 21);
        dialogBuilder.add(comboSerialProvider).xy(3, 21);

        // thread pool size
        final ValueModel fetchNodesWorkerThreadPoolSizeConverterModel =
            new ConverterValueModel(
                presentationModel.getBufferedModel(MiscSettings.PROPERTY_FETCH_NODES_WORKER_THREADPOOL_SIZE),
                new StringToIntegerConverter(Integer.valueOf(60)));

        dialogBuilder.add(Resources.getString(getClass(), "fetchNodesWorkerThreadPoolSize") + ":").xy(1, 23);
        final JTextField fetchNodesWorkerThreadPoolSize = new JTextField();
        final InputValidationDocument numericDocument = new InputValidationDocument(4, InputValidationDocument.NUMERIC);
        numericDocument.setDocumentFilter(new IntegerRangeFilter(1, 20));
        fetchNodesWorkerThreadPoolSize.setDocument(numericDocument);

        // bind manually because we changed the document of the textfield
        Bindings.bind(fetchNodesWorkerThreadPoolSize, fetchNodesWorkerThreadPoolSizeConverterModel, false);

        dialogBuilder.add(fetchNodesWorkerThreadPoolSize).xy(3, 23);

        // default ping interval
        final ValueModel pingIntervalConverterModel =
            new ConverterValueModel(presentationModel.getBufferedModel(MiscSettings.PROPERTY_PING_INTERVAL),
                new StringToIntegerConverter(Integer.valueOf(200)));

        dialogBuilder.add(Resources.getString(getClass(), "pingInterval") + ":").xy(1, 25);
        final JTextField pingInterval = new JTextField();
        pingInterval.setDocument(new InputValidationDocument(5, InputValidationDocument.NUMERIC));

        // bind manually because we changed the document of the textfield
        Bindings.bind(pingInterval, pingIntervalConverterModel, false);

        dialogBuilder.add(pingInterval).xy(3, 25);

        contentPanel = dialogBuilder.build();
        contentPanel.setOpaque(false);
        return contentPanel;
    }

}
