package org.bidib.wizard.mvc.worklist.controller;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import javax.annotation.PreDestroy;
import javax.swing.SwingUtilities;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.event.DefaultLabelsWorkListItemEvent;
import org.bidib.wizard.api.event.WorkListItemEvent;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.mvc.main.controller.MainControllerInterface;
import org.bidib.wizard.mvc.worklist.controller.actions.WorkListAction;
import org.bidib.wizard.mvc.worklist.controller.listener.WorkListControllerListener;
import org.bidib.wizard.mvc.worklist.model.WorkItemListModel;
import org.bidib.wizard.mvc.worklist.model.WorkItemModel;
import org.bidib.wizard.mvc.worklist.model.WorkItems;
import org.bidib.wizard.mvc.worklist.view.WorkListView;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.event.EventListener;

import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

public class WorkListController implements WorkListControllerListener, WorkListItemProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkListController.class);

    public static final String WORKLIST_FILENAME = "workList-bidib-wizard-2.0.json";

    private final DockingDesktop desktop;

    private final MainControllerInterface mainController;

    private final WorkItemListModel workListItemModel;

    private final WizardSettingsInterface wizardSettings;

    private final MiscSettingsInterface miscSettings;

    private final Map<String, Supplier<WorkListAction>> actionMap;

    private WorkListView workListView;

    private final ObjectMapper objectMapper;

    public WorkListController(final DockingDesktop desktop, final MainControllerInterface mainController,
        final WorkItemListModel workListItemModel, final WizardSettingsInterface wizardSettings,
        final MiscSettingsInterface miscSettings, final Map<String, Supplier<WorkListAction>> actionMap) {
        this.desktop = desktop;
        this.mainController = mainController;
        this.workListItemModel = workListItemModel;
        this.wizardSettings = wizardSettings;
        this.miscSettings = miscSettings;
        this.actionMap = actionMap;

        this.objectMapper = new ObjectMapper().registerModule(new Jdk8Module()).registerModule(new JavaTimeModule());
        this.objectMapper.enable(DeserializationFeature.USE_LONG_FOR_INTS);

    }

    public void start() {

        // check if the worklist view is already opened
        String searchKey = DockKeys.WORK_LIST_PANEL;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing worklist view.");
            DockUtils.selectWindow(view);
            return;
        }

        createDockable();
    }

    @PreDestroy
    public void shutdown() {
        LOGGER.info("Save the worklist during shutdown.");
        saveWorkList();
    }

    public Dockable createDockable() {

        LOGGER.info("Create new WorkListView.");

        if (workListView != null) {
            LOGGER.info("Select the existing worklist view.");
            DockUtils.selectWindow(workListView);
            return workListView;
        }

        this.workListView = new WorkListView(this.desktop, this, this.workListItemModel);

        // add the log panel next to the console panel
        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });

        if (desktop.getDockables().length > 1) {

            DockableState nodeListPanel = null;
            // search the console view
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_NODE_LIST_PANEL.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the nodeList panel dockable.");
                    nodeListPanel = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();

            if (nodeListPanel != null) {
                LOGGER.info("Add the worklist view below the node list panel.");
                dock = nodeListPanel.getDockable();

                // add the workListPanel but hidden
                this.desktop.split(dock, workListView, DockingConstants.SPLIT_BOTTOM, 0.8d);
            }
            else if (desktop.getDockables().length > 1) {

                desktop.split(dock, workListView, DockingConstants.SPLIT_BOTTOM);
                desktop.setDockableHeight(workListView, 0.2d);
            }
            else {
                desktop.split(dock, workListView, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(workListView, RelativeDockablePosition.RIGHT);
        }

        desktop.addDockableStateChangeListener(new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(workListView) && event.getNewState().isClosed()) {
                    LOGGER.info("WorkListView was closed, free resources.");

                    saveWorkList();

                    LOGGER.info("Release instance of WorkListView.");
                    workListView = null;
                }
            }
        });

        loadWorkList();

        return workListView;
    }

    @EventListener
    public void addWorkListItem(final WorkListItemEvent item) {
        LOGGER.info("Received worklist item event: {}", item);

        if (SwingUtilities.isEventDispatchThread()) {
            doAddWorkListItem(item);
        }
        else {
            SwingUtilities.invokeLater(() -> {
                doAddWorkListItem(item);
            });
        }
    }

    private void doAddWorkListItem(final WorkListItemEvent item) {
        start();

        if (this.workListView != null) {

            this.workListView.addWorkListItem(item);

            try {
                this.workListView.getDockKey().setNotification(true);
            }
            catch (Exception ex) {
                LOGGER.warn("Set notification on dockKey failed.", ex);
            }

            saveWorkList();
        }
        else {
            LOGGER.warn("No workListView available.");
        }
    }

    private static final String WORKING_DIR_WORKLIST_KEY = "workList";

    protected void loadWorkList() {

        String storedWorkingDirectory = getWorkingDirectory();

        try {

            this.workListItemModel.clear();

            final Path path = Paths.get(storedWorkingDirectory, "worklist");

            File file = path.resolve(WORKLIST_FILENAME).toFile();

            final WorkItems workListItemEvents = objectMapper.readValue(file, WorkItems.class);

            final List<WorkListItemEvent> workListItemModel = new ArrayList<>();
            workListItemModel.addAll(workListItemEvents.workListItems);

            this.workListItemModel.addWorkListItems(workListItemModel);

        }
        catch (Exception ex) {
            LOGGER.warn("Load worklist failed.", ex);
        }
    }

    protected void saveWorkList() {

        try {
            String storedWorkingDirectory = getWorkingDirectory();

            // create an instance of DefaultPrettyPrinter
            ObjectWriter writer = objectMapper.writer(new DefaultPrettyPrinter());

            final Path path = Paths.get(storedWorkingDirectory, "worklist");
            Files.createDirectories(path);

            final WorkItems workListItemEvents = new WorkItems();
            for (WorkItemModel model : workListItemModel.getWorkItems()) {
                workListItemEvents.add(model.getWorkListItemEvent());
            }

            // convert workListItemModel to JSON file
            writer.writeValue(path.resolve(WORKLIST_FILENAME).toFile(), workListItemEvents);
        }
        catch (Exception ex) {
            LOGGER.warn("Save worklist failed.", ex);
        }
    }

    private String getWorkingDirectory() {
        String storedWorkingDirectory = this.wizardSettings.getWorkingDirectory(WORKING_DIR_WORKLIST_KEY);
        if (StringUtils.isBlank(storedWorkingDirectory)) {
            storedWorkingDirectory = this.miscSettings.getBidibConfigDir();
        }
        return storedWorkingDirectory;
    }

    @Override
    public void applyAction(final WorkItemModel selected) {

        String actionIdentifier = selected.getWorkListItemEvent().getAction();

        LOGGER
            .info("Apply action with actionIdentifier: {}, worklist item: {}", actionIdentifier,
                selected.getWorkListItemEvent());

        try {
            final WorkListAction<WorkListItemEvent> workListAction = this.actionMap.get(actionIdentifier).get();

            workListAction.apply(this.desktop, selected.getWorkListItemEvent());

            // remove item from todo list
            workListItemModel.removeWorkListItem(selected);

            saveWorkList();
        }
        catch (ActionAbortedException ex) {
            LOGGER.warn("Apply action was aborted: {}", ex.getMessage());
        }
        catch (Exception ex) {
            LOGGER.warn("Apply action failed.", ex);
        }
        finally {
            if (this.workListView != null) {
                this.workListView.getDockKey().setNotification(false);
            }
            else {
                LOGGER.info("Now workListView opened.");
            }
        }
    }

    @Override
    public WorkItemModel getDefaultLabelsWorkItemModel(String connectionId, final NodeInterface node) {
        LOGGER.info("Get DefaultLabels work item model for connectionId: {}, node: {}", connectionId, node);

        long uniqueId = node.getUniqueId();

        WorkItemModel selected = null;
        for (WorkItemModel model : workListItemModel.getWorkItems()) {

            if (model.getWorkListItemEvent() instanceof DefaultLabelsWorkListItemEvent) {
                DefaultLabelsWorkListItemEvent event = (DefaultLabelsWorkListItemEvent) model.getWorkListItemEvent();
                if (event.getConnectionId().equals(connectionId) && event.getUniqueId() != null
                    && event.getUniqueId() == uniqueId) {
                    selected = model;
                    LOGGER.info("Found pending worklist item: {}", selected);
                    break;
                }
            }
        }

        if (selected == null) {
            // create
            int relevantPidBits = node.getNode().getRelevantPidBits();

            final WorkListItemEvent workListItemEvent =
                new DefaultLabelsWorkListItemEvent(ByteUtils.getUniqueIdAsString(uniqueId),
                    WorkListItemEvent.Status.pending, connectionId, uniqueId, relevantPidBits);
            LOGGER.info("Created new workListItemEvent: {}", workListItemEvent);

            selected = new WorkItemModel(workListItemEvent);
        }

        return selected;
    }

}
