package org.bidib.wizard.mvc.main.controller;

import java.io.File;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.swing.SwingUtilities;

import org.apache.commons.io.FileUtils;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.exchange.vendorcv.VendorCvData;
import org.bidib.jbidibc.exchange.vendorcv.VendorCvFactory;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.CvDefinitionListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.client.common.controller.CvDefinitionPanelControllerInterface;
import org.bidib.wizard.client.common.controller.FeedbackPortStatusChangeProvider;
import org.bidib.wizard.client.common.view.cvdef.CvContainer;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeModelRegistry;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeTableModel;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.listener.TabStatusListener;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.common.utils.SearchPathUtils;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.CvDefinitionPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class CvDefinitionPanelController implements CvDefinitionPanelControllerInterface {

    private static final Logger LOGGER = LoggerFactory.getLogger(CvDefinitionPanelController.class);

    private final MainModel mainModel;

    private final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider;

    private final StatusBar statusBar;

    private CvDefinitionPanel cvDefinitionPanel;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private NodeService nodeService;

    @Autowired
    private MiscSettingsInterface miscSettings;

    private final String connectionId;

    // registry for CvDefinitionTreeTableModel instances
    @Autowired
    private CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry;

    private CvDefinitionListener cvDefinitionListener;

    private NodeInterface selectedNode;

    public CvDefinitionPanelController(final MainModel mainModel,
        final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider, String connectionId,
        final StatusBar statusBar) {
        this.mainModel = mainModel;
        this.feedbackPortStatusChangeProvider = feedbackPortStatusChangeProvider;
        this.connectionId = connectionId;
        this.statusBar = statusBar;
    }

    public void start(final TabStatusListener tabStatusListener) {

        cvDefinitionPanel =
            new CvDefinitionPanel(this, this.feedbackPortStatusChangeProvider, this.mainModel, tabStatusListener,
                settingsService, this.statusBar);

        this.cvDefinitionListener = new CvDefinitionListener() {

            @Override
            public void cvDefinitionValuesChanged(boolean read, List<String> changedNames) {

                if (cvDefinitionPanel != null) {
                    if (SwingUtilities.isEventDispatchThread()) {
                        cvDefinitionPanel.refreshTree();
                    }
                    else {
                        SwingUtilities.invokeLater(() -> cvDefinitionPanel.refreshTree());
                    }
                }
            }

            @Override
            public void cvDefinitionChanged() {

            }
        };

        // add the node list listener

        mainModel.addNodeListListener(new DefaultNodeListListener() {

            @Override
            public void nodeChanged(final NodeInterface node) {

                final NodeInterface selectedNode = mainModel.getSelectedNode();
                LOGGER.info("The selected node has changed: {}", selectedNode);

                if (CvDefinitionPanelController.this.selectedNode != null
                    && !Objects.equals(CvDefinitionPanelController.this.selectedNode, selectedNode)) {
                    LOGGER.info("Selected node has changed, unregister cv definition listener.");
                    CvDefinitionPanelController.this.selectedNode.removeCvDefinitionListener(cvDefinitionListener);

                    CvDefinitionPanelController.this.selectedNode = null;
                }

                if (selectedNode == null) {
                    LOGGER.info("Reset the pending changes in the cvDefinitionPanel.");
                    cvDefinitionPanel.resetPendingChanges();
                }
                else {
                    LOGGER.info("Register cv definition listener.");

                    CvDefinitionPanelController.this.selectedNode = selectedNode;
                    CvDefinitionPanelController.this.selectedNode.addCvDefinitionListener(cvDefinitionListener);
                }
            }

            @Override
            public void listNodeRemoved(final NodeInterface node) {
                LOGGER.info("The node was removed: {}", node);

                try {
                    cvDefinitionTreeModelRegistry.removeCvContainer(node.getUniqueId());
                }
                catch (Exception ex) {
                    LOGGER.warn("Remove CV container from registry failed.", ex);
                }

                if (CvDefinitionPanelController.this.selectedNode != null
                    && Objects.equals(CvDefinitionPanelController.this.selectedNode, node)) {
                    LOGGER.info("Selected node was removed, unregister cv definition listener.");
                    CvDefinitionPanelController.this.selectedNode.removeCvDefinitionListener(cvDefinitionListener);

                    CvDefinitionPanelController.this.selectedNode = null;
                }
            }
        });

    }

    public CvDefinitionPanel getCvDefinitionPanel() {

        return cvDefinitionPanel;
    }

    @Override
    public CvDefinitionTreeTableModel getCvDefinitionTreeTableModel(final NodeInterface node) {
        LOGGER.info("Get the cvDefinitionTreeTableModel for node: {}", node);
        long uniqueId = node.getUniqueId();

        CvContainer cvContainer = null;
        synchronized (cvDefinitionTreeModelRegistry) {

            cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(uniqueId);

            if (cvContainer == null) {
                if (node.getVendorCV() != null) {
                    LOGGER
                        .info("No cvContainer found but vendorCV available. Prepare the vendorCV tree for node: {}",
                            node);
                    final VendorCvData vendorCV = node.getVendorCV();
                    cvDefinitionTreeModelRegistry.prepareVendorCVTree(node, vendorCV, false);

                    cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(uniqueId);
                }

            }
        }

        if (cvContainer != null) {
            return cvContainer.getCvTreeModel();
        }
        return null;
    }

    @Override
    public List<ConfigurationVariable> getConfigVariables(final NodeInterface node) {
        LOGGER.info("Get the confiuration variables for node: {}", node);
        long uniqueId = node.getUniqueId();
        CvContainer cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(uniqueId);

        if (cvContainer != null) {
            return cvContainer.getConfigVariables();
        }
        return Collections.emptyList();
    }

    @Override
    public Map<String, CvNode> getCvNumberToNodeMap(final NodeInterface node) {
        LOGGER.info("Get the cvNumberToNodeMap for node: {}", node);
        long uniqueId = node.getUniqueId();
        CvContainer cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(uniqueId);
        if (cvContainer != null) {
            return cvContainer.getCvNumberToNodeMap();
        }
        return Collections.emptyMap();
    }

    /**
     * Replace the vendor CV tree in the registry.
     * 
     * @param node
     *            the node
     */
    public void replaceVendorCVTree(final NodeInterface node) {
        cvDefinitionTreeModelRegistry.prepareVendorCVTree(node, node.getVendorCV(), true);
    }

    @Override
    public VendorCvData copyCvDefinitionToUserDir(final Context context, NodeInterface node) {
        VendorCvData vendorCV = null;

        if (node != null) {

            try {
                String labelPath = miscSettings.getBidibConfigDir();
                File targetPath = new File(labelPath, "data/BiDiBNodeVendorData");

                vendorCV =
                    VendorCvFactory
                        .getCvDefinition(node.getNode(), context, SearchPathUtils::lookupFilesInClasspath,
                            "classpath:/bidib");

                LOGGER.info("Copy the vendorCV: {}", vendorCV);
                if (vendorCV != null) {
                    String fileName = vendorCV.getFilename();

                    LOGGER.info("Save to target filename: {}", fileName);

                    // make sure the target path exists
                    FileUtils.forceMkdir(targetPath);

                    VendorCvFactory.saveCvDefinition(vendorCV, new File(targetPath, fileName));
                }
                else {
                    LOGGER.warn("No vendorCV available for node: {}", node);
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Get CV definition for node failed.", ex);

                throw new RuntimeException("Copy VendorCV to user dir failed.", ex);
            }

            if (vendorCV != null) {
                // set the new vendorCV for the node
                node.setVendorCV(vendorCV);

                replaceVendorCVTree(node);
            }
            else if (ProductUtils.isOneBootloader(node.getUniqueId())) {
                LOGGER.info("The current node is a OneBootloader.");
            }
            else {
                LOGGER.warn("No CV definition available for node: {}", node);
            }
        }
        mainModel.setCvDefinition(vendorCV);

        return vendorCV;
    }

    @Override
    public VendorCvData loadCvDefinition(final Context context, final NodeInterface node) {
        VendorCvData vendorCV = null;

        LOGGER.info("Load the CV definition for node: {}", node);

        if (node != null) {
            vendorCV = node.getVendorCV();

            if (vendorCV == null) {
                try {
                    String installationPath = this.settingsService.getInstallationPath();
                    LOGGER.info("Current installation path: {}", installationPath);

                    File file = new File(installationPath);
                    file = new File(file.getAbsoluteFile(), "data/BiDiBNodeVendorData");

                    String userHome = System.getProperty("user.home");
                    File searchPathUserHomeWizard = new File(userHome, ".BiDiBWizard/data/BiDiBNodeVendorData");

                    String labelPath = miscSettings.getBidibConfigDir();
                    File searchPathLabelPath = new File(labelPath, "data/BiDiBNodeVendorData");

                    vendorCV =
                        VendorCvFactory
                            .getCvDefinition(node.getNode(), context, SearchPathUtils::lookupFilesInClasspath,
                                searchPathLabelPath.getAbsolutePath(), file.getAbsolutePath(), "classpath:/bidib",
                                searchPathUserHomeWizard.getAbsolutePath());
                }
                catch (Exception ex) {
                    LOGGER.warn("Get CV definition for node failed.", ex);
                }
            }

            if (vendorCV != null) {
                if (node.getVendorCV() == null) {
                    node.setVendorCV(vendorCV);
                }

                LOGGER
                    .info("Prepare the CvDefinitionTreeTableModel in the CvDefinitionPanelController for node: {}",
                        node);

                // store the prepared CvDefinitionTreeTableModel in a registry

                // make sure the cv container is created
                getCvDefinitionTreeTableModel(node);

                List<ConfigurationVariable> configurationVariables = getConfigVariables(node);

                node.setConfigVariables(configurationVariables);

            }
            else if (ProductUtils.isOneBootloader(node.getUniqueId())) {
                LOGGER.info("The current node is a OneBootloader.");
            }
            else {
                LOGGER.warn("No CV definition available for node: {}", node);
            }
        }
        mainModel.setCvDefinition(vendorCV);

        return vendorCV;
    }

    @Override
    public void reloadCvDefinition(final Context context, final NodeInterface node) {
        LOGGER.info("Reload the CV definition for node: {}", node);

        // release the old vendor CV
        node.setVendorCV(null);
        cvDefinitionTreeModelRegistry.removeCvContainer(node.getUniqueId());

        loadCvDefinition(context, node);
    }

    @Override
    public List<ConfigurationVariable> setConfigVariables(
        final NodeInterface node, final List<ConfigurationVariable> configVariables) {

        return nodeService.setConfigVariables(this.connectionId, node, configVariables);
    }

}
