package org.bidib.wizard.mvc.main.view.exchange;

import java.awt.Component;
import java.io.File;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;

import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.bidib2.BiDiB;
import org.bidib.jbidibc.core.schema.bidib2.ConfigurationVariable;
import org.bidib.jbidibc.core.schema.bidib2.MacroPointInput;
import org.bidib.jbidibc.core.schema.bidib2.MacroPointOutput;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.view.DefaultBusyFrame;
import org.bidib.wizard.client.common.view.cvdef.CvContainer;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeModelRegistry;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.mvc.main.controller.MainController;
import org.bidib.wizard.mvc.main.view.MainNodeListActionListener;
import org.bidib.wizard.mvc.main.view.component.NodeErrorsDialog;
import org.bidib.wizard.mvc.main.view.component.SaveNodeConfigurationDialog;
import org.bidib.wizard.mvc.main.view.panel.listener.NodeListActionListener;
import org.bidib.wizard.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeExchangeHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(NodeExchangeHelper.class);

    public static final String NODE_X_EXTENSION = "nodex";

    private static final String WORKING_DIR_NODE_EXCHANGE_KEY = "nodeExchange";

    public void importNode(
        final DefaultBusyFrame parentComponent, final NodeInterface node, final SettingsService settingsService,
        final NodeService nodeService, final SwitchingNodeService switchingNodeService,
        final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry, final FileFilter nodeFilter,
        final WizardLabelWrapper wizardLabelWrapper, final StatusBar statusBar,
        final NodeListActionListener nodeListActionListener) {

        final Map<String, Object> importParams = new HashMap<String, Object>();

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY);

        FileDialog dialog = new FileDialog(parentComponent, FileDialog.OPEN, storedWorkingDirectory, null, nodeFilter) {
            private JCheckBox checkRestoreAccessoryContent;

            private JCheckBox checkRestoreMacroContent;

            private JCheckBox checkRestoreNodeString;

            private JCheckBox checkRestoreFeatures;

            private JCheckBox checkRestoreCVs;

            @Override
            public void approve(String fileName) {

                try {
                    final ExportNodeOptionsDialog exportNodeOptionsDialog =
                        new ExportNodeOptionsDialog(parentComponent,
                            Resources.getString(NodeExchangeHelper.class, "importOptions.title"), node) {

                            private static final long serialVersionUID = 1L;

                            @Override
                            protected Component getAdditionalPanel() {

                                boolean defaultImportCVOnly = false;

                                // check if the current node is the OneControl with GPIO and FW version below 3.x
                                if ((ProductUtils.isOneControl(node.getUniqueId())
                                    || ProductUtils.isOneDriveTurn(node.getUniqueId()))
                                    && SoftwareVersion
                                        .build(3, 0, 0).isHigherThan(node.getNode().getSoftwareVersion())) {
                                    int result =
                                        JOptionPane
                                            .showConfirmDialog(JOptionPane.getFrameForComponent(parentComponent),
                                                Resources
                                                    .getString(MainNodeListActionListener.class,
                                                        "importOneControl.message"),
                                                Resources
                                                    .getString(MainNodeListActionListener.class,
                                                        "importOneControl.title"),
                                                JOptionPane.YES_NO_OPTION, JOptionPane.INFORMATION_MESSAGE);

                                    if (result == JOptionPane.OK_OPTION) {
                                        LOGGER
                                            .info(
                                                "User selected OK to uncheck all options expect the import of CV values.");
                                        defaultImportCVOnly = true;
                                        importParams
                                            .put(SaveNodeConfigurationDialog.DEFAULT_IMPORT_CV_ONLY,
                                                defaultImportCVOnly);
                                    }
                                }

                                // prepare a panel with checkboxes for loading macro content before export
                                final JPanel additionalPanel = new JPanel();
                                additionalPanel.setLayout(new BoxLayout(additionalPanel, BoxLayout.PAGE_AXIS));

                                checkRestoreNodeString =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreNodeString"),
                                        true);
                                additionalPanel.add(checkRestoreNodeString);

                                if (NodeUtils.hasAccessoryFunctions(node.getUniqueId())) {
                                    // TODO check if this is really necessary ... user can use store permanently
                                    // manually
                                    checkRestoreAccessoryContent =
                                        new JCheckBox(
                                            Resources.getString(MainController.class, "checkRestoreAccessoryContent"),
                                            !defaultImportCVOnly);
                                    additionalPanel.add(checkRestoreAccessoryContent);
                                    checkRestoreMacroContent =
                                        new JCheckBox(
                                            Resources.getString(MainController.class, "checkRestoreMacroContent"),
                                            !defaultImportCVOnly);
                                    additionalPanel.add(checkRestoreMacroContent);
                                    if (defaultImportCVOnly) {
                                        checkRestoreAccessoryContent.setEnabled(false);
                                        checkRestoreMacroContent.setEnabled(false);
                                    }
                                }
                                checkRestoreFeatures =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreFeatures"),
                                        false);
                                additionalPanel.add(checkRestoreFeatures);
                                if (defaultImportCVOnly) {
                                    checkRestoreFeatures.setEnabled(false);
                                }
                                checkRestoreCVs =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreCVs"),
                                        defaultImportCVOnly);
                                additionalPanel.add(checkRestoreCVs);
                                if (defaultImportCVOnly) {
                                    checkRestoreCVs.setEnabled(false);
                                }

                                return additionalPanel;
                            }
                        };
                    exportNodeOptionsDialog.setVisible(true);

                    if (exportNodeOptionsDialog.getResult() == JOptionPane.OK_OPTION) {
                        LOGGER.info("User confirmed options.");
                    }
                    else {
                        LOGGER.info("User cancelled options.");

                        return;
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Show exportNodeOptionsDialog failed.", ex);
                    return;
                }

                try {
                    parentComponent.setBusy(true);
                    LOGGER.info("Start importing node, fileName: {}", fileName);

                    LOGGER.info("Use BiDiB data to restore the node.");

                    // load the BiDiB instance from the file
                    final File bidibFile = new File(fileName);
                    final BiDiB bidib = BidibFactory.loadBiDiBFile(bidibFile);

                    // we expect a node in the bidib instance
                    if (bidib == null || bidib.getNodes() == null || bidib.getNodes().getNode().isEmpty()) {
                        LOGGER.warn("The provided BiDiB data is not valid for node import.");

                        throw new IllegalArgumentException("The provided BiDiB data is not valid for node import.");
                    }

                    // transform the bidib instance to match the target product if possible
                    transformMatchingProduct(parentComponent, node, bidib);

                    // only restore the selected parts
                    org.bidib.wizard.utils.NodeUtils
                        .configureFromBiDiB(ConnectionRegistry.CONNECTION_ID_MAIN, nodeService, switchingNodeService,
                            cvDefinitionTreeModelRegistry, node, importParams, bidib, wizardLabelWrapper,
                            (checkRestoreCVs != null ? checkRestoreCVs.isSelected() : false),
                            (checkRestoreFeatures != null ? checkRestoreFeatures.isSelected() : false),
                            (checkRestoreNodeString != null ? checkRestoreNodeString.isSelected() : false),
                            (checkRestoreMacroContent != null ? checkRestoreMacroContent.isSelected() : false),
                            (checkRestoreAccessoryContent != null ? checkRestoreAccessoryContent.isSelected() : false));

                    // save the node labels ....
                    LOGGER.info("Save the node labels.");
                    try {
                        long uniqueId = node.getUniqueId();

                        // save the port labels locally
                        wizardLabelWrapper.saveNodeLabels(uniqueId);
                    }
                    catch (InvalidConfigurationException ex) {
                        LOGGER.warn("Save port labels failed.", ex);

                        String labelPath = ex.getReason();
                        JOptionPane
                            .showMessageDialog(JOptionPane.getFrameForComponent(parentComponent),
                                Resources
                                    .getString(NodeExchangeHelper.class, "labelfileerror.message",
                                        new Object[] { labelPath }),
                                Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                                JOptionPane.ERROR_MESSAGE);
                    }

                    statusBar
                        .setStatusText(
                            String.format(Resources.getString(MainController.class, "importedNodeState"), fileName),
                            StatusBar.DISPLAY_NORMAL);

                    // show a dialog with some checkboxes to allow the user to select all options (port config,
                    // macros, accessories) and move this to saveNode ...
                    SaveNodeConfigurationDialog saveNodeConfigurationDialog =
                        new SaveNodeConfigurationDialog(parentComponent, null, null, node, importParams, true) {
                            private static final long serialVersionUID = 1L;

                            @Override
                            protected void fireContinue(final NodeInterface node) {

                                LOGGER.info("User wants to save the macros to node: {}", node);
                                Map<String, Object> params = new HashMap<String, Object>();

                                params.put(SAVE_MACROS, isSaveMacros());
                                params.put(SAVE_ACCESSORIES, isSaveAccessories());
                                params.put(SAVE_BACKLIGHTPORTS, isSaveBacklightPorts());
                                params.put(SAVE_LIGHTPORTS, isSaveLightPorts());
                                params.put(SAVE_SERVOPORTS, isSaveServoPorts());
                                params.put(SAVE_SWITCHPORTS, isSaveSwitchPorts());
                                params.put(SAVE_FEATURES, isSaveFeatures());
                                params.put(SAVE_CVS, isSaveConfigurationVariables());

                                try {
                                    parentComponent.setBusy(true);

                                    statusBar
                                        .setStatusText(String
                                            .format(Resources.getString(MainController.class, "writeToNode"), fileName),
                                            StatusBar.DISPLAY_NORMAL);

                                    nodeListActionListener.saveOnNode(node, params);
                                }
                                finally {
                                    statusBar
                                        .setStatusText(String
                                            .format(Resources.getString(MainController.class, "writeToNodeFinished"),
                                                fileName),
                                            StatusBar.DISPLAY_NORMAL);

                                    parentComponent.setBusy(false);
                                }

                                if (params.containsKey(SAVE_ERRORS)) {
                                    LOGGER.warn("The save on node operation has finished with errors!!!");
                                    // show an error dialog with the list of save errors during update node
                                    NodeErrorsDialog nodeErrorsDialog =
                                        new NodeErrorsDialog(parentComponent, null, true);

                                    // set the error information
                                    nodeErrorsDialog
                                        .setErrors(
                                            Resources.getString(MainController.class, "save-values-on-node-failed"),
                                            (List<String>) params.get(SAVE_ERRORS));
                                    nodeErrorsDialog.showDialog();
                                }
                            }

                            @Override
                            protected void fireCancel(final NodeInterface node) {
                                LOGGER.info("User cancelled upload of changes to node: {}", node);
                            }
                        };

                    saveNodeConfigurationDialog.setSaveFeaturesEnabled(checkRestoreFeatures.isSelected());
                    saveNodeConfigurationDialog.setSaveCVsEnabled(checkRestoreCVs.isSelected());

                    saveNodeConfigurationDialog.showDialog();

                    final String workingDir = Paths.get(fileName).getParent().toString();
                    LOGGER.info("Save current workingDir: {}", workingDir);

                    wizardSettings.setWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY, workingDir);
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw new RuntimeException(Resources.getString(MainNodeListActionListener.class, ex.getReason()));
                }
                catch (IllegalArgumentException ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw ex;
                }
                catch (Exception ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw new RuntimeException("Load node configuration failed.");
                }
                finally {
                    parentComponent.setBusy(false);
                }
            }
        };
        dialog.showDialog();
    }

    protected void transformMatchingProduct(final Component parent, final NodeInterface node, final BiDiB bidib) {

        // we expect a node in the bidib instance
        if (bidib == null || bidib.getNodes() == null || bidib.getNodes().getNode().isEmpty()) {
            LOGGER.warn("The provided BiDiB data is not valid for node import.");

            throw new IllegalArgumentException("The provided BiDiB data is not valid for node import.");
        }

        final org.bidib.jbidibc.core.schema.bidib2.Node importNode = bidib.getNodes().getNode().get(0);
        LOGGER.info("Transform the bidib instance, node: {}, schemaNode: {}", node, importNode.getProductName());

        long sourceUniqueId = importNode.getUniqueId();
        long targetUniqueId = node.getUniqueId();

        if (ProductUtils.isProductEqual(targetUniqueId, sourceUniqueId)) {
            // nothing to do here
            return;
        }

        if (ProductUtils.getVendorId(sourceUniqueId) == 13 && ProductUtils.getVendorId(targetUniqueId) == 13) {
            // OpenDCC
            if ((ProductUtils.getPid(sourceUniqueId) == 142 && ProductUtils.getPid(targetUniqueId) == 140)
                || (ProductUtils.getPid(sourceUniqueId) == 141 && ProductUtils.getPid(targetUniqueId) == 117)) {
                // OneControl-04-16-48-24 --> OneControl-08-00-48-24
                // OneControl 04-16-32-20 --> OneControl 08-00-32-20
                LOGGER
                    .info(
                        "Found import of OneControl-04-16-48-24 --> OneControl-08-00-48-24 or OneControl 04-16-32-20 --> OneControl 08-00-32-20. Must change the port numbers for all ports with portnumber > 3.");
                transformOneControlOneDriveTurn(parent, importNode);
            }
            else if ((ProductUtils.getPid(sourceUniqueId) == 145 && ProductUtils.getPid(targetUniqueId) == 143)
                || (ProductUtils.getPid(sourceUniqueId) == 144 && ProductUtils.getPid(targetUniqueId) == 122)) {
                // OneControl-04-16-48-24 --> OneControl-08-00-48-24
                // OneControl 04-16-32-20 --> OneControl 08-00-32-20
                LOGGER
                    .info(
                        "Found import of OneDriveTurn-04-16-48-24 --> OneDriveTurn-08-00-48-24 or OneDriveTurn 04-16-32-20 --> OneDriveTurn 08-00-32-20. Must change the port numbers for all ports with portnumber > 3.");
                transformOneControlOneDriveTurn(parent, importNode);
            }
        }
    }

    protected void transformOneControlOneDriveTurn(
        final Component parent, final org.bidib.jbidibc.core.schema.bidib2.Node importNode) {

        JOptionPane
            .showMessageDialog(JOptionPane.getFrameForComponent(parent),
                Resources.getString(NodeExchangeHelper.class, "import-with-transformation.message"),
                Resources.getString(NodeExchangeHelper.class, "import-with-transformation.title"),
                JOptionPane.INFORMATION_MESSAGE);

        // process the ports
        if (importNode.getPorts() != null) {
            importNode.getPorts().getPort().stream().map(port -> {
                LOGGER.info("Process port: {}", port);
                if (port.getNumber() > 3) {
                    LOGGER.info("Port: change port number from {} to {}", port.getNumber(), port.getNumber() + 4);
                    port.setNumber(port.getNumber() + 4);
                }
                return port;
            }).collect(Collectors.toList());
        }
        else {
            LOGGER.warn("No ports available.");
        }

        // process the macros
        if (importNode.getMacros() != null) {
            importNode.getMacros().getMacro().forEach(macro -> {
                if (macro.getMacroPoints() != null) {
                    macro
                        .getMacroPoints().getMacroPoint().stream()
                        .filter(mp -> mp instanceof MacroPointOutput || mp instanceof MacroPointInput).map(mp -> {
                            if (mp instanceof MacroPointOutput) {
                                MacroPointOutput mpo = (MacroPointOutput) mp;
                                if (mpo.getOutputNumber() > 3) {
                                    LOGGER
                                        .info("Macro: change port number from {} to {}", mpo.getOutputNumber(),
                                            mpo.getOutputNumber() + 4);
                                    mpo.setOutputNumber(mpo.getOutputNumber() + 4);
                                }
                            }
                            else if (mp instanceof MacroPointInput) {
                                MacroPointInput mpi = (MacroPointInput) mp;
                                if (mpi.getInputNumber() > 3) {
                                    LOGGER
                                        .info("Macro: change input port number from {} to {}", mpi.getInputNumber(),
                                            mpi.getInputNumber() + 4);
                                    mpi.setInputNumber(mpi.getInputNumber() + 4);
                                }
                            }
                            return mp;
                        }).collect(Collectors.toList());
                }
            });
        }
        else {
            LOGGER.warn("No macros available.");
        }

        // process the configuration variables
        if (importNode.getConfigurationVariables() != null) {
            LOGGER.info("Process CV values.");

            Collections.sort(importNode.getConfigurationVariables().getConfigurationVariable(), (cv1, cv2) -> {

                return Integer.compare(Integer.parseInt(cv1.getName()), Integer.parseInt(cv2.getName()));
            });

            LOGGER
                .info("Number of CV values: {}",
                    importNode.getConfigurationVariables().getConfigurationVariable().size());
            if (LOGGER.isDebugEnabled()) {
                importNode
                    .getConfigurationVariables().getConfigurationVariable()
                    .forEach(cv -> LOGGER.debug("Current CV, num: {}, value: {}", cv.getName(), cv.getValue()));
            }

            // drop all cv values >= 10_000
            final List<ConfigurationVariable> filteredCVs =
                importNode
                    .getConfigurationVariables().getConfigurationVariable().stream()
                    .filter(cv -> Integer.parseInt(cv.getName()) < 10_000).collect(Collectors.toList());

            LOGGER.info("Number of filtered CV values: {}", filteredCVs.size());

            importNode.getConfigurationVariables().getConfigurationVariable().clear();
            importNode.getConfigurationVariables().getConfigurationVariable().addAll(filteredCVs);
        }
    }

    public void exportNode(
        final DefaultBusyFrame parentComponent, final NodeInterface node, final SettingsService settingsService,
        final NodeService nodeService, final SwitchingNodeService switchingNodeService,
        final WizardLabelWrapper wizardLabelWrapper, final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry,
        final FileFilter nodeFilter, final StatusBar statusBar, String lang) {

        LOGGER.info("export node: {}", node);

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();

        String fileName = prepareFileName(wizardSettings, node);

        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY);

        // export node data
        FileDialog dialog =
            new FileDialog(parentComponent, FileDialog.SAVE, storedWorkingDirectory, fileName, nodeFilter) {

                private JCheckBox checkLoadCVs;

                @Override
                public void approve(String fileName) {

                    try {
                        final ExportNodeOptionsDialog exportNodeOptionsDialog =
                            new ExportNodeOptionsDialog(parentComponent,
                                Resources.getString(NodeExchangeHelper.class, "exportOptions.title"), node) {

                                private static final long serialVersionUID = 1L;

                                @Override
                                protected Component getAdditionalPanel() {
                                    // prepare a panel with checkboxes for loading macro content before export
                                    JPanel additionalPanel = new JPanel();
                                    additionalPanel.setLayout(new BoxLayout(additionalPanel, BoxLayout.PAGE_AXIS));

                                    checkLoadCVs =
                                        new JCheckBox(Resources.getString(MainController.class, "checkLoadCVs"), true);
                                    additionalPanel.add(checkLoadCVs);
                                    return additionalPanel;
                                }
                            };
                        exportNodeOptionsDialog.setVisible(true);

                        if (exportNodeOptionsDialog.getResult() == JOptionPane.OK_OPTION) {
                            LOGGER.info("User confirmed options.");
                        }
                        else {
                            LOGGER.info("User cancelled options.");

                            return;
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Show exportNodeOptionsDialog failed.", ex);
                        return;
                    }

                    try {
                        parentComponent.setBusy(true);
                        LOGGER.info("Save node state for node: {}, fileName: {}", node, fileName);

                        // must load missing data from node
                        org.bidib.wizard.utils.NodeUtils
                            .loadDataFromNode(ConnectionRegistry.CONNECTION_ID_MAIN, nodeService, switchingNodeService,
                                node, true, checkLoadCVs.isSelected());

                        final BiDiB bidib =
                            prepareBiDiB(node, cvDefinitionTreeModelRegistry, lang, !checkLoadCVs.isSelected(),
                                wizardLabelWrapper);
                        LOGGER.info("Export node, converted node to bidib: {}", bidib);

                        BidibFactory.saveBiDiB(bidib, fileName, false);

                        LOGGER.info("Save node state passed, fileName: {}", fileName);

                        final String workingDir = Paths.get(fileName).getParent().toString();
                        LOGGER.info("Save current workingDir: {}", workingDir);

                        wizardSettings.setWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY, workingDir);

                        statusBar
                            .setStatusText(
                                String.format(Resources.getString(MainController.class, "exportedNodeState"), fileName),
                                StatusBar.DISPLAY_NORMAL);
                    }
                    finally {
                        parentComponent.setBusy(false);
                    }
                }
            };
        dialog.showDialog();
    }

    public BiDiB prepareBiDiB(
        final NodeInterface node, final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry, String lang,
        boolean skipCvValues, final WizardLabelWrapper wizardLabelWrapper) {
        final CvContainer cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(node.getUniqueId());

        final Map<String, CvNode> cvNumberToNodeMap =
            cvContainer != null ? cvContainer.getCvNumberToNodeMap() : new HashMap<>();

        final BiDiB bidib =
            org.bidib.wizard.utils.NodeUtils
                .convertToBiDiB(node, cvNumberToNodeMap, lang, skipCvValues, wizardLabelWrapper);
        LOGGER.debug("prepareBiDiB, converted node to bidib: {}", bidib);
        return bidib;
    }

    public static String prepareFileName(final WizardSettingsInterface wizardSettings, final NodeInterface node) {

        String nodeLabel = org.bidib.wizard.utils.NodeUtils.prepareLabel(node);

        boolean nodeExportAppendDateEnabled = wizardSettings.isNodeExportAppendDateEnabled();

        if (nodeExportAppendDateEnabled) {
            LocalDateTime now = LocalDateTime.now();
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("_yyyyMMdd_HHmmss");
            String currentDateTime = now.format(formatter);

            nodeLabel = nodeLabel + currentDateTime;
        }

        LOGGER.info("export node, prepared nodeLabel as filename: {}", nodeLabel);

        if (!FileUtils.isPathValid(nodeLabel) || !FileUtils.isFilenameValid(nodeLabel)) {
            LOGGER.warn("The default filename is not a valid filename: {}", nodeLabel);
            nodeLabel = FileUtils.escapeInvalidFilenameCharacters(nodeLabel, "_");
            LOGGER.info("export node, prepared escaped filename: {}", nodeLabel);
        }

        return nodeLabel + "." + NodeExchangeHelper.NODE_X_EXTENSION;
    }

}
