package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Objects;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.model.settings.WizardSettings;
import org.bidib.wizard.mvc.main.controller.MacroPanelController;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SelectedMacroModel;
import org.bidib.wizard.mvc.main.model.listener.MacroSelectionListener;
import org.bidib.wizard.mvc.main.view.menu.listener.MacroListMenuListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;

public class MacroPanel extends JPanel {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(MacroPanel.class);

    private static final String ENCODED_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, pref:grow";

    private static final String ENCODED_ROW_SPECS = "pref, 3dlu, pref, 3dlu, fill:200px:grow, 3dlu, pref";

    private final MacroPanelController macroPanelController;

    private final MainModel mainModel;

    private final SettingsService settingsService;

    private final MacroListMenuListener macroListMenuListener;

    private final SelectedMacroModel selectedMacroModel;

    private final ValueModel selectedMacroValueModel;

    private final PropertyChangeListener labelChangedListener;

    private final MacroParameterPanel macroParameterPanel;

    private final MacroContentPanel macroContentPanel;

    private final JPanel buttonPanel;

    private Boolean showPowerUser;

    private final static class MacroConverter implements BindingConverter<Macro, String> {

        @Override
        public String targetValue(Macro sourceValue) {
            return Objects.toString(sourceValue, null);
        }

        @Override
        public Macro sourceValue(String targetValue) {
            return null;
        }
    }

    public MacroPanel(final MacroPanelController macroPanelController, final MainModel mainModel,
        final SettingsService settingsService, final MacroListMenuListener macroListMenuListener) {
        this.macroPanelController = macroPanelController;
        this.mainModel = mainModel;
        this.settingsService = settingsService;
        this.macroListMenuListener = macroListMenuListener;

        setLayout(new BorderLayout());

        selectedMacroModel = new SelectedMacroModel();

        labelChangedListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("The label has changed.");

                selectedMacroModel.triggerLabelChanged();
            }
        };

        final FormBuilder formBuilder = FormBuilder.create().columns(ENCODED_COLUMN_SPECS).rows(ENCODED_ROW_SPECS);

        formBuilder.border(BorderFactory.createEmptyBorder(5, 5, 5, 5));

        selectedMacroValueModel =
            new PropertyAdapter<SelectedMacroModel>(selectedMacroModel, SelectedMacroModel.PROPERTY_SELECTED_MACRO,
                true);

        ValueModel valueConverterModel = new ConverterValueModel(selectedMacroValueModel, new MacroConverter());

        JLabel selectedMacroLabel = WizardComponentFactory.createLabel(valueConverterModel);

        formBuilder.add(Resources.getString(MacroListPanel.class, "macroName")).xy(1, 1);
        formBuilder.add(selectedMacroLabel).xyw(3, 1, 3);

        macroParameterPanel = new MacroParameterPanel(mainModel);

        formBuilder.add(macroParameterPanel).xyw(1, 3, 5);

        macroContentPanel = new MacroContentPanel(mainModel);
        formBuilder.add(macroContentPanel).xyw(1, 5, 5);

        this.buttonPanel = new JPanel();
        buttonPanel.setLayout(new GridLayout(1, 0, 5, 0));

        formBuilder.add(buttonPanel).xyw(1, 7, 5);

        createAndAddButtonPanel(settingsService.getWizardSettings().isPowerUser());

        add(formBuilder.build(), BorderLayout.CENTER);

        this.mainModel.addMacroSelectionListener(new MacroSelectionListener() {

            @Override
            public void macroChanged() {
                LOGGER.info("The selected macro has changed.");

                // the selected accessory has changed
                Macro selectedAccessory = selectedMacroModel.getSelectedMacro();
                if (selectedAccessory != null) {
                    selectedAccessory.removePropertyChangeListener(Macro.PROPERTY_LABEL, labelChangedListener);
                }

                // get the new selected macro
                final Macro macro = MacroPanel.this.mainModel.getSelectedMacro();

                if (macro != null) {
                    macro.addPropertyChangeListener(Macro.PROPERTY_LABEL, labelChangedListener);
                }
                else {
                    LOGGER.info("No macro selected.");
                    selectedMacroModel.triggerLabelChanged();
                }

                selectedMacroModel.setSelectedMacro(macro);
            }
        });

        try {
            final WizardSettingsInterface ws = settingsService.getWizardSettings();

            ws.addPropertyChangeListener(WizardSettings.PROPERTY_POWER_USER, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    Object value = evt.getNewValue();
                    if (value instanceof Boolean) {
                        boolean powerUser = (Boolean) value;
                        LOGGER.info("The power user flag has changed: {}", powerUser);
                        createAndAddButtonPanel(powerUser);
                    }
                }
            });
        }
        catch (Exception ex) {
            LOGGER.warn("Add property change listeners failed.", ex);
        }

    }

    private void createAndAddButtonPanel(boolean isPowerUser) {
        if (buttonPanel != null && showPowerUser == Boolean.valueOf(isPowerUser)) {
            LOGGER.info("Power user flag has not changed.");
            return;
        }

        if (isPowerUser) {
            preparePowerUserButtonPanel(this.buttonPanel);
        }
        else {
            prepareNormalUserButtonPanel(this.buttonPanel);
        }
        showPowerUser = Boolean.valueOf(isPowerUser);

        buttonPanel.validate();
        buttonPanel.repaint();
    }

    private JPanel prepareNormalUserButtonPanel(final JPanel buttonPanel) {

        buttonPanel.removeAll();

        JButton reloadButton = new JButton(Resources.getString(MacroListPanel.class, "load"));
        reloadButton.setToolTipText(Resources.getString(MacroListPanel.class, "load.tooltip"));

        reloadButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.reloadMacro();
            }
        });
        buttonPanel.add(reloadButton);

        JButton saveButton = new JButton(Resources.getString(MacroListPanel.class, "save"));
        saveButton.setToolTipText(Resources.getString(MacroListPanel.class, "save.tooltip"));

        saveButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.saveMacro();
            }
        });
        buttonPanel.add(saveButton);

        JButton testButton = new JButton(Resources.getString(MacroListPanel.class, "test"));
        testButton.setToolTipText(Resources.getString(MacroListPanel.class, "test.tooltip"));

        // test macro
        testButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.testMacro(false);
            }
        });
        buttonPanel.add(testButton);

        JButton stopButton = new JButton(Resources.getString(MacroListPanel.class, "stop"));
        stopButton.setToolTipText(Resources.getString(MacroListPanel.class, "stop.tooltip"));

        stopButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.stopMacro();
            }
        });
        buttonPanel.add(stopButton);

        return buttonPanel;
    }

    private JPanel preparePowerUserButtonPanel(final JPanel buttonPanel) {

        buttonPanel.removeAll();

        JButton reloadButton = new JButton(Resources.getString(MacroListPanel.class, "reload"));
        reloadButton.setToolTipText(Resources.getString(MacroListPanel.class, "reload.tooltip"));

        reloadButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.reloadMacro();
            }
        });
        buttonPanel.add(reloadButton);

        JButton transferButton = new JButton(Resources.getString(MacroListPanel.class, "transfer"));
        transferButton.setToolTipText(Resources.getString(MacroListPanel.class, "transfer.tooltip"));

        transferButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("Transfer button was pressed.");
                MacroPanel.this.macroListMenuListener.transferMacro();
            }
        });
        buttonPanel.add(transferButton);

        JButton testButton = new JButton(Resources.getString(MacroListPanel.class, "testPower"));
        testButton.setToolTipText(Resources.getString(MacroListPanel.class, "testPower.tooltip"));

        // test macro
        testButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.testMacro(true);
            }
        });
        buttonPanel.add(testButton);

        JButton stopButton = new JButton(Resources.getString(MacroListPanel.class, "stop"));
        stopButton.setToolTipText(Resources.getString(MacroListPanel.class, "stop.tooltip"));

        stopButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.stopMacro();
            }
        });
        buttonPanel.add(stopButton);

        JButton saveButton = new JButton(Resources.getString(MacroListPanel.class, "save"));
        saveButton.setToolTipText(Resources.getString(MacroListPanel.class, "save.tooltip"));

        saveButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.saveMacro();
            }
        });
        buttonPanel.add(saveButton);

        JButton remoteStartButton = new JButton(Resources.getString(MacroListPanel.class, "remoteStart"));
        remoteStartButton.setToolTipText(Resources.getString(MacroListPanel.class, "remoteStart.tooltip"));

        remoteStartButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MacroPanel.this.macroListMenuListener.remoteStartMacro();
            }
        });
        buttonPanel.add(remoteStartButton);

        return buttonPanel;
    }

    public void refreshView() {
        LOGGER.info("Refresh the macro panel.");

        this.macroContentPanel.refreshView();
    }
}
