package org.bidib.wizard.mvc.worklist.controller.actions;

import java.util.Locale;

import javax.swing.JOptionPane;
import javax.swing.UIManager;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.core.schema.bidibbase.DefaultLabelsActionType;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.core.schema.exception.InvalidContentException;
import org.bidib.jbidibc.core.schema.exception.InvalidSchemaException;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.enums.PortModelEnum;
import org.bidib.jbidibc.messages.exception.UnexpectedCharacterException;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.event.DefaultLabelsWorkListItemEvent;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.NodeProvider;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.common.exception.ConnectionException;
import org.bidib.wizard.common.labels.AccessoryLabelUtils;
import org.bidib.wizard.common.labels.BidibLabelUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.worklist.controller.ActionAbortedException;
import org.bushe.swing.event.EventBus;
import org.oxbow.swingbits.dialog.task.TaskDialogs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;

import com.vlsolutions.swing.docking.DockingDesktop;

public class ApplyDefaultLabelsAction implements WorkListAction<DefaultLabelsWorkListItemEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ApplyDefaultLabelsAction.class);

    private final ConnectionService connectionService;

    private final WizardLabelWrapper wizardLabelWrapper;

    private final ApplicationEventPublisher applicationEventPublisher;

    private final ConsoleService consoleService;

    public ApplyDefaultLabelsAction(final ConnectionService connectionService,
        final WizardLabelWrapper wizardLabelWrapper, final ApplicationEventPublisher applicationEventPublisher,
        final ConsoleService consoleService) {
        this.connectionService = connectionService;
        this.wizardLabelWrapper = wizardLabelWrapper;
        this.applicationEventPublisher = applicationEventPublisher;
        this.consoleService = consoleService;
    }

    @Override
    public void apply(final DockingDesktop desktop, final DefaultLabelsWorkListItemEvent event)
        throws ActionAbortedException {

        final Long uniqueId = event.getUniqueId();
        final String connectionId = event.getConnectionId();

        LOGGER
            .info("Apply the default labels for node with uniqueId: {}, connectionId: {}",
                ByteUtils.getUniqueIdAsString(uniqueId), connectionId);

        try {

            final NodeProvider nodeProvider = this.connectionService.find(connectionId).getNodeProvider();
            // apply labels to node before apply default labels
            final NodeLabels nodeLabels = wizardLabelWrapper.loadLabels(uniqueId);

            askApplyDefaultLabels(desktop, nodeProvider, uniqueId, nodeLabels);
        }
        catch (ConnectionException ex) {
            LOGGER.warn("No connection available.");
            TaskDialogs
                .error(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                    Resources.getString(ApplyDefaultLabelsAction.class, "error-connection-not-available"),
                    Resources.getString(ApplyDefaultLabelsAction.class, "action-not-performed"));

            throw new ActionAbortedException(
                "The connection for the referenced node is not available: " + connectionId);
        }
        catch (UnexpectedCharacterException ex) {
            LOGGER.warn("Apply default labels failed.", ex);
            // show error dialog
            TaskDialogs
                .build(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                    Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.instruction"),
                    Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.text", ex.getMessage()))
                .title(Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.title"))
                .showException(ex);

            throw new ActionAbortedException("Apply default labels failed.");
        }
        catch (InvalidContentException | InvalidSchemaException ex) {
            LOGGER.warn("Apply default labels failed.", ex);
            // show error dialog
            TaskDialogs
                .build(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                    Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.instruction"),
                    Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.text", ex.getMessage()))
                .title(Resources.getString(ApplyDefaultLabelsAction.class, "load_from_file_failed.title"))
                .showException(ex);

            throw new ActionAbortedException("Apply default labels failed.");
        }
    }

    private void askApplyDefaultLabels(
        final DockingDesktop desktop, final NodeProvider nodeProvider, final Long uniqueId, final NodeLabels nodeLabels)
        throws ActionAbortedException {

        final NodeInterface node =
            org.bidib.wizard.utils.NodeUtils.findNodeByUuid(nodeProvider.getNodes(), uniqueId.longValue());

        if (node == null) {
            LOGGER.warn("The referenced node is not available.");
            TaskDialogs
                .error(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                    Resources.getString(ApplyDefaultLabelsAction.class, "error-node-not-available"),
                    Resources.getString(ApplyDefaultLabelsAction.class, "action-not-performed"));
            throw new ActionAbortedException(
                "The referenced node is not available: " + ByteUtils.getUniqueIdAsString(uniqueId));
        }

        int relevantPidBits = node.getNode().getRelevantPidBits();

        BidibNodeNameUtils.NodeLabelData labelData = BidibNodeNameUtils.prepareLabel(node, nodeLabels, true, false);

        Locale l = Locale.getDefault();
        String okButtonText = UIManager.getString("OptionPane.okButtonText", l);
        String ignoreButtonText = Resources.getString(ApplyDefaultLabelsAction.class, "ignore-button.text");

        Object[] options = { okButtonText, ignoreButtonText };

        int result =
            JOptionPane
                .showOptionDialog(JOptionPane.getFrameForComponent(desktop.getRootPane()), Resources
                    .getString(ApplyDefaultLabelsAction.class, "apply-default-labels.text", labelData.getNodeLabel()),
                    Resources.getString(ApplyDefaultLabelsAction.class, "apply-default-labels.title"),
                    JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[0]);

        String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
        final PortModelEnum portModel = PortModelEnum.getPortModel(node.getNode());
        final SoftwareVersion softwareVersion = node.getNode().getSoftwareVersion();

        if (result == JOptionPane.YES_OPTION) {
            LOGGER.info("Apply the default labels on node: {}", node);

            // apply the default labels
            wizardLabelWrapper
                .setDefaultLabelsApplied(lang, uniqueId, softwareVersion, relevantPidBits, portModel,
                    DefaultLabelsActionType.APPLIED);

            try {
                // and reload the labels
                reloadLabels(node);
            }
            catch (Exception ex) {
                LOGGER.warn("Load labels from BiDiB failed.", ex);

                // display warning in console
                ConsoleController.ensureConsoleVisible();

                // show an error message in the console
                consoleService
                    .addConsoleLine(ConsoleColor.red,
                        String
                            .format("Load lables failed for node with uniqueId: %s",
                                ByteUtils.getUniqueIdAsString(uniqueId)));
            }

            node.setLabel(BidibLabelUtils.getNodeLabel(nodeLabels));

            // change labels of accessories
            if (CollectionUtils.isNotEmpty(node.getAccessories())) {

                for (Accessory accessory : node.getAccessories()) {

                    final AccessoryLabel labelType =
                        AccessoryLabelUtils.getAccessoryLabel(nodeLabels, accessory.getId());
                    String accessoryLabel = labelType.getLabel();
                    LOGGER.info("Set the label of accessory id: {}, label: {}", accessory.getId(), accessoryLabel);
                    accessory.setLabel(accessoryLabel);
                }
            }

            // force a reload of the changed labels
            final LabelsChangedEvent labelsChangedEvent = new LabelsChangedEvent(uniqueId);
            LOGGER.info("Publish the labelsChangedEvent: {}", labelsChangedEvent);
            EventBus.publish(labelsChangedEvent);

            this.applicationEventPublisher.publishEvent(labelsChangedEvent);
        }
        else if (result == JOptionPane.NO_OPTION) {
            LOGGER.info("User declined to apply the default labels on node: {}", node);

            wizardLabelWrapper
                .setDefaultLabelsApplied(lang, uniqueId, softwareVersion, relevantPidBits, portModel,
                    DefaultLabelsActionType.IGNORED);
        }
        else {
            LOGGER.info("Apply the default labels dialog was cancelled.");
            throw new ActionAbortedException("Apply the default labels dialog was cancelled.");
        }
    }

    private void reloadLabels(final NodeInterface node) {
        // load the labels of the node
        LOGGER.info("Load labels for node: {}", node);

        if (node == null) {
            LOGGER.info("No node selected to load the labels.");
            return;
        }

        // load all labels
        wizardLabelWrapper.loadLabels(node.getUniqueId());

        LOGGER.info("Finished load labels for node: {}", node);
    }
}
