package org.bidib.wizard.mvc.locolist.controller;

import java.awt.Point;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.AddressTypeEnum;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.CsQueryTypeEnum;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.model.listener.NodeSelectionListener;
import org.bidib.wizard.api.notification.LocoListChangeAction;
import org.bidib.wizard.api.service.core.LocoService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.api.utils.NodeUtils;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.PomProgrammerControllerFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.core.service.node.BidibNodeService;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.locolist.controller.listener.LocoTableControllerListener;
import org.bidib.wizard.mvc.locolist.model.LocoLibraryModel;
import org.bidib.wizard.mvc.locolist.model.LocoLibraryModel.ExecutionResult;
import org.bidib.wizard.mvc.locolist.model.LocoTableModel;
import org.bidib.wizard.mvc.locolist.view.LocoTableView;
import org.bidib.wizard.mvc.main.controller.MainControllerInterface;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.pom.controller.PomProgrammerController;
import org.bidib.wizard.mvc.pom.controller.listener.PomProgrammerControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;

public class LocoTableController implements LocoTableControllerListener, PropertyChangeListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoTableController.class);

    private final JFrame parent;

    private final MainModel mainModel;

    private MainControllerInterface mainController;

    private LocoTableView locoTableView;

    private LocoTableModel locoTableModel;

    private NodeInterface selectedNode;

    @Autowired
    private BidibNodeService bidibNodeService;

    @Autowired
    private ConnectionService connectionService;

    @Autowired
    private CommandStationService commandStationService;

    @Autowired
    private LocoService locoService;

    private LocoLibraryModel locoLibrary;

    private final CompositeDisposable disp = new CompositeDisposable();

    private final DockingDesktop desktop;

    private DockableStateChangeListener dockableStateChangeListener;

    public LocoTableController(NodeInterface node, final DockingDesktop desktop, JFrame parent,
        final MainModel mainModel) {
        this.parent = parent;
        this.desktop = desktop;
        this.mainModel = mainModel;
    }

    public void start(final MainControllerInterface mainController) {

        // check if the loco table view is already opened
        String searchKey = DockKeys.LOCO_TABLE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing loco list table view.");
            DockUtils.selectWindow(view);
            return;
        }

        // keep the mainController instance
        this.mainController = mainController;

        LOGGER.info("Create new LocoTableView.");

        this.compDisp = new CompositeDisposable();

        locoTableModel = new LocoTableModel(this);

        locoTableView = new LocoTableView(this, locoTableModel);

        // add the loco panel next to the booster panel
        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState boosterTableView = null;
            // search the booster table view
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_BOOSTER_TABLE_VIEW.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the booster table view dockable.");
                    boosterTableView = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (boosterTableView != null) {
                LOGGER.info("Add the loco panel view to the booster table view panel.");
                dock = boosterTableView.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new loco panel at order: {}", order);

                desktop.createTab(dock, locoTableView, order, true);
                desktop.setDockableHeight(locoTableView, 0.3d);
            }
            else {
                desktop.split(dock, locoTableView, DockingConstants.SPLIT_RIGHT);
                desktop.setDockableHeight(locoTableView, 0.3d);
            }
        }
        else {
            desktop.addDockable(locoTableView, RelativeDockablePosition.RIGHT);
        }

        // create the nodeList listener
        final NodeListListener nodeListListener = new DefaultNodeListListener() {

            @Override
            public void listNodeAdded(NodeInterface node) {
                LOGGER.info("The nodelist has a new node: {}", node);

                // nodeNew(node);
            }

            @Override
            public void listNodeRemoved(NodeInterface node) {
                LOGGER.info("The nodelist has a node removed: {}", node);
                nodeLost(node);
            }
        };
        // register as nodeList listener at the main controller
        this.mainController.addNodeListListener(nodeListListener);

        NodeSelectionListener nodeSelectionListener = new NodeSelectionListener() {

            @Override
            public void selectedNodeChanged(final NodeInterface selectedNode) {
                LOGGER.info("The selected node has changed: {}", selectedNode);
                setSelectedNode(selectedNode);
            }
        };
        this.mainController.addNodeSelectionListListener(nodeSelectionListener);

        try {
            Disposable dispConnStatus = connectionService.subscribeConnectionStatusChanges(connectionInfo -> {

                if (connectionInfo.getConnectionId().equals(ConnectionRegistry.CONNECTION_ID_MAIN)) {
                    LOGGER.info("Current state: {}", connectionInfo.getConnectionState());

                    switch (connectionInfo.getConnectionState().getActualPhase()) {
                        case CONNECTED:
                            LOGGER.info("The communication was opened.");
                            break;
                        case DISCONNECTED:
                            LOGGER.info("The communication was closed. Remove all locos from the table.");

                            SwingUtilities.invokeLater(() -> {
                                cleanup();
                            });
                            break;
                        default:
                            break;
                    }
                }

            }, error -> {
                LOGGER.warn("The connection status change caused an error.", error);
            });

            this.disp.add(dispConnStatus);
        }
        catch (Exception ex) {
            LOGGER.warn("Register controller as connection status listener failed.", ex);
        }

        this.dockableStateChangeListener = new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(locoTableView) && event.getNewState().isClosed()) {
                    LOGGER.info("LocoTableView was closed, free resources.");

                    try {
                        desktop.removeDockableStateChangeListener(dockableStateChangeListener);
                    }
                    catch (Exception ex) {
                        LOGGER
                            .warn("Remove dockableStateChangeListener from desktop failed: "
                                + dockableStateChangeListener, ex);
                    }
                    finally {
                        dockableStateChangeListener = null;
                    }

                    try {
                        // remove node listener from communication factory
                        if (nodeListListener != null) {
                            mainController.removeNodeListListener(nodeListListener);
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Unregister controller as node listener failed.", ex);
                    }

                    try {
                        // remove node listener from communication factory
                        if (nodeSelectionListener != null) {
                            mainController.removeNodeSelectionListener(nodeSelectionListener);
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Unregister controller as node selection listener failed.", ex);
                    }

                    final ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
                    LocoTableController locoTableController =
                        applicationContext
                            .get(DefaultApplicationContext.KEY_LOCO_TABLE_CONTROLLER, LocoTableController.class);

                    if (locoTableController == LocoTableController.this) {
                        LOGGER.info("Unregister the LocoTableController from the context.");
                        applicationContext.unregister(DefaultApplicationContext.KEY_LOCO_TABLE_CONTROLLER);
                    }

                }
            }
        };
        desktop.addDockableStateChangeListener(this.dockableStateChangeListener);
    }

    private void cleanup() {

        if (this.compDisp != null) {
            LOGGER.info("Dispose the subscription to locoList actions.");
            this.compDisp.dispose();
            this.compDisp = null;
        }

        locoTableModel.removeAllLocos();
        setSelectedNode(null);

        LOGGER.info("Close the locoTableView.");
        desktop.close(locoTableView);
    }

    private void nodeLost(final NodeInterface node) {

        if (node != null && node.equals(selectedNode)) {
            LOGGER.info("The selected node was removed. Clear the list of loco for node: {}", node);
            if (SwingUtilities.isEventDispatchThread()) {
                cleanup();
            }
            else {
                SwingUtilities.invokeLater(() -> {
                    cleanup();
                });
            }
        }
    }

    private CompositeDisposable compDisp;

    public void setSelectedNode(final NodeInterface node) {
        LOGGER.info("Set the selected node: {}", node);

        if (node != null && org.bidib.jbidibc.messages.utils.NodeUtils.hasCommandStationFunctions(node.getUniqueId())) {
            this.selectedNode = node;
            locoTableModel.setCsNodeSelected(true);

            if (locoTableView != null) {

                locoTableView.setDockTabName(node);

                // get the locolist of the node from the service
                // final List<LocoListModel> locoList =
                // this.locoService.getLocoList(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode());
                // this.locoTableModel.setLocoListModel(locoList);
                // this.locoTableView.setLocoListModel(locoList);

                // load the locolist from the service
                final List<LocoListModel> locoList =
                    this.locoService.loadLocoList(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode());
                this.locoTableModel.setLocoListModel(locoList);

                final Disposable dispConnectionActions = locoService.subscribeLocoListActions(ca -> {
                    if (ConnectionRegistry.CONNECTION_ID_MAIN.equals(ca.getConnectionId())) {

                        processLocoListAction(ca);
                    }
                }, error -> {
                    LOGGER.warn("The locoList actions signalled a failure: {}", error);
                });
                compDisp.add(dispConnectionActions);

                // TODO make sure we load the current values until the changes of loco values are not updated by
                // commands
                if (this.selectedNode != null) {
                    queryCommandStationValue(this.selectedNode, CsQueryTypeEnum.LOCO_LIST, null);
                }
            }
        }
        else {
            this.selectedNode = null;
            locoTableModel.setCsNodeSelected(false);
        }
    }

    private void processLocoListAction(final LocoListChangeAction ca) {
        SwingUtilities.invokeLater(() -> this.locoTableModel.processLocoAction(ca));
    }

    @Override
    public void queryCommandStationList(CsQueryTypeEnum csQueryType, Integer locoAddress) {
        LOGGER
            .info("Remove all locos from table before query the loco list from command station, locoAddress: {}",
                locoAddress);

        queryCommandStationValue(selectedNode, csQueryType, locoAddress);
    }

    private void queryCommandStationValue(final NodeInterface node, CsQueryTypeEnum csQueryType, Integer locoAddress) {

        LOGGER
            .info("Query the loco list from command station, csQueryType: {}, locoAddress: {}", csQueryType,
                locoAddress);

        if (node != null && node.getCommandStationNode() != null) {
            try {

                // trigger the command station value for loco list
                this.locoService
                    .queryCommandStationList(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode(),
                        CsQueryTypeEnum.LOCO_LIST, locoAddress);
            }
            catch (Exception ex) {
                LOGGER.warn("Query the loco list from command station failed.", ex);

                // TODO set an error flag or something in the node
            }
        }
        else {
            LOGGER.warn("No command station node available.");
        }

    }

    /**
     * Push the loco data to the loco database in the mouse
     * 
     * @param entryId
     *            the entry id, allowed values {@code 1..8}
     * @param speedSteps
     *            the speed steps
     * @param address
     *            the address of the loco
     * @param locoName
     *            the name of the loco
     */
    @Override
    public void pushLocoToMouse(int entryId, int speedSteps, int address, String locoName) {

        // MSG_VENDOR_SET "XP_LOC_0/7" "128\1234\RE485"
        //
        // Der V_NAME ist also die Xpressnet Nachricht: Lokdatenbank, Eintrag 0 von 0..7.
        // Der V_VALUE ist der Inhalt: Fahrstufen 128, Adresse 1234, Name RE485.

        if (entryId < 0 && entryId > 7) {
            throw new IllegalArgumentException("entryId is limited to values between 0..7");
        }

        // max len of name
        int maxWidth = 10;

        String vendorName = String.format("XP_LOC_%d/8", entryId);
        String vendorValue = String.format("%d\\%d\\%s", speedSteps, address, StringUtils.truncate(locoName, maxWidth));

        if (selectedNode != null) {
            try {
                List<ConfigurationVariable> cvList = new LinkedList<>();

                // prepare the list of CV values
                cvList.add(new ConfigurationVariable(vendorName, vendorValue));

                // trigger the vendor message
                List<ConfigurationVariable> configVars =
                    bidibNodeService.setConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, cvList);

                if (CollectionUtils.isEmpty(configVars)) {
                    LOGGER.warn("No changed CV values returned. Notify the user.");
                    throw new RuntimeException("No changed CV values returned. Push loco data to mouse failed.");
                }
                LOGGER.info("Received configVars: {}", configVars);
            }
            catch (Exception ex) {
                LOGGER.warn("Push the loco name to the mouse failed.", ex);

                // TODO set an error flag or something in the node
            }
        }
    }

    @Override
    public void setLocoLibraryModel(LocoLibraryModel locoLibrary) {

        if (this.locoLibrary != null) {
            this.locoLibrary.removePropertyChangeListener(LocoLibraryModel.PROPERTY_PUSH_TO_MOUSE, this);
        }

        this.locoLibrary = locoLibrary;

        if (this.locoLibrary != null) {
            this.locoLibrary.addPropertyChangeListener(LocoLibraryModel.PROPERTY_PUSH_TO_MOUSE, this);
        }
    }

    private final ScheduledExecutorService locoLibraryWorker =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("locoLibraryWorkers-thread-%d").build());

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (Boolean.TRUE.equals(evt.getNewValue())) {
            LOGGER.info("The push to mouse is now enabled. Start the process to publish the loco library.");

            if (locoLibrary != null) {

                int delay = 200;
                // use executor to send response
                locoLibraryWorker.schedule(() -> {
                    LOGGER.info("Start push loco library to mouse.");

                    try {
                        locoLibrary.setExecutionStatus("Start push loco library to mouse.");
                        locoLibrary.setExecutionResult(ExecutionResult.PENDING);

                        int entryId = 0;
                        for (LocoListModel locoModel : locoLibrary.getLocoList()) {
                            LOGGER.info("Push loco data to mouse, loco: {}, entryId: {}", locoModel, entryId);

                            if (locoModel != null) {
                                String locoName =
                                    StringUtils.isNotBlank(locoModel.getLocoName()) ? locoModel.getLocoName()
                                        : Integer.toString(locoModel.getLocoAddress());
                                locoLibrary.setExecutionStatus("Push loco: " + locoName);

                                pushLocoToMouse(entryId, SpeedSteps.valueOf(locoModel.getSpeedSteps()),
                                    locoModel.getLocoAddress(), locoName);

                                LOGGER.info("Wait after push.");
                                try {
                                    Thread.sleep(50);
                                }
                                catch (InterruptedException ex) {
                                    LOGGER.warn("Wait 50ms after push the loco to the mouse was interrupted.", ex);
                                }
                            }
                            else {
                                LOGGER.info("No loco at index: {}", entryId);
                            }
                            entryId++;
                        }

                        LOGGER.info("Finished push loco library to mouse.");
                        locoLibrary.setExecutionStatus("Finished push loco library to mouse.");

                        locoLibrary.setExecutionResult(ExecutionResult.PASSED);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Push loco library to mouse failed.", ex);

                        locoLibrary.setExecutionResult(ExecutionResult.FAILED);
                    }
                    finally {
                        locoLibrary.setPushToMouse(false);
                    }
                }, delay, TimeUnit.MILLISECONDS);
            }
            else {
                LOGGER.info("The push to mouse is not enabled.");
            }
        }
    }

    @Override
    public void openLocoDialog(final LocoListModel locoModel) {
        LOGGER.info("Open the loco dialog for locoModel: {}", locoModel);

        CommandStationNodeInterface node =
            NodeUtils.findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final LocoControllerFactory locoControllerFactory = ctx.getBean(LocoControllerFactory.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node, (JFrame) JOptionPane.getFrameForComponent(parent),
                        mainModel.getNodeProvider());

            AddressData initialAddress = null;
            if (locoModel != null) {
                initialAddress =
                    new AddressData(locoModel.getLocoAddress(), locoModel.getDirection() == DirectionStatus.FORWARD
                        ? AddressTypeEnum.LOCOMOTIVE_FORWARD : AddressTypeEnum.LOCOMOTIVE_BACKWARD);
            }
            locoController.start(initialAddress, locoModel != null ? locoModel.getSpeedSteps() : null, locoModel);

        }
    }

    @Override
    public void openPomDialog(final LocoListModel locoModel) {
        LOGGER.info("Open the POM dialog, locoModel: {}", locoModel);

        //
        CommandStationNodeInterface node =
            NodeUtils.findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final PomProgrammerControllerFactory pomProgrammerControllerFactory =
                ctx.getBean(PomProgrammerControllerFactory.class);
            final PomProgrammerController pomProgrammerController =
                pomProgrammerControllerFactory
                    .createPomProgrammerController(node, (JFrame) JOptionPane.getFrameForComponent(parent),
                        new Point(0, 0));

            pomProgrammerController.addPomProgrammerControllerListener(new PomProgrammerControllerListener() {

                @Override
                public void sendRequest(
                    CommandStationNodeInterface node, PomAddressData locoAddress, CommandStationPom opCode,
                    int cvNumber, int cvValue) {
                    LOGGER.info("Send POM request.");

                    PomAcknowledge pomAck =
                        commandStationService
                            .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node, locoAddress, opCode,
                                cvNumber, cvValue);
                    LOGGER.info("Received pomAck: {}", pomAck);
                }

                @Override
                public void close() {
                }
            });

            AddressData initialAddress = null;
            if (locoModel != null) {
                initialAddress =
                    new AddressData(locoModel.getLocoAddress(), locoModel.getDirection() == DirectionStatus.FORWARD
                        ? AddressTypeEnum.LOCOMOTIVE_FORWARD : AddressTypeEnum.LOCOMOTIVE_BACKWARD);
            }

            final DockingDesktop desktop =
                DefaultApplicationContext
                    .getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);

            // open the dialog
            pomProgrammerController.start(desktop, initialAddress);
        }

    }
}
