package org.bidib.wizard.mvc.main.controller;

import java.awt.Frame;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.BiPredicate;

import javax.swing.JOptionPane;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.Int16PortConfigValue;
import org.bidib.jbidibc.messages.port.Int32PortConfigValue;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.jbidibc.messages.port.RgbPortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.event.MacroChangedEvent;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroSaveState;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.SwitchingNodeInterface;
import org.bidib.wizard.api.model.listener.CvDefinitionListener;
import org.bidib.wizard.api.model.listener.CvDefinitionRequestListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.LightPortListener;
import org.bidib.wizard.api.script.ScriptCommand;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.client.common.view.BusyFrame;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeTableModel;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.cvdef.CvNodeUtils;
import org.bidib.wizard.client.common.view.cvdef.LongCvNode;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.BidibLabelUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.script.DefaultScriptContext;
import org.bidib.wizard.common.script.common.WaitCommand;
import org.bidib.wizard.common.script.engine.ScriptEngine;
import org.bidib.wizard.common.script.switching.LightPortCommand;
import org.bidib.wizard.common.script.switching.PortScripting;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.PortTypeAware;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.LightPortStatus;
import org.bidib.wizard.mvc.common.view.cvdefinition.CvValueUtils;
import org.bidib.wizard.mvc.main.controller.wrapper.NodePortWrapper;
import org.bidib.wizard.mvc.main.model.LightPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.LightPortModelListener;
import org.bidib.wizard.mvc.main.view.component.InsertPortsConfirmDialog;
import org.bidib.wizard.mvc.main.view.component.RemovePortConfirmDialog;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.LightPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.event.EventListener;

import com.jidesoft.grid.DefaultExpandableRow;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class LightPortPanelController
    implements CvDefinitionListener, PortScripting, InsertPortsAware<LightPortTableModel> {

    private static final Logger LOGGER = LoggerFactory.getLogger(LightPortPanelController.class);

    // NeoControl
    public static final String KEYWORD_CHANNEL_B_START_NUMBER = "channel_B_start_number";

    // NeoEWS
    public static final String KEYWORD_CHANNEL_A_LAST_ELEMENT = "channel_A_last_element";

    private final MainModel mainModel;

    private LightPortListPanel lightPortListPanel;

    private final Map<NodeInterface, NodePortWrapper> testToggleRegistry = new HashMap<>();

    private Map<String, CvNode> mapKeywordToNode;

    private List<CvDefinitionRequestListener> cvDefinitionRequestListeners = new ArrayList<>();

    private List<ConfigurationVariable> requiredConfigVariables = new ArrayList<>();

    private Integer channelBStartNumber;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private MacroPanelController macroPanelController;

    @Autowired
    private ConsoleService consoleService;

    private LightPortModelListener lightPortModelListener;

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject = PublishSubject.create();

    private CompositeDisposable compDisp;

    private final ApplicationEventPublisher applicationEventPublisher;

    private final StatusBar statusBar;

    private NodeInterface selectedNode;

    public LightPortPanelController(final MainModel mainModel,
        final ApplicationEventPublisher applicationEventPublisher, final StatusBar statusBar) {
        this.mainModel = mainModel;
        this.applicationEventPublisher = applicationEventPublisher;
        this.statusBar = statusBar;

        this.compDisp = new CompositeDisposable();
    }

    public LightPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final LightPortTableModel tableModel = new LightPortTableModel();

        tableModel.setPortListener(lightPortModelListener = new LightPortModelListener() {

            @Override
            public void labelChanged(LightPort port, String label) {
                LOGGER.info("The label is changed, port: {}, label: {}", port, label);
                port.setLabel(label);

                try {
                    NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save light port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void configChanged(LightPort port, PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port value are changed for port: {}, portConfigKeys: {}", port, portConfigKeys);

                Map<Byte, PortConfigValue<?>> values = new HashMap<>();

                // we must get the configured port
                final List<LightPort> lightPorts = mainModel.getSelectedNode().getLightPorts();
                final LightPort lightPort = PortListUtils.findPortByPortNumber(lightPorts, port.getId());

                for (PortConfigKeys key : portConfigKeys) {

                    if (!lightPort.isPortConfigKeySupported(key)) {
                        LOGGER.info("Unsupported port config key detected: {}", key);
                        continue;
                    }

                    switch (key) {
                        case BIDIB_PCFG_DIMM_DOWN:
                        case BIDIB_PCFG_DIMM_DOWN_8_8:
                            int dimMin = port.getDimMin();

                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8,
                                        new Int16PortConfigValue(ByteUtils.getWORD(dimMin)));
                            }
                            else if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_DOWN,
                                        new BytePortConfigValue(ByteUtils.getLowByte(dimMin)));
                            }
                            break;
                        case BIDIB_PCFG_DIMM_UP:
                        case BIDIB_PCFG_DIMM_UP_8_8:
                            int dimMax = port.getDimMax();

                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8,
                                        new Int16PortConfigValue(ByteUtils.getWORD(dimMax)));
                            }
                            else if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_UP,
                                        new BytePortConfigValue(ByteUtils.getLowByte(dimMax)));
                            }
                            break;
                        case BIDIB_PCFG_LEVEL_PORT_ON:
                            int pwmMax = port.getPwmMax();
                            values
                                .put(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON,
                                    new BytePortConfigValue(ByteUtils.getLowByte(pwmMax)));
                            break;
                        case BIDIB_PCFG_LEVEL_PORT_OFF:
                            int pwmMin = port.getPwmMin();
                            values
                                .put(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF,
                                    new BytePortConfigValue(ByteUtils.getLowByte(pwmMin)));
                            break;
                        case BIDIB_PCFG_RGB:
                            Integer rgbValue = port.getRgbValue();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_RGB) && rgbValue != null) {
                                LOGGER.info("The lightport has RGB support: {}", port);

                                values
                                    .put(BidibLibrary.BIDIB_PCFG_RGB,
                                        new RgbPortConfigValue(ByteUtils.getRGB(rgbValue)));
                            }
                            break;
                        case BIDIB_PCFG_TRANSITION_TIME:
                            Integer transitionTime = port.getTransitionTime();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TRANSITION_TIME)
                                && transitionTime != null) {
                                LOGGER.info("The lightport has TransitionTime support: {}", port);

                                values
                                    .put(BidibLibrary.BIDIB_PCFG_TRANSITION_TIME,
                                        new Int16PortConfigValue(transitionTime));
                            }
                            break;
                        case BIDIB_PCFG_OUTPUT_MAP:
                            Integer dmxMapping = port.getDmxMapping();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)
                                && dmxMapping != null) {
                                LOGGER
                                    .info("The lightport has dmxMapping support: {}, dmxMapping: {}", port, dmxMapping);

                                values
                                    .put(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP, new BytePortConfigValue(
                                        ByteUtils.getLowByte(dmxMapping /*- 1*/ /* dmx mapping offset */)));
                            }
                            break;
                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        LOGGER.info("Set the port params: {}", values);
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the lightport parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }

            }

            @Override
            public void testButtonPressed(final LightPort port, LightPortStatus requestedStatus) {
                LOGGER.info("The test button was pressed for port: {}, requestedStatus: {}", port, requestedStatus);

                final NodeInterface node = mainModel.getSelectedNode();

                if (LightPortStatus.TEST != port.getStatus()) {
                    stopTestToggleTask(node, port);

                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), port);
                }
                else {
                    addTestToggleTask(node, port);
                }
            }

            @Override
            public void changePortType(LcOutputType portType, LightPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                Map<Byte, PortConfigValue<?>> values = new HashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);
            }

        });

        final LightPortListPanel lightPortListPanel =
            new LightPortListPanel(this, tableModel, mainModel, tabVisibilityListener, portConfigChangeEventSubject);

        lightPortListPanel.setPortListener(new LightPortListener() {

            @Override
            public Class<?> getPortClass() {
                return LightPort.class;
            }

            @Override
            public void labelChanged(final LightPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                final NodeLabels nodeLabels = getNodeLabels();
                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, port.getId(), port.getLabel());
                saveLabels();

                lightPortListPanel.repaint();
            }

            @Override
            public void statusChanged(final NodeInterface node, final LightPort port) {
                LOGGER.info("Status of light port has changed, port: {}, status: {}", port, port.getStatus());

                // update port status
                SwingUtils.executeInEDT(() -> tableModel.notifyPortStatusChanged(port));
            }

            @Override
            public void configChanged(final NodeInterface node, final LightPort port) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                SwingUtils.executeInEDT(() -> tableModel.notifyPortConfigChanged(port));
            }

            @Override
            public void testButtonPressed(NodeInterface node, LightPort port, LightPortStatus requestedStatus) {

            }
        });

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeWillChange(final NodeInterface node) {
                LOGGER.info("The selected node will change!");
                try {
                    List<NodeInterface> nodes = new ArrayList<>();
                    for (NodeInterface currentNode : testToggleRegistry.keySet()) {
                        nodes.add(currentNode);
                    }
                    LOGGER.info("Found nodes to stop the test toggle task: {}", nodes);
                    for (NodeInterface currentNode : nodes) {
                        stopTestToggleTask(currentNode, null);
                    }
                    LOGGER.info("Stop the test toggle task passed for nodes: {}", nodes);
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop test toggle tasks failed.", ex);
                }

                clearKeywordToNodeMap();
            }

            @Override
            public void nodeChanged(final NodeInterface node) {
                super.nodeChanged(node);

                LOGGER.info("The selected node has been changed: {}", node);

                compDisp.dispose();
                compDisp.clear();

                compDisp = new CompositeDisposable();

                if (LightPortPanelController.this.selectedNode != null) {
                    LightPortPanelController.this.selectedNode
                        .removeCvDefinitionListener(LightPortPanelController.this);
                }

                if (node != null) {
                    LightPortPanelController.this.selectedNode = node;

                    addLightPortModelListener(node);

                    node.addCvDefinitionListener(LightPortPanelController.this);
                }
            }
        });

        // keep the reference
        this.lightPortListPanel = lightPortListPanel;

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null) {

            if (this.selectedNode != null) {
                this.selectedNode.removeCvDefinitionListener(this);
            }
            this.selectedNode = selectedNode;

            addLightPortModelListener(selectedNode);

            // register listener
            selectedNode.addCvDefinitionListener(this);
        }

        return lightPortListPanel;
    }

    public void addTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Add test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.remove(node);
        ScriptEngine<PortScripting> scriptEngine = null;
        if (nodePortWrapper != null) {
            scriptEngine = nodePortWrapper.removePort(port);
        }

        if (scriptEngine != null) {
            LOGGER.info("Found a node scripting engine in the registry: {}", scriptEngine);
            try {
                scriptEngine.stopScript(Long.valueOf(2000));
            }
            catch (Exception ex) {
                LOGGER.warn("Stop script failed.", ex);
            }
        }

        ApplicationContext context = new DefaultScriptContext();
        context.register(DefaultScriptContext.KEY_SELECTED_NODE, node);
        context.register(DefaultScriptContext.KEY_MAIN_MODEL, mainModel);

        scriptEngine = new ScriptEngine<PortScripting>(this, context);

        List<ScriptCommand<PortScripting>> scriptCommands = new ArrayList<ScriptCommand<PortScripting>>();
        LightPortCommand<PortScripting> spc = new LightPortCommand<>(this.consoleService);
        spc.parse(LightPortCommand.KEY + " " + port.getId() + " ON");
        scriptCommands.add(spc);
        WaitCommand<PortScripting> wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);
        spc = new LightPortCommand<>(this.consoleService);
        spc.parse(LightPortCommand.KEY + " " + port.getId() + " OFF");
        scriptCommands.add(spc);
        wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);

        LOGGER.info("Prepared list of commands: {}", scriptCommands);

        scriptEngine.setScriptCommands(scriptCommands);
        // repeating
        scriptEngine.setScriptRepeating(true);

        if (nodePortWrapper == null) {
            LOGGER.info("Create new NodePortWrapper for node: {}", node);
            nodePortWrapper = new NodePortWrapper(node);
        }

        LOGGER.info("Put script engine in registry for node: {}", node);
        nodePortWrapper.addPort(port, scriptEngine);

        testToggleRegistry.put(node, nodePortWrapper);

        scriptEngine.startScript();
    }

    public void stopTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Stop test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.get(node);

        if (nodePortWrapper != null) {
            Set<Port<?>> toRemove = new HashSet<>();
            if (port != null) {
                toRemove.add(port);
            }
            else {
                toRemove.addAll(nodePortWrapper.getKeySet());
            }

            for (Port<?> removePort : toRemove) {
                ScriptEngine<PortScripting> engine = nodePortWrapper.removePort(removePort);

                if (engine != null) {
                    LOGGER.info("Found a node scripting engine in the registry: {}", engine);
                    try {
                        engine.stopScript(Long.valueOf(2000));
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Stop script failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No scripting engine found for node: {}", node);
                }
            }

            if (nodePortWrapper.isEmpty()) {
                LOGGER.info("No more ports registered for node: {}", node);
                testToggleRegistry.remove(node);
            }
        }
    }

    @Override
    public void sendPortStatusAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, BidibStatus portStatus) {

        LOGGER.info("Switch the light port: {}, portStatus: {}", port, portStatus);
        try {
            final SwitchingNodeInterface selectedNode =
                node != null ? node : mainModel.getSelectedNode().getSwitchingNode();

            LightPortStatus lightPortStatus = (LightPortStatus) portStatus;

            LightPort lightPort = new LightPort();
            lightPort.setId(port);
            lightPort.setStatus(lightPortStatus);
            switchingNodeService.setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, lightPort);
        }
        catch (Exception ex) {
            LOGGER.warn("Switch light port failed.", ex);
        }
    }

    @Override
    public void sendPortValueAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, int portValue) {
        // not used

    }

    private void clearKeywordToNodeMap() {
        LOGGER.info("Clear the keywordToNode map.");
        mapKeywordToNode = null;
    }

    private void loadMacros(final NodeInterface node) {
        final List<Macro> macroList = new ArrayList<Macro>();
        for (Macro macro : node.getMacros()) {

            if (MacroSaveState.NOT_LOADED_FROM_NODE.equals(macro.getMacroSaveState())) {

                Macro macroWithContent =
                    switchingNodeService
                        .getMacroContent(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), macro);
                LOGGER.info("Load macro content: {}", macroWithContent);

                // reset the changed flag on the macro
                macroWithContent.setMacroSaveState(MacroSaveState.PERMANENTLY_STORED_ON_NODE);
                macroWithContent.setFlatPortModel(node.getNode().isPortFlatModelAvailable());

                macroList.add(macroWithContent);
            }
            else {
                macroList.add(macro);
            }
        }
        LOGGER.info("Set the loaded macros for the node: {}", macroList);
        node.setMacros(macroList);
    }

    @Override
    public void insertPorts(int selectedRow, final LightPortTableModel lightPortTableModel) {

        BusyFrame busyFrame = null;
        StopWatch sw = new StopWatch();

        // show a confirm dialog
        final InsertPortsConfirmDialog insertPortsConfirmDialog =
            new InsertPortsConfirmDialog(JOptionPane.getFrameForComponent(null), true);

        int result = insertPortsConfirmDialog.getResult();

        if (result == JOptionPane.CANCEL_OPTION) {
            LOGGER.info("User canceled the insert ports operation.");
            return;
        }

        int portsCount = insertPortsConfirmDialog.getPortsCount();
        LOGGER.info("Insert number of ports: {}", portsCount);

        final NodeInterface node = mainModel.getSelectedNode();

        try {
            Frame frame = JOptionPane.getFrameForComponent(lightPortListPanel);
            if (frame instanceof BusyFrame) {
                busyFrame = (BusyFrame) frame;
                busyFrame.setBusy(true);
            }

            sw.start();

            // make sure all macros were loaded from the node
            if (node.hasUnloadedMacros()) {
                LOGGER.info("Unloaded macros detected. Load the macros from the node.");
                loadMacros(node);
            }

            List<LightPort> allLightPorts = node.getLightPorts();

            LightPort firstPort =
                (LightPort) lightPortTableModel.getValueAt(selectedRow, LightPortTableModel.COLUMN_PORT_INSTANCE);
            LOGGER
                .info("Found lightPort to start insert the new ports: {}, port number: {}", firstPort,
                    firstPort.getId());

            int startPortRangeValue = firstPort.getId() + portsCount - 1;
            Integer portRangeUpperBound = null;
            if (channelBStartNumber != null) {

                portRangeUpperBound = testValidRange(firstPort, portsCount, channelBStartNumber);

                LOGGER.info("Validated range, portRangeUpperBound: {}", portRangeUpperBound);
            }

            final NodeLabels nodeLabels = getNodeLabels();

            // inverted sort order
            final Set<LightPort> ports =
                findAllPortsInRange(allLightPorts, startPortRangeValue, portRangeUpperBound,
                    (LightPort o1, LightPort o2) -> Integer.compare(o2.getId(), o1.getId()),
                    (lp, portNum) -> lp.getId() > portNum);

            for (LightPort currentPort : ports) {

                LOGGER.info("Change the label of the port: {}, id: {}", currentPort, currentPort.getId());
                // and move the port label and the port config

                int portNum = currentPort.getId();

                // get the predecessor port
                LightPort predecessorPort = PortListUtils.findPortByPortNumber(allLightPorts, portNum - portsCount);

                if (predecessorPort != null) {
                    LOGGER
                        .info("Current predecessorPort: {}, port number: {}", predecessorPort, predecessorPort.getId());

                    // move the label
                    String label = predecessorPort.getLabel();
                    currentPort.setLabel(label);
                    predecessorPort.setLabel(null);

                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, currentPort.getId(),
                            currentPort.getLabel());

                    // move the port config
                    Map<Byte, PortConfigValue<?>> configX = predecessorPort.getPortConfigX();
                    currentPort.setPortConfigX(configX);

                    // initiate transfer to node
                    List<PortConfigKeys> portConfigKeys = new ArrayList<>();
                    for (byte key : currentPort.getKnownPortConfigKeys()) {
                        portConfigKeys.add(PortConfigKeys.valueOf(key));
                    }
                    lightPortModelListener.configChanged(currentPort, portConfigKeys.toArray(new PortConfigKeys[0]));
                }
                else {
                    LOGGER.info("No predecessorPort available with port number: {}", portNum - portsCount);
                }
            }

            // replace the labels of the inserted ports
            for (int portNum = 0; portNum < portsCount; portNum++) {
                LightPort replacedPort = PortListUtils.findPortByPortNumber(allLightPorts, firstPort.getId() + portNum);
                if (replacedPort != null) {
                    LOGGER
                        .info("Remove label and configuration of port: {}, port number: {}", replacedPort,
                            replacedPort.getId());
                    replacedPort.setLabel(null);

                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, replacedPort.getId(),
                            replacedPort.getLabel());

                    // we have to initialize the values of the inserted node
                    initialzePortConfigValues(replacedPort);

                    // initiate transfer to node
                    List<PortConfigKeys> portConfigKeys = new ArrayList<>();
                    for (byte key : replacedPort.getKnownPortConfigKeys()) {
                        portConfigKeys.add(PortConfigKeys.valueOf(key));
                    }
                    lightPortModelListener.configChanged(replacedPort, portConfigKeys.toArray(new PortConfigKeys[0]));
                }
            }

            saveLabels();

            // move the ports in all macros and store the changed macro on the node
            macroPanelController
                .movePortsInAllMacros(mainModel, node, firstPort, allLightPorts, portsCount,
                    portNum -> portNum + portsCount);

            lightPortTableModel.fireTableDataChanged();

            // force refresh of macro list
            forceRefreshMacroList(node);
        }
        catch (InvalidConfigurationException ex) {
            LOGGER.warn("Save port labels failed.", ex);

            String labelPath = ex.getReason();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"), JOptionPane.ERROR_MESSAGE);
        }
        catch (Exception ex) {
            LOGGER.warn("Save port labels failed.", ex);

            String message = ex.getMessage();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { message }),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"), JOptionPane.ERROR_MESSAGE);
        }
        finally {
            if (busyFrame != null) {
                LOGGER.info("Unset busy frame.");
                busyFrame.setBusy(false);
            }
            sw.stop();
        }

        LOGGER.info("Insert ports has finished! Total loading duration: {}", sw);
        this.statusBar
            .setStatusText(String
                .format(Resources.getString(LightPortPanelController.class, "insert-ports-finished"), portsCount, sw),
                StatusBar.DISPLAY_NORMAL);
    }

    @Override
    public void removePort(int selectedRow, final LightPortTableModel lightPortTableModel) {

        BusyFrame busyFrame = null;
        StopWatch sw = new StopWatch();

        // show a confirm dialog
        final RemovePortConfirmDialog removePortConfirmDialog =
            new RemovePortConfirmDialog(JOptionPane.getFrameForComponent(null), true);

        int result = removePortConfirmDialog.getResult();

        if (result == JOptionPane.CANCEL_OPTION) {
            LOGGER.info("User canceled the remove port operation.");
            return;
        }

        final NodeInterface node = mainModel.getSelectedNode();
        String originalPortLabel = null;

        try {
            Frame frame = JOptionPane.getFrameForComponent(lightPortListPanel);
            if (frame instanceof BusyFrame) {
                busyFrame = (BusyFrame) frame;
                busyFrame.setBusy(true);
            }

            sw.start();

            // make sure all macros were loaded from the node
            if (node.hasUnloadedMacros()) {
                LOGGER.info("Unloaded macros detected. Load the macros from the node.");
                loadMacros(node);
            }

            final LightPort firstPort =
                (LightPort) lightPortTableModel.getValueAt(selectedRow, LightPortTableModel.COLUMN_PORT_INSTANCE);
            LOGGER
                .info("Found lightPort to start insert the new ports: {}, port number: {}", firstPort,
                    firstPort.getId());

            originalPortLabel = firstPort.getLabel();
            if (StringUtils.isBlank(originalPortLabel)) {
                originalPortLabel = firstPort.toString();
            }

            final List<LightPort> allLightPorts = node.getLightPorts();

            final int portsCount = 1;
            int startPortRangeValue = firstPort.getId();
            Integer portRangeUpperBound = null;
            if (channelBStartNumber != null) {

                portRangeUpperBound = testValidRange(firstPort, portsCount, channelBStartNumber);

                LOGGER.info("Validated range, portRangeUpperBound: {}", portRangeUpperBound);
            }

            final NodeLabels nodeLabels = getNodeLabels();

            // normal sort order
            final List<LightPort> ports =
                new ArrayList<>(findAllPortsInRange(allLightPorts, startPortRangeValue, portRangeUpperBound,
                    (LightPort o1, LightPort o2) -> Integer.compare(o1.getId(), o2.getId()),
                    (lp, portNum) -> lp.getId() >= portNum));

            for (LightPort currentPort : ports) {

                LOGGER.info("Change the label of the port: {}, id: {}", currentPort, currentPort.getId());
                // and move the port label and the port config

                int portNum = currentPort.getId();

                // get the predecessor port
                LightPort successorPort = PortListUtils.findPortByPortNumber(ports, portNum + portsCount);

                if (successorPort != null) {
                    LOGGER.info("Current successorPort: {}, port number: {}", successorPort, successorPort.getId());

                    // move the label
                    String label = successorPort.getLabel();
                    currentPort.setLabel(label);
                    successorPort.setLabel(null);

                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, currentPort.getId(),
                            currentPort.getLabel());

                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.lightPort, successorPort.getId(),
                            successorPort.getLabel());

                    // move the port config
                    Map<Byte, PortConfigValue<?>> configX = successorPort.getPortConfigX();
                    currentPort.setPortConfigX(configX);

                    // initiate transfer to node
                    List<PortConfigKeys> portConfigKeys = new ArrayList<>();
                    for (byte key : currentPort.getKnownPortConfigKeys()) {
                        portConfigKeys.add(PortConfigKeys.valueOf(key));
                    }
                    lightPortModelListener.configChanged(currentPort, portConfigKeys.toArray(new PortConfigKeys[0]));

                    // check if the next successor port exists
                    if (PortListUtils.findPortByPortNumber(ports, portNum + portsCount + 1) == null) {
                        LOGGER
                            .info(
                                "The next successor port does not exist. Initialize the port configuration values of the current successor port: {}",
                                successorPort);
                        // we have to initialize the values of the successor port node
                        initialzePortConfigValues(successorPort);

                        // initiate transfer to node
                        List<PortConfigKeys> successorPortConfigKeys = new ArrayList<>();
                        for (byte key : successorPort.getKnownPortConfigKeys()) {
                            successorPortConfigKeys.add(PortConfigKeys.valueOf(key));
                        }
                        lightPortModelListener
                            .configChanged(successorPort, successorPortConfigKeys.toArray(new PortConfigKeys[0]));
                    }
                }
                else {
                    LOGGER.info("No successorPort available with port number: {}", portNum + portsCount);
                }
            }

            saveLabels();

            // move the ports in all macros and store the changed macro on the node
            macroPanelController
                .movePortsInAllMacros(mainModel, node, firstPort, allLightPorts, portsCount,
                    portNum -> portNum - portsCount);

            lightPortTableModel.fireTableDataChanged();

            // force refresh of macro list
            forceRefreshMacroList(node);
        }
        catch (InvalidConfigurationException ex) {
            LOGGER.warn("Save port labels failed.", ex);

            String labelPath = ex.getReason();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"), JOptionPane.ERROR_MESSAGE);
        }
        catch (Exception ex) {
            LOGGER.warn("Save port labels failed.", ex);

            String message = ex.getMessage();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { message }),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"), JOptionPane.ERROR_MESSAGE);
        }
        finally {
            if (busyFrame != null) {
                LOGGER.info("Unset busy frame.");
                busyFrame.setBusy(false);
            }
            sw.stop();
        }

        LOGGER.info("Remove port has finished! Total loading duration: {}", sw);
        this.statusBar
            .setStatusText(String
                .format(Resources.getString(LightPortPanelController.class, "remove-port-finished"), originalPortLabel,
                    sw),
                StatusBar.DISPLAY_NORMAL);
    }

    private void forceRefreshMacroList(final NodeInterface node) {
        // force a reload of the changed labels
        final MacroChangedEvent macroChangedEvent = new MacroChangedEvent(node.getUniqueId());
        LOGGER.info("Publish the macroChangedEvent: {}", macroChangedEvent);

        this.applicationEventPublisher.publishEvent(macroChangedEvent);
    }

    private void initialzePortConfigValues(final LightPort port) {
        final Map<Byte, PortConfigValue<?>> sourceConfigX = port.getPortConfigX();
        final Map<Byte, PortConfigValue<?>> updatedConfigX = new HashMap<>();
        for (Entry<Byte, PortConfigValue<?>> pcv : sourceConfigX.entrySet()) {

            // special init values
            if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(0)));
            }
            else if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(255)));
            }
            else if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(70)));
            }
            else if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_DIMM_DOWN))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(70)));
            }
            else if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(70)));
            }
            else if (Objects.equals(pcv.getKey(), Byte.valueOf(BidibLibrary.BIDIB_PCFG_DIMM_UP))) {
                updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(70)));
            }
            else {
                // default init values
                if (pcv.getValue() instanceof BytePortConfigValue) {
                    updatedConfigX.put(pcv.getKey(), new BytePortConfigValue(ByteUtils.getLowByte(0)));
                }
                else if (pcv.getValue() instanceof Int16PortConfigValue) {
                    updatedConfigX.put(pcv.getKey(), new Int16PortConfigValue(0));
                }
                else if (pcv.getValue() instanceof Int32PortConfigValue) {
                    updatedConfigX.put(pcv.getKey(), new Int32PortConfigValue(0L));
                }
                else if (pcv.getValue() instanceof RgbPortConfigValue) {
                    updatedConfigX.put(pcv.getKey(), new RgbPortConfigValue(0));
                }
            }
        }
        port.setPortConfigX(updatedConfigX);
    }

    private void readCurrentCvNodeValuesFromCVDefinition(final NodeInterface node) {
        LOGGER.info("Read the values from CV for node: {}", node);

        if (node == null
            || (!ProductUtils.isNeoControl(node.getUniqueId()) && !ProductUtils.isNeoEWS(node.getUniqueId()))) {
            LOGGER.warn("No node available or not a NeoControl: {}", node);

            if (mapKeywordToNode != null) {
                mapKeywordToNode.clear();
            }
            return;
        }

        LOGGER.info("Get the cvDefinitionTreeTableModel from the node: {}", node);

        final CvDefinitionPanelController cvDefinitionPanelController =
            DefaultApplicationContext
                .getInstance()
                .get(DefaultApplicationContext.KEY_CVDEFINITIONPANEL_CONTROLLER, CvDefinitionPanelController.class);

        CvDefinitionTreeTableModel cvDefinitionTreeTableModel =
            cvDefinitionPanelController.getCvDefinitionTreeTableModel(node);

        if (cvDefinitionTreeTableModel != null) {
            // search the keywords
            mapKeywordToNode = new HashMap<>();

            DefaultExpandableRow rootNode = (DefaultExpandableRow) cvDefinitionTreeTableModel.getRoot();
            if (rootNode != null) {
                CvNodeUtils.harvestKeywordNodes(rootNode, mapKeywordToNode);
            }

            LOGGER.info("Found keywords in nodes: {}", mapKeywordToNode.keySet());
        }

        // read the values from the node
        if (MapUtils.isNotEmpty(mapKeywordToNode)) {
            List<ConfigurationVariable> configVariables = new ArrayList<>();
            for (CvNode cvNode : mapKeywordToNode.values()) {
                // LOGGER.info("Process cvNode: {}", cvNode);

                prepareConfigVariables(cvNode, configVariables, getCvNumberToNodeMap(node));
            }

            // keep the list of config variables
            this.requiredConfigVariables.clear();
            if (CollectionUtils.isNotEmpty(configVariables)) {
                this.requiredConfigVariables.addAll(configVariables);
            }

            // // add the current position with the special CV
            // prepareGetCurrentPosition(configVariables);

            fireLoadConfigVariables(configVariables);

            // get the B channel start number
            processCvDefinitionValuesChanged(true);
        }
        else {
            LOGGER.info("No values available in mapKeywordToNode!");
        }

    }

    public void addCvDefinitionRequestListener(CvDefinitionRequestListener l) {
        cvDefinitionRequestListeners.add(l);
    }

    private void fireLoadConfigVariables(List<ConfigurationVariable> configVariables) {
        // TODO decouple the AWT-thread from this work?
        LOGGER.info("Load the config variables.");
        for (CvDefinitionRequestListener l : cvDefinitionRequestListeners) {
            l.loadCvValues(configVariables);
        }

        LOGGER.info("Load the config variables has finished.");
    }

    private void prepareConfigVariables(
        final CvNode cvNode, final List<ConfigurationVariable> configVariables,
        final Map<String, CvNode> cvNumberToNodeMap) {

        try {
            switch (cvNode.getCV().getType()) {
                case LONG:
                    // LONG nodes are processed
                    LongCvNode masterNode = ((LongCvNode) cvNode).getMasterNode();
                    configVariables.add(masterNode.getConfigVar());
                    for (CvNode slaveNode : masterNode.getSlaveNodes()) {
                        configVariables.add(slaveNode.getConfigVar());
                    }
                    break;
                case INT:
                    // INT nodes are processed
                    configVariables.add(cvNode.getConfigVar());
                    int highCvNum = Integer.parseInt(cvNode.getCV().getHigh());
                    int cvNumber = Integer.parseInt(cvNode.getCV().getNumber());

                    if (highCvNum == cvNumber) {
                        // search the low CV
                        CvNode lowCvNode = cvNumberToNodeMap.get(cvNode.getCV().getLow());
                        configVariables.add(lowCvNode.getConfigVar());
                    }
                    else {
                        // search the high CV
                        CvNode highCvNode = cvNumberToNodeMap.get(cvNode.getCV().getHigh());
                        if (highCvNode == null) {
                            LOGGER.warn("The highCvNode is not available: {}", cvNode.getCV());
                        }
                        configVariables.add(highCvNode.getConfigVar());
                    }
                    break;
                default:
                    configVariables.add(cvNode.getConfigVar());
                    break;
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Prepare config variables to read from node failed.", ex);
        }

    }

    private CvNode getCvNodeByKeyword(String keyword) {
        CvNode cvNode = null;
        if (mapKeywordToNode != null) {
            cvNode = mapKeywordToNode.get(keyword);
        }
        return cvNode;
    }

    private Integer getConfigVarIntValue(String keyword, final NodeInterface selectedNode) {
        CvNode cvNode = getCvNodeByKeyword(keyword);
        Integer value = CvValueUtils.getConfigVarIntValue(cvNode, getCvNumberToNodeMap(selectedNode));
        return value;
    }

    private Map<String, CvNode> getCvNumberToNodeMap(final NodeInterface node) {
        // get the prepared cvNumberToNodeMap from the CvDefinitionPanelController
        final CvDefinitionPanelController cvDefinitionPanelController =
            DefaultApplicationContext
                .getInstance()
                .get(DefaultApplicationContext.KEY_CVDEFINITIONPANEL_CONTROLLER, CvDefinitionPanelController.class);

        Map<String, CvNode> cvNumberToNodeMap = cvDefinitionPanelController.getCvNumberToNodeMap(node);
        return cvNumberToNodeMap;
    }

    /**
     * Find all ports greater than the provided port number in reverse order.
     * 
     * @param ports
     *            the ports
     * @param portNum
     *            the port number
     * @param portNumUpperBound
     *            the port number that is used as upper limit (exclusive)
     * @return the set of ports in reverse order
     */
    public static <T extends Port<?>> Set<T> findAllPortsInRange(
        List<T> ports, final int portNum, final Integer portNumUpperBound, final Comparator<T> orderComparator,
        final BiPredicate<T, Integer> filterPredicate) {
        LOGGER.info("Find all ports with port number greater than: {}, lower than: {}", portNum, portNumUpperBound);

        final Set<T> filteredSet = new TreeSet<>(orderComparator);
        filteredSet.addAll(ports);

        CollectionUtils.filter(filteredSet, (port) -> {
            if (filterPredicate.test(port, portNum)) {
                // check if upper bound is provided
                if (portNumUpperBound != null && portNumUpperBound > port.getId()) {
                    return true;
                }
            }
            return false;
        });

        return filteredSet;
    }

    protected Integer testValidRange(final LightPort firstPort, int portsCount, Integer channelBStartNumber) {

        LOGGER
            .info("Test the valid range: {}, portsCount: {}, channelBStartNumber: {}", firstPort, portsCount,
                channelBStartNumber);

        if (channelBStartNumber == null) {
            throw new IllegalArgumentException("The channelBStartNumber must be provided.");
        }

        Integer upperRangeValue = channelBStartNumber * 3 /* 3 ports per WS28xx */;
        // situation 1: start is below channel b start num and portsCount will not cross the channel border

        int startPortRangeValue = firstPort.getId() + portsCount - 1;
        if (firstPort.getId() < upperRangeValue) {
            if (startPortRangeValue < upperRangeValue) {
                LOGGER
                    .info(
                        "We want to move ports in channel A and the number of ports to insert will match into channel A range.");
                // we move ports in channel A -> the upper range is the channel A limit
            }
            else {
                LOGGER
                    .warn(
                        "The number of inserted port crosses the channel border. Ports configurations will be dropped out.");
            }
        }
        else {
            LOGGER.info("The inserted ports will start in channel B. No upper range limit required.");
            upperRangeValue = null;
        }

        return upperRangeValue;
    }

    @Override
    public void cvDefinitionValuesChanged(final boolean read, final List<String> changedNames) {
        LOGGER.info("The CV definition values have changed, read: {}", read);
        SwingUtils.executeInEDT(() -> processCvDefinitionValuesChanged(read));
    }

    protected void processCvDefinitionValuesChanged(final boolean read) {

        // reset the B channel start number
        channelBStartNumber = null;

        final NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode == null || MapUtils.isEmpty(selectedNode.getConfigVariables())
            || (!ProductUtils.isNeoControl(selectedNode.getUniqueId())
                && !ProductUtils.isNeoEWS(selectedNode.getUniqueId()))) {
            LOGGER.info("No special processing for non NeoControl or NeoEWS.");
        }
        else {

            // TODO check if the keyword has changed
            // look for keyword
            if (ProductUtils.isNeoControl(selectedNode.getUniqueId())) {
                channelBStartNumber = getConfigVarIntValue(KEYWORD_CHANNEL_B_START_NUMBER, selectedNode);
            }
            else {
                Integer channelALastElement = getConfigVarIntValue(KEYWORD_CHANNEL_A_LAST_ELEMENT, selectedNode);
                if (channelALastElement != null) {
                    LOGGER
                        .info("Get the channel B start number from channel A last element value: {}",
                            channelALastElement);
                    channelBStartNumber = channelALastElement + 1;
                }
            }
            LOGGER.info("Fetched channel B start number: {}", channelBStartNumber);
        }

        triggerLoadCvValues(selectedNode);
    }

    @Override
    public void cvDefinitionChanged() {
        LOGGER.info("The CV definition has changed.");

        // channelBStartNumber = null;
        //
        // final NodeInterface node = mainModel.getSelectedNode();
        // if (node == null) {
        // return;
        // }
        //
        // triggerLoadCvValues(node);
    }

    private void triggerLoadCvValues(final NodeInterface node) {

        if (mapKeywordToNode == null
            && (ProductUtils.isNeoControl(node.getUniqueId()) || ProductUtils.isNeoEWS(node.getUniqueId()))) {
            LOGGER.info("A NeoControl or NeoEWS is detected. Get the start number of port for channel B.");
            readCurrentCvNodeValuesFromCVDefinition(node);
        }
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.lightPortListPanel != null) {
            SwingUtils.executeInEDT(() -> this.lightPortListPanel.refreshView());
        }
    }

    private void addLightPortModelListener(final NodeInterface selectedNode) {

        LOGGER.info("Add light port model listener for node: {}", selectedNode);

        final Disposable disp = this.portConfigChangeEventSubject.subscribe(evt -> {
            LOGGER.info("Received event: {}", evt);

            final PortTypeAware port = evt.getPort();

            // update port config
            try {
                final LightPort lightPort = new LightPort();
                lightPort.setId(port.getPortNumber());

                LOGGER.info("Prepared light port: {}", lightPort);

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(), lightPort,
                        null, evt.getPortConfig());
            }
            catch (Exception ex) {
                LOGGER.warn("Set the lightport config failed.", ex);
                selectedNode.setNodeHasError(true);
                selectedNode.setReasonData("Set the lightport config failed.");
            }
        });

        compDisp.add(disp);
    }
}
