package org.bidib.wizard.mvc.main.controller;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.MapUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.SwitchingNodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.ServoPortValueListener;
import org.bidib.wizard.api.script.ScriptCommand;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.client.common.model.ServoPortTableModel;
import org.bidib.wizard.client.common.model.listener.ServoPortModelListener;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.script.DefaultScriptContext;
import org.bidib.wizard.common.script.engine.ScriptEngine;
import org.bidib.wizard.common.script.switching.PortScripting;
import org.bidib.wizard.common.script.switching.ServoPortCommand;
import org.bidib.wizard.common.script.switching.ServoPortStatusCommand;
import org.bidib.wizard.common.script.switching.WaitCommand;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.PortTypeAware;
import org.bidib.wizard.model.ports.ServoPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.ServoPortStatus;
import org.bidib.wizard.mvc.main.controller.wrapper.NodePortWrapper;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.ServoPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class ServoPortPanelController implements PortScripting {

    private static final Logger LOGGER = LoggerFactory.getLogger(ServoPortPanelController.class);

    private final MainModel mainModel;

    private final Map<NodeInterface, NodePortWrapper> testToggleRegistry = new LinkedHashMap<>();

    private ServoPortListPanel servoPortListPanel;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private ConsoleService consoleService;

    @Autowired
    private SettingsService settingsService;

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject = PublishSubject.create();

    private CompositeDisposable compDisp;


    public ServoPortPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        this.compDisp = new CompositeDisposable();
    }

    public ServoPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final ServoPortTableModel tableModel = new ServoPortTableModel();

        tableModel.setPortListener(new ServoPortModelListener() {

            @Override
            public void labelChanged(final ServoPort port, String label) {
                port.setLabel(label);

                try {
                    final NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.servoPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save servo port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }

            }

            @Override
            public void configChanged(final ServoPort port, final PortConfigKeys... portConfigKeys) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                try {
                    final NodeInterface node = mainModel.getSelectedNode();
                    switchingNodeService
                        .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), port);
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the servoport config failed.", ex);

                    mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                }
            }

            @Override
            public void testButtonPressed(final ServoPort port, Integer requestedValue) {
                LOGGER.info("The test button was pressed for port: {}, requestedValue: {}", port, requestedValue);

                try {
                    // create a new instance of the port
                    ServoPort servoPort = new ServoPort();
                    servoPort.setId(port.getId());
                    servoPort.setValue(requestedValue);

                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN,
                            mainModel.getSelectedNode().getSwitchingNode(), servoPort);
                    LOGGER.info("Test pressed for port: {}, port.value: {}", port, port.getValue());
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the servoport status failed.", ex);

                    mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                }
            }

            @Override
            public void toggleTest(final ServoPort servoPort) {
                LOGGER.info("Toggle the test task for port: {}, status: {}", servoPort, servoPort.getStatus());

                final NodeInterface node = mainModel.getSelectedNode();

                if (ServoPortStatus.TEST != servoPort.getStatus()) {
                    stopTestToggleTask(node, servoPort);
                }
                else {
                    addTestToggleTask(node, servoPort);
                }
            }

        });

        final ServoPortListPanel servoPortListPanel =
            new ServoPortListPanel(this, tableModel, mainModel, settingsService, tabVisibilityListener,
                portConfigChangeEventSubject);

        servoPortListPanel.setPortListener(new ServoPortValueListener() {
            @Override
            public Class<?> getPortClass() {
                return ServoPort.class;
            }

            @Override
            public void labelChanged(final ServoPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                final NodeLabels nodeLabels = getNodeLabels();
                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.servoPort, port.getId(), port.getLabel());
                saveLabels();

                servoPortListPanel.repaint();
            }

            @Override
            public void valueChanged(final NodeInterface node, final ServoPort port) {
                LOGGER.info("Value has changed for servo port: {}", port);

                SwingUtilities.invokeLater(() -> tableModel.notifyPortStatusChanged(port));
            }

            @Override
            public void configChanged(final NodeInterface node, final ServoPort port) {
                LOGGER.info("The configuration of the port has changed, node: {}, port: {}", node, port);

                // update port config
                SwingUtilities.invokeLater(() -> tableModel.notifyPortConfigChanged(port));
            }

            @Override
            public void valuesChanged(final ServoPort port, final PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port value are changed for port: {}, portConfigKeys: {}", port, portConfigKeys);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                // we must get the configured port
                final List<ServoPort> servoPorts = mainModel.getSelectedNode().getServoPorts();
                final ServoPort servoPort = PortListUtils.findPortByPortNumber(servoPorts, port.getId());

                for (PortConfigKeys key : portConfigKeys) {

                    if (!servoPort.isPortConfigKeySupported(key)) {
                        LOGGER.info("Unsupported port config key detected: {}", key);
                        continue;
                    }

                    switch (key) {
                        case BIDIB_PCFG_SERVO_SPEED:
                            int servoSpeed = port.getSpeed();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_SERVO_SPEED)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_SERVO_SPEED,
                                        new BytePortConfigValue(ByteUtils.getLowByte(servoSpeed)));
                            }
                            break;
                        case BIDIB_PCFG_SERVO_ADJ_L:
                            int servoAdjL = port.getTrimDown();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_SERVO_ADJ_L)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_SERVO_ADJ_L,
                                        new BytePortConfigValue(ByteUtils.getLowByte(servoAdjL)));
                            }
                            break;
                        case BIDIB_PCFG_SERVO_ADJ_H:
                            int servoAdjH = port.getTrimDown();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_SERVO_ADJ_H)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_SERVO_ADJ_H,
                                        new BytePortConfigValue(ByteUtils.getLowByte(servoAdjH)));
                            }
                            break;
                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        LOGGER.info("Set the port params: {}", values);
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the servoport parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }
            }

            @Override
            public void testButtonPressed(final NodeInterface node, final ServoPort port, int requestedValue) {
                LOGGER.info("The test button was pressed for port: {}, requestedValue: {}", port, requestedValue);

                stopTestToggleTask(node, port);

                // create a new instance of the port
                ServoPort servoPort = new ServoPort();
                servoPort.setId(port.getId());
                servoPort.setValue(requestedValue);

                switchingNodeService
                    .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), servoPort);
            }
        });

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeWillChange(final NodeInterface node) {

                try {
                    List<NodeInterface> nodes = new LinkedList<>();
                    for (Entry<NodeInterface, NodePortWrapper> entry : testToggleRegistry.entrySet()) {
                        nodes.add(entry.getKey());
                    }

                    for (NodeInterface currentNode : nodes) {
                        stopTestToggleTask(currentNode, null);
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop test toggle tasks failed.", ex);
                }
            }

            @Override
            public void nodeChanged(NodeInterface node) {
                super.nodeChanged(node);

                LOGGER.info("The selected node has been changed: {}", node);

                compDisp.dispose();
                compDisp.clear();

                compDisp = new CompositeDisposable();

                if (node != null) {
                    addServoPortModelListener(node);
                }
            }
        });

        // keep the reference
        this.servoPortListPanel = servoPortListPanel;

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null) {
            addServoPortModelListener(selectedNode);
        }

        return this.servoPortListPanel;
    }

    @Override
    public void sendPortStatusAction(final SwitchingNodeInterface node, int portNum, BidibStatus portStatus) {
        LOGGER.info("Set the port status, portNum: {}, portStatus: {}", portNum, portStatus);
        try {
            final SwitchingNodeInterface selectedNode =
                node != null ? node : mainModel.getSelectedNode().getSwitchingNode();
            ServoPortStatus servoPortStatus = (ServoPortStatus) portStatus;

            ServoPort servoPort = PortListUtils.findPortByPortNumber(selectedNode.getNode().getServoPorts(), portNum);
            servoPort.setStatus(servoPortStatus);
        }
        catch (Exception ex) {
            LOGGER.warn("Set servo port status failed.", ex);
        }
    }

    @Override
    public void sendPortValueAction(final SwitchingNodeInterface node, int port, int portValue) {
        LOGGER.info("Set the position on port: {}, portValue: {}", port, portValue);

        final SwitchingNodeInterface selectedNode =
            node != null ? node : mainModel.getSelectedNode().getSwitchingNode();
        ServoPort servoPort = PortListUtils.findPortByPortNumber(selectedNode.getNode().getServoPorts(), port);

        if (portValue >= 0 /* servoPort.getTrimDown() */ && portValue <= 255 /* servoPort.getTrimUp() */) {
            try {

                // int absoluteValue = ServoPort.getAbsoluteValue(portValue);

                // create a new instance of the port
                ServoPort tempServoPort = new ServoPort();
                tempServoPort.setId(servoPort.getId());
                tempServoPort.setValue(portValue);
                switchingNodeService.setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, tempServoPort);
            }
            catch (Exception ex) {
                LOGGER.warn("Set the position on port failed.", ex);
            }
        }
        else {
            LOGGER.warn("The portValue is not in allowed range: {}", portValue);
        }
    }

    public void addTestToggleTask(final NodeInterface node, final ServoPort port) {
        LOGGER.info("Add test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.remove(node);
        ScriptEngine<PortScripting> scriptEngine = null;
        if (nodePortWrapper != null) {
            scriptEngine = nodePortWrapper.removePort(port);
        }

        if (scriptEngine != null) {
            LOGGER.info("Found a node scripting engine in the registry: {}", scriptEngine);
            try {
                scriptEngine.stopScript(Long.valueOf(2000));
            }
            catch (Exception ex) {
                LOGGER.warn("Stop script failed.", ex);
            }
        }

        final ApplicationContext context = new DefaultScriptContext();
        context.register(DefaultScriptContext.KEY_SELECTED_NODE, node);
        context.register(DefaultScriptContext.KEY_MAIN_MODEL, mainModel);

        scriptEngine = new ScriptEngine<PortScripting>(this, context);

        // final ServoPort servoPort = port;
        int startPosition = 0 /* servoPort.getTrimDown() */;
        int endPosition = 255 /* servoPort.getTrimUp() */;

        // add the script
        List<ScriptCommand<PortScripting>> scriptCommands = new LinkedList<ScriptCommand<PortScripting>>();
        ServoPortCommand<PortScripting> spc = new ServoPortCommand<>(this.consoleService);
        spc.parse(ServoPortCommand.KEY + " " + port.getId() + " " + startPosition);
        scriptCommands.add(spc);
        WaitCommand<PortScripting> wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);

        for (int pos = startPosition + 1; pos < endPosition; pos++) {
            spc = new ServoPortCommand<>(this.consoleService);
            spc.parse(ServoPortCommand.KEY + " " + port.getId() + " " + pos);
            scriptCommands.add(spc);
            wc = new WaitCommand<>(this.consoleService);
            wc.parse(WaitCommand.KEY + " 1000");
            scriptCommands.add(wc);
        }

        spc.parse(ServoPortCommand.KEY + " " + port.getId() + " " + endPosition);
        scriptCommands.add(spc);
        wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 500");
        scriptCommands.add(wc);

        ServoPortStatusCommand<PortScripting> spsc = new ServoPortStatusCommand<>(this.consoleService);
        spsc.parse(ServoPortStatusCommand.KEY + " " + port.getId() + " start");
        scriptCommands.add(spc);

        LOGGER.info("Prepared list of commands: {}", scriptCommands);

        scriptEngine.setScriptCommands(scriptCommands);

        if (nodePortWrapper == null) {
            LOGGER.info("Create new NodePortWrapper for node: {}", node);
            nodePortWrapper = new NodePortWrapper(node);
        }

        LOGGER.info("Put script engine in registry for node: {}", node);
        nodePortWrapper.addPort(port, scriptEngine);

        testToggleRegistry.put(node, nodePortWrapper);

        scriptEngine.startScript();
    }

    public void stopTestToggleTask(final NodeInterface node, final ServoPort port) {
        LOGGER.info("Stop test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.get(node);

        if (nodePortWrapper != null) {
            Set<Port<?>> toRemove = new HashSet<>();
            if (port != null) {
                toRemove.add(port);
            }
            else {
                toRemove.addAll(nodePortWrapper.getKeySet());
            }

            for (Port<?> removePort : toRemove) {
                ScriptEngine<PortScripting> engine = nodePortWrapper.removePort(removePort);

                if (engine != null) {
                    LOGGER.info("Found a node scripting engine in the registry: {}", engine);
                    try {
                        engine.stopScript(Long.valueOf(2000));
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Stop script failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No scripting engine found for node: {}", node);
                }
            }

            if (nodePortWrapper.isEmpty()) {
                LOGGER.info("No more ports registered for node: {}", node);
                testToggleRegistry.remove(node);
            }
        }
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save servo labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.servoPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.servoPortListPanel.refreshView());
        }
    }

    private void addServoPortModelListener(final NodeInterface selectedNode) {

        LOGGER.info("Add servo port model listener for node: {}", selectedNode);

        final Disposable disp = this.portConfigChangeEventSubject.subscribe(evt -> {
            LOGGER.info("Received event: {}", evt);

            final PortTypeAware port = evt.getPort();

            // update port config
            try {
                final ServoPort servoPort = new ServoPort();
                servoPort.setId(port.getPortNumber());

                LOGGER.info("Prepared servo port: {}", servoPort);

                switchingNodeService
                        .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(), servoPort,
                                null, evt.getPortConfig());
            }
            catch (Exception ex) {
                LOGGER.warn("Set the servoport config failed.", ex);
                selectedNode.setNodeHasError(true);
                selectedNode.setReasonData("Set the servoport config failed.");
            }

            boolean sendPositionUpdate = true;
            if (evt.getPortConfig().size() == 1
                    && evt.getPortConfig().get(BidibLibrary.BIDIB_PCFG_SERVO_SPEED) != null) {
                // only servo speed was changed
                sendPositionUpdate = false;
                LOGGER.info("The servo speed was changed.");
            }

            if (sendPositionUpdate && evt.getPortValue() != null) {
                // adjust the port value to the new bounds
                try {
                    // create a new instance of the port
                    final ServoPort servoPort = new ServoPort();
                    servoPort.setId(port.getPortNumber());

                    // we must change the port value to get new positioning
                    int currentPortValue = evt.getPortValue().intValue();
                    servoPort.setValue(currentPortValue < 255 ? currentPortValue + 1 : currentPortValue - 1);

                    switchingNodeService
                            .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(),
                                    servoPort);
                    LOGGER.info("Test pressed for port: {}, port.value: {}", port, servoPort.getValue());

                    Thread.sleep(50);

                    servoPort.setValue(currentPortValue);
                    switchingNodeService
                            .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(),
                                    servoPort);
                    LOGGER.info("Test pressed for port: {}, port.value: {}", port, servoPort.getValue());

                }
                catch (Exception ex) {
                    LOGGER.warn("Set the servoport status failed.", ex);
                    selectedNode.setNodeHasError(true);
                    selectedNode.setReasonData("Set the servoport status failed.");
                }
            }
        });

        compDisp.add(disp);
    }

}
