package org.bidib.wizard.mvc.main.model;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import javax.swing.table.DefaultTableModel;

import org.apache.commons.lang3.StringUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.LabelAware;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelChangedListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SingleColumnTableModel<T extends LabelAware> extends DefaultTableModel {

    private static final Logger LOGGER = LoggerFactory.getLogger(SingleColumnTableModel.class);

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_ZERO = 0;

    private final Class<T> modelClazz;

    private String[] columnNames;

    final List<LabelChangedListener<T>> labelChangedListeners = new LinkedList<>();

    public SingleColumnTableModel(Class<T> modelClazz) {
        this.modelClazz = modelClazz;
        initialize();
        setColumnIdentifiers(columnNames);
    }

    protected void initialize() {
        columnNames = new String[] { Resources.getString(modelClazz, "label") };
    }

    @Override
    public boolean isCellEditable(int row, int column) {

        return true;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_ZERO:
                return modelClazz;
            default:
                return String.class;
        }
    }

    public void addRow(T model) {
        if (model != null) {
            Object[] rowData = new Object[columnNames.length];
            rowData[COLUMN_ZERO] = model;
            addRow(rowData);
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        switch (column) {
            case COLUMN_ZERO:

                if (value != null && StringUtils.isBlank(value.toString())) {
                    value = null;
                    LOGGER.info("Set the null value instead of empty string.");
                }

                T model = (T) super.getValueAt(row, COLUMN_ZERO);
                String label = value != null ? value.toString() : null;

                if (!Objects.equals(model.getLabel(), label)) {

                    model.setLabel(label);
                    super.setValueAt(model, row, COLUMN_ZERO);

                    fireLabelChanged(model, label);
                }
            default:
                break;
        }
    }

    public void addLabelChangedListener(LabelChangedListener<T> macroLabelListener) {
        if (!this.labelChangedListeners.contains(macroLabelListener)) {
            this.labelChangedListeners.add(macroLabelListener);
        }
    }

    private void fireLabelChanged(T model, String label) {
        for (LabelChangedListener<T> l : labelChangedListeners) {
            l.labelChanged(model, label);
        }
    }

}
