package org.bidib.wizard.mvc.loco.model.command;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

import javax.swing.Timer;

import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.PomOperation;
import org.bidib.jbidibc.messages.enums.PomProgState;
import org.bidib.jbidibc.messages.utils.CollectionUtils;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.model.loco.LocoModel;
import org.bidib.wizard.mvc.loco.model.SpeedometerProgBeanModel;
import org.bidib.wizard.mvc.loco.view.PomProgrammerRequestListener;
import org.bidib.wizard.mvc.loco.view.command.PomProgResultListener;
import org.bidib.wizard.mvc.pom.model.PomProgrammerModel;
import org.bidib.wizard.mvc.pom.model.ProgCommandAwareBeanModel;
import org.bidib.wizard.mvc.pom.model.ProgCommandAwareBeanModel.ExecutionType;
import org.bidib.wizard.mvc.pom.model.command.PomOperationCommand;
import org.bidib.wizard.mvc.pom.model.command.PomOperationContinueAfterTimeoutCommand;
import org.bidib.wizard.mvc.pom.model.command.PomOperationIfElseCommand;
import org.bidib.wizard.mvc.pom.model.listener.ProgCommandListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.reactivex.rxjava3.core.SingleObserver;

public class PomRequestProcessor<M extends ProgCommandAwareBeanModel> {

    private static final Logger LOGGER = LoggerFactory.getLogger(PomRequestProcessor.class);

    public static final int DEFAULT_TIMEOUT = 1000;

    private int timeout = DEFAULT_TIMEOUT;

    private final SpeedometerProgBeanModel speedoProgBeanModel;

    private final PomProgrammerModel cvProgrammerModel;

    private final PomProgrammerRequestListener pomProgrammerRequestListener;

    private final PomProgResultListener pomProgResultListener;

    private Timer readTimeout;

    private Object readTimeoutLock = new Object();

    private final SettingsService settingsService;

    public PomRequestProcessor(final SpeedometerProgBeanModel speedoProgBeanModel, final LocoModel locoModel,
        final PomProgrammerModel cvProgrammerModel, final PomProgrammerRequestListener pomProgrammerRequestListener,
        final PomProgResultListener pomProgResultListener, final SettingsService settingsService) {
        this.speedoProgBeanModel = speedoProgBeanModel;
        // this.locoModel = locoModel;
        this.cvProgrammerModel = cvProgrammerModel;
        this.pomProgrammerRequestListener = pomProgrammerRequestListener;
        this.pomProgResultListener = pomProgResultListener;
        this.settingsService = settingsService;
    }

    public void submitProgCommands(
        final List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands,
        final SingleObserver<String> finishAction) {

        LOGGER.info("Submit the new commands for execution: {}", pomProgCommands);

        List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
            speedoProgBeanModel.getProgCommands();
        progCommands.clear();

        progCommands.addAll(pomProgCommands);

        LOGGER.info("Set the finish action in the speedoProgBeanModel: {}", finishAction);
        speedoProgBeanModel.setFinishAction(finishAction);

        // clear the executed commands
        speedoProgBeanModel.getExecutedProgCommands().clear();

        timeout = settingsService.getExperimentalSettings().getSpeedometerTimeout();
        LOGGER.info("Current value of speedometer timeout: {}", timeout);

        startTimeoutControl(timeout);

        fireNextCommand();
    }

    protected void sendRequest(PomAddressData decoderAddress, PomOperation operation, int cvNumber, int cvValue) {
        LOGGER.info("Send the POM request, cvNumber: {}, cvValue: {}", cvNumber, cvValue);

        cvProgrammerModel.setCvNumber(cvNumber);
        cvProgrammerModel.setCvValue(null);

        cvProgrammerModel.setPomProgState(PomProgState.POM_PROG_START);

        // must trigger the controller to send the POM request
        pomProgrammerRequestListener.sendRequest(decoderAddress, operation, cvNumber, cvValue);

    }

    protected void fireNextCommand() {
        // disable the input elements
        pomProgResultListener.disableInputElements();

        List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
            speedoProgBeanModel.getProgCommands();
        LOGGER.info("Prepared commands for addressing the decoder: {}", progCommands);

        // remove the executing command from the prog commands list ...
        PomOperationCommand<?> progCommand = progCommands.remove(0);

        speedoProgBeanModel.setCurrentOperation(progCommand.getPomOperation());
        LOGGER.info("Updated current operation: {}", speedoProgBeanModel.getCurrentOperation());

        switch (progCommand.getPomOperation()) {
            case WR_BIT:
            case WR_BYTE:
                speedoProgBeanModel.setExecution(ExecutionType.WRITE);
                break;
            default:
                speedoProgBeanModel.setExecution(ExecutionType.READ);
                break;
        }
        speedoProgBeanModel.setExecutingProgCommand(progCommand);

        publishLogText("Prog write request: {}", progCommand);

        // send to bidib
        sendRequest(progCommand.getDecoderAddress(), progCommand.getPomOperation(), progCommand.getCvNumber(),
            progCommand.getCvValue());
    }

    protected void startTimeoutControl(int timeout) {
        LOGGER.info("Timeout control is started, timeout: {}.", timeout);

        synchronized (readTimeoutLock) {
            if (readTimeout != null) {
                LOGGER.info("The timeout control is already assigned: {}", readTimeout);
                if (readTimeout.isRunning()) {
                    LOGGER.info("The timeout control is already running and will be stopped.");
                    readTimeout.stop();
                }
                readTimeout = null;
            }

            readTimeout = new Timer(timeout, new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {

                    LOGGER
                        .warn("Timeout control has expired. Current executing command: {}",
                            speedoProgBeanModel.getExecutingProgCommand());
                    publishLogText("Timeout control has expired. No answer from decoder received.");

                    final PomOperationCommand<? extends ProgCommandAwareBeanModel> currentExecutingProgCommand =
                        speedoProgBeanModel.getExecutingProgCommand();
                    // TODO support retry count
                    if (currentExecutingProgCommand != null && currentExecutingProgCommand.decrementRetryCount() > -1) {

                        // add the command again at head
                        List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
                            speedoProgBeanModel.getProgCommands();

                        LOGGER.info("Add current executing command at head: {}", currentExecutingProgCommand);
                        progCommands.add(0, currentExecutingProgCommand);
                    }

                    // check if we have more commands
                    if (CollectionUtils.hasElements(speedoProgBeanModel.getProgCommands())) {
                        LOGGER.info("Prepare the next command to send.");
                        startTimeoutControl(timeout);
                        fireNextCommand();
                    }
                    else if (speedoProgBeanModel.getExecutingProgCommand() instanceof PomOperationIfElseCommand) {
                        boolean ignoreError = false;

                        PomOperationCommand<M> executingProgCommand =
                            (PomOperationCommand<M>) speedoProgBeanModel.getExecutingProgCommand();
                        LOGGER.info("The executingProgCommand: {}", executingProgCommand);

                        if (executingProgCommand instanceof PomOperationIfElseCommand) {
                            PomOperationIfElseCommand<M> ifElseCommand =
                                (PomOperationIfElseCommand<M>) executingProgCommand;
                            LOGGER.info("The executing command is a if-else-command: {}", ifElseCommand);

                            List<PomOperationCommand<M>> failureCommands = ifElseCommand.getProgCommandsFailure();
                            if (CollectionUtils.hasElements(failureCommands)) {
                                LOGGER.info("Found failure commands to be executed: {}", failureCommands);

                                // check if we can continue after timeout and ignore the failure ...
                                if (failureCommands.get(0) instanceof PomOperationContinueAfterTimeoutCommand) {
                                    LOGGER
                                        .info(
                                            "The first failure command is a PomOperationContinueAfterTimeoutCommand. This is no error, we can proceed.");

                                    // add the new commands to process
                                    List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
                                        speedoProgBeanModel.getProgCommands();
                                    progCommands.clear();
                                    progCommands.addAll(failureCommands);

                                    // ignore the error
                                    ignoreError = true;
                                }
                            }
                        }
                        if (!ignoreError) {
                            LOGGER.info("No more commands to send, enable the input elements.");

                            // show an error and enable input elements
                            cvProgrammerModel.setPomProgState(PomProgState.POM_PROG_NO_ANSWER);

                            SingleObserver<String> finishAction = speedoProgBeanModel.getFinishAction();
                            if (finishAction != null) {
                                finishAction.onError(null);
                            }
                        }
                        else {
                            LOGGER.info("Start the timeout control and fire the next command.");
                            startTimeoutControl(timeout);
                            fireNextCommand();
                        }
                    }
                    else {
                        LOGGER.info("Timeout expired. No more commands to send, enable the input elements.");
                        // show an error and enable input elements
                        cvProgrammerModel.setPomProgState(PomProgState.POM_PROG_NO_ANSWER);

                        SingleObserver<String> finishAction = speedoProgBeanModel.getFinishAction();
                        if (finishAction != null) {
                            finishAction.onError(null);
                        }
                    }
                }
            });
            readTimeout.setRepeats(false);
            readTimeout.start();
        }
    }

    protected void stopTimeoutControl() {
        LOGGER.info("Timeout control is stopped.");

        synchronized (readTimeoutLock) {
            if (readTimeout != null) {
                LOGGER.info("The timeout control is assigned and will be stopped: {}", readTimeout);
                readTimeout.stop();

                readTimeout = null;
            }
        }
    }

    private void publishLogText(String logLine, Object... args) {

        pomProgResultListener.addLogText(logLine, args);
    }

    public void addPomProgrammerModelListeners() {
        LOGGER.info("Add the pom programmer model listeners.");

        // handle the finished prog commands
        cvProgrammerModel.addProgCommandListener(new ProgCommandListener() {

            @Override
            public void progPomFinished(PomProgState pomProgState) {
                LOGGER.info("The prog command has finished, pomProgState: {}", pomProgState);

                if (!PomProgState.POM_PROG_START.equals(pomProgState)
                    && !PomProgState.POM_PROG_RUNNING.equals(pomProgState)) {

                    PomOperationCommand<M> executingProgCommand =
                        (PomOperationCommand<M>) speedoProgBeanModel.getExecutingProgCommand();
                    if (executingProgCommand != null) {
                        LOGGER.info("The executingProgCommand: {}", executingProgCommand);
                        executingProgCommand.setProgStateResult(pomProgState);

                        boolean ignoreError = false;

                        if (executingProgCommand instanceof PomOperationIfElseCommand) {
                            PomOperationIfElseCommand<M> ifElseCommand =
                                (PomOperationIfElseCommand<M>) executingProgCommand;
                            LOGGER.info("The executing command is a if-else-command: {}", ifElseCommand);

                            // we must check if the result is successful or not
                            if (!PomProgState.POM_PROG_OKAY.equals(ifElseCommand.getProgStateResult())
                                || !ifElseCommand.isExpectedResult()) {
                                LOGGER
                                    .info(
                                        "The command was not successful executed or expected result was not received!");
                                List<PomOperationCommand<M>> failureCommands = ifElseCommand.getProgCommandsFailure();
                                if (CollectionUtils.hasElements(failureCommands)) {
                                    LOGGER.info("Found failure commands to be executed: {}", failureCommands);
                                    // add the new commands to process
                                    List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
                                        speedoProgBeanModel.getProgCommands();
                                    progCommands.clear();
                                    progCommands.addAll(failureCommands);

                                    // ignore the error
                                    ignoreError = true;
                                }
                            }
                            else {
                                LOGGER.info("The command was successful executed!");
                                List<PomOperationCommand<M>> successCommands = ifElseCommand.getProgCommandsSuccess();
                                if (CollectionUtils.hasElements(successCommands)) {
                                    LOGGER.info("Found success commands to be executed: {}", successCommands);
                                    // add the new commands to process
                                    List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> progCommands =
                                        speedoProgBeanModel.getProgCommands();
                                    progCommands.clear();
                                    progCommands.addAll(successCommands);
                                }

                            }
                        }

                        // we must check if the result is successful or not
                        if (PomProgState.POM_PROG_OKAY.equals(executingProgCommand.getProgStateResult())) {
                            LOGGER
                                .info(
                                    "PostExecute the command. This will set the value in the speedoProgBeanModel, command: {}",
                                    executingProgCommand);

                            // update the value in the bean model
                            executingProgCommand.postExecute((M) speedoProgBeanModel);
                        }
                        else if (!ignoreError) {
                            // if an error occurs we must stop processing!!!!

                            LOGGER
                                .warn("Clear remaining prog commands because command has finished with an error: {}",
                                    executingProgCommand);
                            if (CollectionUtils.hasElements(speedoProgBeanModel.getProgCommands())) {
                                speedoProgBeanModel.getProgCommands().clear();
                            }

                            publishLogText("Error detected. Please try again.");
                        }

                        // keep the executed commands
                        speedoProgBeanModel.getExecutedProgCommands().add(executingProgCommand);
                        executingProgCommand = null;
                    }

                    if (CollectionUtils.hasElements(speedoProgBeanModel.getProgCommands())) {
                        LOGGER.info("Prepare the next command.");

                        startTimeoutControl(timeout);
                        LOGGER.info("Fire the next command.");
                        fireNextCommand();
                    }
                    else {
                        LOGGER.info("No more commands to send, enable the input elements.");
                        stopTimeoutControl();
                        pomProgResultListener.enableInputElements();

                        SingleObserver<String> finishAction = speedoProgBeanModel.getFinishAction();
                        LOGGER.info("Registered finish action: {}", finishAction);
                        if (finishAction != null) {

                            LOGGER.info("Call the finish action.");
                            finishAction.onSuccess(null);

                            // LOGGER.info("Clear the finish action.");
                            // speedoProgBeanModel.setFinishAction(null);
                        }
                    }
                }
            }
        });

        // react on changes of CV value
        cvProgrammerModel
            .addPropertyChangeListener(PomProgrammerModel.PROPERTYNAME_CVVALUE, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    LOGGER
                        .info("The CV value has been changed: {}, currentOperation: {}", evt.getNewValue(),
                            speedoProgBeanModel.getCurrentOperation());

                    if (speedoProgBeanModel.getCurrentOperation() != null) {
                        switch (speedoProgBeanModel.getCurrentOperation()) {
                            case RD_BIT:
                            case RD_BYTE:
                                PomOperationCommand<? extends ProgCommandAwareBeanModel> progCommand =
                                    speedoProgBeanModel.getExecutingProgCommand();
                                progCommand.setCvValueResult((Integer) evt.getNewValue());

                                if (evt.getNewValue() != null) {
                                    publishLogText("Read operation returned: {}", progCommand.getCvValueResult());
                                }
                                break;
                            case WR_BYTE:
                                progCommand = speedoProgBeanModel.getExecutingProgCommand();
                                progCommand.setCvValueResult((Integer) evt.getNewValue());

                                if (evt.getNewValue() != null) {
                                    publishLogText("Write operation returned: {}", progCommand.getCvValueResult());
                                }
                                break;
                            // case RD_BIT:
                            case WR_BIT:
                                /*
                                 * if (ExecutionType.READ.equals(progCommandAwareBeanModel.getExecution())) { Object
                                 * value = evt.getNewValue(); LOGGER.info("Returned read bit value: {}", value); if
                                 * (value != null) { int val = ((Integer) value).intValue(); int bitValue = (val & 0x08)
                                 * == 0x08 ? 1 : 0;
                                 * progCommandAwareBeanModel.getExecutingProgCommand().setCvValueResult(bitValue);
                                 * 
                                 * addLogText("Read bit returned: {}", bitValue); } } else { // nothing to do }
                                 */
                                break;
                            default:
                                break;
                        }
                    }
                    else {
                        LOGGER.info("No operation performed.");
                    }
                }
            });
    }
}
