package org.bidib.wizard.mvc.main.view;

import java.awt.Component;
import java.awt.Point;
import java.io.File;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.bidib.jbidibc.core.schema.bidib2.BiDiB;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.experimental.excel.ExcelExportFactory;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.enums.AccessoryAcknowledge;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.CommandStationPt;
import org.bidib.jbidibc.messages.enums.CommandStationState;
import org.bidib.jbidibc.messages.enums.DetachedState;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.enums.TimeBaseUnitEnum;
import org.bidib.jbidibc.messages.enums.TimingControlEnum;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.LookupService;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.Flag;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroRef;
import org.bidib.wizard.api.model.MacroSaveState;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.BoosterService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.mvc.firmware.controller.FirmwareController;
import org.bidib.wizard.client.common.mvc.firmware.controller.listener.FirmwareControllerListener;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.client.common.view.BusyFrame;
import org.bidib.wizard.client.common.view.cvdef.CvContainer;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeModelRegistry;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.BidibLabelUtils;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.config.AccessoryControllerFactory;
import org.bidib.wizard.config.DccAdvControllerFactory;
import org.bidib.wizard.config.FeaturesControllerFactory;
import org.bidib.wizard.config.FirmwareControllerFactory;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.LocoTableControllerFactory;
import org.bidib.wizard.config.PomProgrammerControllerFactory;
import org.bidib.wizard.config.PtProgrammerControllerFactory;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.dcca.client.controller.DccAdvController;
import org.bidib.wizard.dialog.NodeDetailsDialog;
import org.bidib.wizard.dmx.client.config.DmxModelerControllerFactory;
import org.bidib.wizard.dmx.client.controller.DmxModelerController;
import org.bidib.wizard.firmwarerepo.core.FirmwareRepoService;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.accessory.controller.AccessoryController;
import org.bidib.wizard.mvc.accessory.controller.listener.AccessoryControllerListener;
import org.bidib.wizard.mvc.features.controller.FeaturesController;
import org.bidib.wizard.mvc.features.controller.listener.FeaturesControllerListener;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.locolist.controller.LocoTableController;
import org.bidib.wizard.mvc.main.controller.MainController;
import org.bidib.wizard.mvc.main.controller.MainControllerInterface;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.component.BulkSwitchNodeOperationsDialog;
import org.bidib.wizard.mvc.main.view.exchange.ExportNodeOptionsDialog;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.exchange.SaveConfigOnNodeHelper;
import org.bidib.wizard.mvc.main.view.panel.NodeListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.NodeListActionListener;
import org.bidib.wizard.mvc.pom.controller.PomProgrammerController;
import org.bidib.wizard.mvc.pom.controller.listener.PomProgrammerControllerListener;
import org.bidib.wizard.mvc.pt.controller.PtProgrammerController;
import org.bidib.wizard.mvc.pt.controller.listener.PtProgrammerControllerListener;
import org.bidib.wizard.mvc.pt.view.PtConfirmDialog;
import org.bidib.wizard.utils.FileUtils;
import org.oxbow.swingbits.dialog.task.TaskDialogs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * This listener is called after the value is set in the node.
 */
public class MainNodeListActionListener implements NodeListActionListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(MainNodeListActionListener.class);

    // description, suffix for node files
    private String nodeDescription;

    private final MainView view;

    private final MainModel model;

    @Autowired
    private NodeService nodeService;

    @Autowired
    private CommandStationService commandStationService;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private BoosterService boosterService;

    @Autowired
    private LocoControllerFactory locoControllerFactory;

    @Autowired
    private LocoTableControllerFactory locoTableControllerFactory;

    @Autowired
    private FirmwareControllerFactory firmwareControllerFactory;

    @Autowired
    private PtProgrammerControllerFactory ptProgrammerControllerFactory;

    @Autowired
    private PomProgrammerControllerFactory pomProgrammerControllerFactory;

    @Autowired
    private AccessoryControllerFactory accessoryControllerFactory;

    @Autowired
    private DmxModelerControllerFactory dmxModelerControllerFactory;

    @Autowired
    private DccAdvControllerFactory dccAdvControllerFactory;

    @Autowired
    private FeaturesControllerFactory featuresControllerFactory;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    // registry for CvDefinitionTreeTableModel instances
    @Autowired
    private CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry;

    @Autowired
    private ConsoleService consoleService;

    @Autowired
    private FirmwareRepoService firmwareRepoService;

    private final StatusBar statusBar;

    private final LookupService lookupService;

    public MainNodeListActionListener(final MainView view, final MainModel model, final StatusBar statusBar,
        final LookupService lookupService) {
        this.view = view;
        this.model = model;
        this.statusBar = statusBar;
        this.lookupService = lookupService;

        nodeDescription = Resources.getString(MainController.class, "nodeDescription");
    }

    private void setWaitCursor() {
        view.setBusy(true);
    }

    private void setDefaultCursor() {
        view.setBusy(false);
    }

    private FileFilter getNodeFilter() {

        FileFilter nodeFilter = new FileNameExtensionFilter(nodeDescription, NodeExchangeHelper.NODE_X_EXTENSION);
        return nodeFilter;
    }

    @Override
    public void enableAddressMessages(NodeInterface node) {

        // TODO find a way to set the property on the node and add a listener on the node that writes the information to
        // the node on the bus

        List<Feature> features =
            Arrays
                .asList(new Feature(BidibLibrary.FEATURE_BM_ADDR_DETECT_ON, (node.isAddressMessagesEnabled() ? 1 : 0)));
        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, features);
    }

    @Override
    public void enableDccStart(NodeInterface node) {
        List<Feature> features =
            Arrays.asList(new Feature(BidibLibrary.FEATURE_CTRL_MAC_START_DCC, (node.isDccStartEnabled() ? 1 : 0)));
        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, features);
    }

    @Override
    public void enableExternalStart(NodeInterface node) {
        List<Feature> features =
            Arrays
                .asList(new Feature(BidibLibrary.FEATURE_CTRL_MAC_START_MAN, (node.isExternalStartEnabled() ? 1 : 0)));
        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, features);
    }

    @Override
    public void enableFeedbackMessages(NodeInterface node) {
        List<Feature> features =
            Arrays.asList(new Feature(BidibLibrary.FEATURE_BM_ON, (node.isFeedbackMessagesEnabled() ? 1 : 0)));
        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, features);
    }

    @Override
    public void disableFeedbackMirror(NodeInterface node, boolean disable) {

        nodeService.setFeedbackMirrorDisabled(ConnectionRegistry.CONNECTION_ID_MAIN, node, disable);
    }

    @Override
    public void enableKeyMessages(NodeInterface node) {
        List<Feature> features =
            Arrays.asList(new Feature(BidibLibrary.FEATURE_CTRL_INPUT_NOTIFY, (node.isKeyMessagesEnabled() ? 1 : 0)));
        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, features);
    }

    @Override
    public void features(NodeInterface node) {
        setWaitCursor();

        final FeaturesController featuresController =
            this.featuresControllerFactory
                .createFeaturesController(node, view.getFrame(), this.settingsService.getWizardSettings());

        featuresController.addFeaturesControllerListener(new FeaturesControllerListener() {
            @Override
            public void close() {
                setDefaultCursor();
            }

            @Override
            public void readAll(NodeInterface node) {
                StopWatch sw = new StopWatch();
                sw.start();

                List<Feature> features =
                    nodeService.queryAllFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, false);

                // keep the features in the node
                List<Feature> featureList = new LinkedList<>();
                featureList.addAll(features);
                node.getNode().setFeatures(featureList);

                // set the features in the controller
                featuresController.setFeatures(features);
                sw.stop();

                LOGGER.info("Load features has finished! Total loading duration: {}", sw);
            }

            @Override
            public void write(NodeInterface node, Feature feature) {
                LOGGER.debug("Write feature to node: {}, feature: {}", node, feature);

                nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, Arrays.asList(feature));
            }
        });
        featuresController.start();
    }

    @Override
    public void exportNode(final NodeInterface node) {

        LOGGER.info("export node: {}", node);

        final NodeExchangeHelper helper = new NodeExchangeHelper();
        String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
        helper
            .exportNode(view.getFrame(), node, settingsService, nodeService, switchingNodeService, wizardLabelWrapper,
                cvDefinitionTreeModelRegistry, getNodeFilter(), statusBar, lang);
    }

    @Override
    public void importNode(final NodeInterface node) {

        try {
            final NodeExchangeHelper helper = new NodeExchangeHelper();
            helper
                .importNode(view.getFrame(), node, settingsService, nodeService, switchingNodeService,
                    cvDefinitionTreeModelRegistry, getNodeFilter(), wizardLabelWrapper, statusBar, this,
                    this.lookupService);
        }
        catch (Exception ex) {
            String message = ex.getMessage();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                    .getString(MainNodeListActionListener.class, "importNode.error.message", new Object[] { message }),
                    Resources.getString(MainNodeListActionListener.class, "importNode.error.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void firmwareUpdate(final NodeInterface node) {
        // setWaitCursor();

        final SoftwareVersion newFirmware = node.getUpdateFirmwareVersion();
        LOGGER.info("Show the firmware update dialog for node: {}, newFirmware: {}", node, newFirmware);

        FirmwareController firmwareController =
            firmwareControllerFactory.createFirmwareController(this.firmwareRepoService, node, view.getFrame());

        firmwareController.addFirmwareControllerListener(new FirmwareControllerListener() {
            @Override
            public void viewClosed() {
                LOGGER.info("The firmware update view was closed.");
                setDefaultCursor();
            }
        });
        firmwareController.start(newFirmware);
    }

    @Override
    public void identify(final NodeInterface node) {

        nodeService.identify(ConnectionRegistry.CONNECTION_ID_MAIN, node, node.getIdentifyState());
    }

    @Override
    public void dccAdvView(final NodeInterface node) {
        final DccAdvController dccAdvController = dccAdvControllerFactory.createController(node, view.getDesktop());
        dccAdvController.start();
    }

    @Override
    public void ping(final NodeInterface node, byte[] data) {

        nodeService.ping(ConnectionRegistry.CONNECTION_ID_MAIN, node, data, 0);
    }

    @Override
    public void enableNode(final NodeInterface node) {
        nodeService.enable(ConnectionRegistry.CONNECTION_ID_MAIN, node);
    }

    @Override
    public void disableNode(final NodeInterface node) {
        nodeService.disable(ConnectionRegistry.CONNECTION_ID_MAIN, node);
    }

    @Override
    public Long readUniqueId(final NodeInterface node) {

        return nodeService.readUniqueId(ConnectionRegistry.CONNECTION_ID_MAIN, node);
    }

    @Override
    public void detachAttachNode(final NodeInterface node, boolean detach) {
        try {
            LOGGER.info("detachAttachNode, node: {}, detach: {}", node, detach);
            Long uniqueId = settingsService.getNetBidibSettings().getNetBidibUniqueId();
            if (detach) {
                nodeService.detach(ConnectionRegistry.CONNECTION_ID_MAIN, node, uniqueId);

                node.setDetachedState(DetachedState.DETACHED);
            }
            else {
                // try to attach the node
                nodeService.attach(ConnectionRegistry.CONNECTION_ID_MAIN, node, uniqueId);

                node.setDetachedState(DetachedState.ATTACHED);
            }
        }
        catch (RuntimeException ex) {
            String message = ex.getMessage();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources
                        .getString(MainNodeListActionListener.class, "localLogoffRejected.error.message",
                            new Object[] { message }),
                    Resources.getString(MainNodeListActionListener.class, "localLogoffRejected.error.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void labelChanged(final NodeInterface node, String label) {
        LOGGER.info("The label has changed, node: {}, label: {}", node, label);

        // write label to node if feature is available
        int stringSize = node.getNode().getStringSize();
        if (stringSize > 0) {
            String userName = label;
            if (label.length() > stringSize) {
                userName = label.substring(0, stringSize);
                LOGGER.info("Shrinked node string from '{}' to:' {}'", label, userName);
            }

            LOGGER.info("Set the new username on node: {}", userName);

            nodeService.setNodeDetails(ConnectionRegistry.CONNECTION_ID_MAIN, node, userName);
        }

        // change the value in the list of node labels
        try {
            NodeLabels nodeLabels = getNodeLabels(node);
            BidibLabelUtils.replaceNodeLabel(nodeLabels, label);
            saveLabels(node);
        }
        catch (InvalidConfigurationException ex) {
            LOGGER.warn("Save node labels failed.", ex);

            String labelPath = ex.getReason();
            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(null),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                    Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"), JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void loco(final NodeInterface node) {
        LOGGER.info("Open the loco controller, node: {}", node);

        try {
            LocoController locoController =
                locoControllerFactory
                    .createLocoController(node.getCommandStationNode(), view.getFrame(), model.getNodeProvider());

            locoController.start(null, null, null);
        }
        catch (Exception ex) {
            LOGGER.warn("Create loco controller failed.", ex);
        }
    }

    @Override
    public void dccAccessory(NodeInterface node, int x, int y) {

        final AccessoryController accessoryController =
            accessoryControllerFactory.createAccessoryController(node, view.getFrame(), new Point(x, y));
        accessoryController.addAccessoryControllerListener(new AccessoryControllerListener() {

            @Override
            public AccessoryAcknowledge sendAccessoryRequest(
                final NodeInterface node, AddressData dccAddress, Integer aspect, Integer switchTime,
                TimeBaseUnitEnum timeBaseUnit, TimingControlEnum timingControl) {
                LOGGER.info("send accessory request, node: {}", node);
                AccessoryAcknowledge acknowledgde =
                    commandStationService
                        .setDccAccessory(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode(),
                            dccAddress, aspect, switchTime, timeBaseUnit, timingControl);

                return acknowledgde;
            }
        });

        accessoryController.start();
    }

    @Override
    public void locoCv(NodeInterface node, int x, int y) {

        final PomProgrammerController pomProgrammerController =
            pomProgrammerControllerFactory
                .createPomProgrammerController(node.getCommandStationNode(), view.getFrame(), new Point(x, y));

        pomProgrammerController.addPomProgrammerControllerListener(new PomProgrammerControllerListener() {

            @Override
            public void sendRequest(
                final CommandStationNodeInterface node, PomAddressData locoAddress, CommandStationPom opCode,
                int cvNumber, int cvValue) {
                LOGGER.info("Send POM request.");

                PomAcknowledge pomAck =
                    commandStationService
                        .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node, locoAddress, opCode, cvNumber,
                            cvValue);
                LOGGER.info("Received pomAck: {}", pomAck);
            }

            @Override
            public void close() {
            }
        });

        pomProgrammerController.start(view.getDesktop(), null);
    }

    @Override
    public void locoCvPt(NodeInterface node, int x, int y) {
        setWaitCursor();

        if (!settingsService.getWizardSettings().isPtModeDoNotConfirmSwitch()) {
            PtConfirmDialog ptConfirmDialog = new PtConfirmDialog(node, settingsService, new Point(x, y));
            if (JOptionPane.CANCEL_OPTION == ptConfirmDialog.getResult()) {
                LOGGER.info("User cancelled ptConfirmDialog.");
                setDefaultCursor();
                return;
            }
        }
        else {
            LOGGER.info("Switch to PT programming is disabled by user!");
        }
        setDefaultCursor();

        final PtProgrammerController ptProgrammerController =
            ptProgrammerControllerFactory.createPtProgrammerController(node, view.getFrame(), new Point(x, y));

        ptProgrammerController.addPtProgrammerControllerListener(new PtProgrammerControllerListener() {
            @Override
            public void close() {
            }

            @Override
            public void sendRequest(final NodeInterface node, CommandStationPt opCode, int cvNumber, int cvValue) {
                LOGGER.info("Send PT request, opCode: {}, cvNumber: {}, cvValue: {}", opCode, cvNumber, cvValue);
                commandStationService
                    .sendCvPtRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode(), opCode,
                        cvNumber, cvValue);
            }

            @Override
            public void sendCommandStationStateRequest(
                final NodeInterface node, CommandStationState commandStationState) {
                LOGGER.info("Send the commandStationState: {}", commandStationState);

                CommandStationStatus commandStationStatus = CommandStationStatus.valueOf(commandStationState);
                commandStationService
                    .setCommandStationState(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode(),
                        commandStationStatus);

            }

            @Override
            public CommandStationState getCurrentCommandStationState(final NodeInterface node) {
                LOGGER.info("Query the command station state.");

                return commandStationService
                    .queryCommandStationStateBlocking(ConnectionRegistry.CONNECTION_ID_MAIN,
                        node.getCommandStationNode());
            }
        });
        ptProgrammerController.start(view.getDesktop());
    }

    @Override
    public void nodeDetails(final NodeInterface node, int x, int y) {
        // show the node details dialog
        new NodeDetailsDialog(node, x, y);

    }

    @Override
    public void bulkSwitchDialog(final NodeInterface node, int x, int y) {

        // show the reset dialog for the selected node
        final BulkSwitchNodeOperationsDialog dialog =
            new BulkSwitchNodeOperationsDialog(view.getDesktop(), node, this.nodeService, this.switchingNodeService,
                this.boosterService, this.settingsService, this.consoleService);
        dialog.showDialog();
    }

    @Override
    public void saveOnNode(final NodeInterface node, Map<String, Object> params) {

        LOGGER.info("Save the configuration to the node: {}", node);

        final SaveConfigOnNodeHelper configOnNodeHelper = new SaveConfigOnNodeHelper();
        configOnNodeHelper.saveOnNode(model.getConnectionId(), nodeService, switchingNodeService, node, params);
    }

    @Override
    public void dmxModeler(final NodeInterface node) {
        setWaitCursor();
        LOGGER.info("Open the DMX modeler for node: {}", node);
        try {
            DmxModelerController dmxModelerController =
                dmxModelerControllerFactory
                    .createDmxModelerController(node, view.getFrame(), () -> this.model.getNodeProvider());

            dmxModelerController.start(view.getDesktop(), model);
        }
        catch (Exception ex) {
            LOGGER.warn("Open DMX modeler failed.", ex);
            // show dialog for user
            JOptionPane
                .showMessageDialog(view.getFrame(), "Open DMX modeler failed.", "Open DMX modeler",
                    JOptionPane.ERROR_MESSAGE);
        }
        finally {
            setDefaultCursor();
        }
    }

    @Override
    public void locoList(final NodeInterface node) {
        setWaitCursor();
        LOGGER.info("Open the loco table for node: {}", node);
        try {

            final ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            final MainControllerInterface mainController =
                applicationContext.get(DefaultApplicationContext.KEY_MAIN_CONTROLLER, MainControllerInterface.class);

            LocoTableController locoTableController =
                applicationContext.get(DefaultApplicationContext.KEY_LOCO_TABLE_CONTROLLER, LocoTableController.class);
            if (locoTableController == null) {
                LOGGER.info("Create new locoTableController instance and register in context.");
                locoTableController =
                    locoTableControllerFactory.createLocoTableController(node, view.getDesktop(), view.getFrame());
                applicationContext.register(DefaultApplicationContext.KEY_LOCO_TABLE_CONTROLLER, locoTableController);
            }
            locoTableController.start(mainController);
            locoTableController.setSelectedNode(node);

        }
        catch (Exception ex) {
            LOGGER.warn("Open loco table failed.", ex);
            // show dialog for user
            JOptionPane
                .showMessageDialog(view.getFrame(), "Open loco table failed.", "Open loco table",
                    JOptionPane.ERROR_MESSAGE);
        }
        finally {
            setDefaultCursor();
        }
    }

    @Override
    public void reset(NodeInterface node) {
        LOGGER.info("Reset the node: {}", node);

        nodeService.reset(ConnectionRegistry.CONNECTION_ID_MAIN, node);
    }

    private static final String WORKING_DIR_EXCEL_EXCHANGE_KEY = "excelExchange";

    @Override
    public void generateDocumentation(final NodeInterface node) {

        if (node != null) {

            try {
                setWaitCursor();

                LOGGER.info("Generate documentation for node: {}", node);

                String bidibNodeFilesDescription =
                    Resources.getString(NodeListPanel.class, "bidibNodeFilesDescription");
                final FileFilter bidibNodesFilter =
                    new FileNameExtensionFilter(bidibNodeFilesDescription, FileUtils.EXCEL_EXTENSION);

                String defaultFileName = node.toString() + "." + FileUtils.EXCEL_EXTENSION;
                if (StringUtils.isNotBlank(node.getNode().getStoredString(StringData.INDEX_USERNAME))) {
                    defaultFileName =
                        node.getNode().getStoredString(StringData.INDEX_USERNAME) + "." + FileUtils.EXCEL_EXTENSION;
                }

                defaultFileName = FileUtils.escapeInvalidFilenameCharacters(defaultFileName, "_");

                final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
                String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_EXCEL_EXCHANGE_KEY);

                final FileDialog dialog =
                    new FileDialog(view.getFrame(), FileDialog.SAVE, storedWorkingDirectory, defaultFileName,
                        bidibNodesFilter) {

                        private JCheckBox checkLoadMacroContent;

                        private JCheckBox checkLoadFeatures;

                        private JCheckBox checkLoadCVs;

                        @Override
                        public void approve(String selectedFile) {

                            try {
                                final ExportNodeOptionsDialog exportNodeOptionsDialog =
                                    new ExportNodeOptionsDialog(view.getFrame(),
                                        Resources.getString(MainNodeListActionListener.class, "generateOptions.title"),
                                        node) {

                                        private static final long serialVersionUID = 1L;

                                        @Override
                                        protected Component getAdditionalPanel() {

                                            // prepare a panel with checkboxes for loading macro content before export
                                            JPanel additionalPanel = new JPanel();
                                            additionalPanel
                                                .setLayout(new BoxLayout(additionalPanel, BoxLayout.PAGE_AXIS));

                                            if (NodeUtils.hasSwitchFunctions(node.getUniqueId())) {
                                                checkLoadMacroContent =
                                                    new JCheckBox(
                                                        Resources
                                                            .getString(MainController.class, "checkLoadMacroContent"),
                                                        true);
                                                additionalPanel.add(checkLoadMacroContent);
                                            }
                                            checkLoadFeatures =
                                                new JCheckBox(
                                                    Resources.getString(MainController.class, "checkLoadFeatures"),
                                                    true);
                                            additionalPanel.add(checkLoadFeatures);
                                            checkLoadCVs =
                                                new JCheckBox(Resources.getString(MainController.class, "checkLoadCVs"),
                                                    true);
                                            additionalPanel.add(checkLoadCVs);
                                            return additionalPanel;
                                        }
                                    };
                                exportNodeOptionsDialog.setVisible(true);

                                if (exportNodeOptionsDialog.getResult() == JOptionPane.OK_OPTION) {
                                    LOGGER.info("User confirmed options.");
                                }
                                else {
                                    LOGGER.info("User cancelled options.");

                                    return;
                                }
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Show exportNodeOptionsDialog failed.", ex);
                                return;
                            }

                            final String workingDir = Paths.get(selectedFile).toString();
                            LOGGER.info("Save current workingDir: {}", workingDir);
                            wizardSettings.setWorkingDirectory(WORKING_DIR_EXCEL_EXCHANGE_KEY, workingDir);

                            final File file = new File(selectedFile);
                            String fileName = file.getPath();

                            BusyFrame busyFrame =
                                DefaultApplicationContext
                                    .getInstance().get(DefaultApplicationContext.KEY_MAIN_FRAME, BusyFrame.class);

                            try {
                                busyFrame.setBusy(true);

                                // collect information of the node
                                // fetch all macros content from the node
                                LOGGER.info("Load the macro content of the node before export: {}", node);

                                boolean nodeHasFlatPortModel = node.getNode().isPortFlatModelAvailable();
                                LOGGER.info("Prepare the nodeState, nodeHasFlatPortModel: {}", nodeHasFlatPortModel);

                                List<Macro> macroList = new LinkedList<Macro>();
                                for (Macro macro : node.getMacros()) {
                                    Macro macroWithContent =
                                        switchingNodeService
                                            .getMacroContent(ConnectionRegistry.CONNECTION_ID_MAIN,
                                                node.getSwitchingNode(), macro);
                                    LOGGER.info("Load macro content: {}", macroWithContent);

                                    // reset the changed flag on the macro
                                    macroWithContent.setMacroSaveState(MacroSaveState.PERMANENTLY_STORED_ON_NODE);
                                    macroWithContent.setFlatPortModel(nodeHasFlatPortModel);

                                    macroList.add(macroWithContent);
                                }
                                LOGGER.info("Set the new macros for the node: {}", macroList);
                                node.setMacros(macroList);

                                final CvContainer cvContainer =
                                    cvDefinitionTreeModelRegistry.getCvContainer(node.getUniqueId());

                                final Map<String, CvNode> cvNumberToNodeMap =
                                    cvContainer != null ? cvContainer.getCvNumberToNodeMap() : new HashMap<>();

                                String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
                                final BiDiB bidib =
                                    org.bidib.wizard.utils.NodeUtils
                                        .convertToBiDiB(node, cvNumberToNodeMap, lang, !checkLoadCVs.isSelected(),
                                            wizardLabelWrapper);
                                LOGGER.info("Converted bidib: {}", bidib);

                                ExcelExportFactory.exportDocumentation(bidib, fileName, lang);

                                MainNodeListActionListener.this.statusBar
                                    .setStatusText(
                                        Resources
                                            .getString(NodeListPanel.class, "export-documentation-passed", fileName),
                                        StatusBar.DISPLAY_NORMAL);
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Generate documentation failed.", ex);

                                TaskDialogs
                                    .build(JOptionPane.getFrameForComponent(view.getFrame()),
                                        Resources
                                            .getString(NodeListPanel.class, "export-documentation-failed.instruction"),
                                        Resources.getString(NodeListPanel.class, "export-documentation-failed"))
                                    .title(Resources.getString(NodeListPanel.class, "export-documentation.title"))
                                    .exception(ex);
                            }
                            finally {
                                busyFrame.setBusy(false);
                            }
                        }
                    };
                dialog.showDialog();
            }
            catch (Exception ex) {
                LOGGER.warn("Generate documentation failed.", ex);
            }
            finally {
                setDefaultCursor();
            }
        }
    }

    @Override
    public void deleteNodeLabels(final NodeInterface node) {

        if (node != null) {
            final NodeLabels nodeLabels = getNodeLabels(node);
            String label = BidibNodeNameUtils.prepareLabel(node, nodeLabels, false, false).getNodeLabel();

            boolean delete =
                TaskDialogs
                    .ask(JOptionPane.getFrameForComponent(view.getFrame()),
                        Resources.getString(MainNodeListActionListener.class, "deleteLabels"),
                        Resources.getString(MainNodeListActionListener.class, "deleteLabelsOfNode", label));

            if (delete) {
                wizardLabelWrapper.deleteNodeLabels(node.getUniqueId());

                for (Accessory accessory : node.getAccessories()) {
                    accessory.setLabel(null);
                    for (MacroRef aspect : accessory.getAspects()) {
                        aspect.setLabel(null);
                    }
                }
                for (Macro macro : node.getMacros()) {
                    macro.setLabel(null);
                }
                for (Flag flag : node.getFlags()) {
                    flag.setLabel(null);
                }

                for (Port<?> port : node.getPorts()) {
                    port.setLabel(null);
                }

                node.setLabel(null);
            }
        }
    }

    private NodeLabels getNodeLabels(final NodeInterface node) {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(node.getUniqueId());
        return nodeLabels;
    }

    private void saveLabels(final NodeInterface node) {
        try {
            long uniqueId = node.getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void reloadNode(final NodeInterface node) {
        LOGGER.info("Reload the node configuration: {}", node);

        nodeService.reloadNode(ConnectionRegistry.CONNECTION_ID_MAIN, node);
    }
}
