package org.bidib.wizard.mvc.stepcontrol.view;

import java.awt.Component;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.image.BufferedImage;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.JViewport;
import javax.swing.UIManager;
import javax.swing.table.TableModel;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.common.exception.VetoChangeException;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.mvc.common.view.table.CustomCheckBox;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlAspect;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlModel;
import org.bidib.wizard.mvc.stepcontrol.model.TurnTableType;
import org.bidib.wizard.mvc.stepcontrol.view.AspectEditorPanel.EditorType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.HierarchicalPanel;
import com.jidesoft.grid.HierarchicalTable;
import com.jidesoft.grid.HierarchicalTableComponentFactory;
import com.jidesoft.grid.TableColumnChooser;

public class AspectTable extends HierarchicalTable {
    private static final Logger LOGGER = LoggerFactory.getLogger(AspectTable.class);

    private static final long serialVersionUID = 1L;

    private final AspectTableModel aspectTableModel;

    private final String emptyTableText;

    private final ImageIcon selectedIcon;

    private final ImageIcon unselectedIcon;

    private final ImageIcon selectedRolloverIcon;

    private final ImageIcon unselectedRolloverIcon;

    private final AspectCallbackListener<StepControlAspect> callbackListener;

    public AspectTable(final AspectTableModel tableModel, String emptyTableText, final ImageIcon selectedIcon,
        final ImageIcon unselectedIcon, final AspectCallbackListener<StepControlAspect> callbackListener) {
        super(tableModel);

        this.aspectTableModel = tableModel;
        this.emptyTableText = emptyTableText;

        this.callbackListener = callbackListener;

        this.selectedIcon = selectedIcon;
        this.unselectedIcon = unselectedIcon;

        this.selectedRolloverIcon = addBorderToIcon(selectedIcon);
        this.unselectedRolloverIcon = addBorderToIcon(unselectedIcon);

    }

    @Override
    public TableModel getStyleModel() {
        return aspectTableModel; // designate it as the style model
    }

    public void turnTableTypeChanged(TurnTableType turnTableType) {

        LOGGER.info("The turntable type has changed: {}", turnTableType);

        collapseAllRows();
        clearSelection();

        // show or hide angle column
        if (TurnTableType.round == turnTableType) {
            TableColumnChooser.showAllColumns(this);
        }
        else {
            // TableColumnChooser.hideColumn(this, AspectTableModel.COLUMN_ANGLE);
        }
    }

    public String getEmptyTableText() {
        return emptyTableText;
    }

    private static ImageIcon addBorderToIcon(final ImageIcon selectedIcon) {
        final ImageIcon borderIcon = ImageUtils.createImageIcon(CustomCheckBox.class, "/icons/stepcontrol/border.png");
        BufferedImage bufImgBorder = ImageUtils.toBufferedImage(borderIcon.getImage());
        BufferedImage bufImgSelected = ImageUtils.toBufferedImage(selectedIcon.getImage());
        final BufferedImage combinedImageSelected =
            new BufferedImage(bufImgBorder.getWidth(), bufImgBorder.getHeight(), BufferedImage.TYPE_INT_ARGB);
        Graphics2D g = combinedImageSelected.createGraphics();
        g.drawImage(bufImgBorder, 0, 0, null);
        g.drawImage(bufImgSelected, 0, 0, null);
        g.dispose();

        final ImageIcon selectedRolloverIcon = new ImageIcon(combinedImageSelected);
        return selectedRolloverIcon;
    }

    public AspectEditorPanel createAspectEditorPanel(
        final AspectCallbackListener<StepControlAspect> aspectCallbackListener, final StepControlAspect aspect,
        final StepControlModel stepControlModel, EditorType editorType) {

        final AspectEditorPanel aspectEditorPanel =
            new AspectEditorPanel(aspectCallbackListener, aspect, selectedIcon, unselectedIcon, selectedRolloverIcon,
                unselectedRolloverIcon, stepControlModel, editorType);
        return aspectEditorPanel;
    }

    public void createComponentFactory(final StepControlModel stepControlModel) {

        setComponentFactory(new HierarchicalTableComponentFactory() {
            @Override
            public Component createChildComponent(HierarchicalTable table, Object value, int row) {

                if (value instanceof StepControlAspect) {

                    final StepControlAspect aspect = (StepControlAspect) value;

                    final AspectEditorPanel aspectEditorPanel =
                        createAspectEditorPanel(callbackListener, aspect, stepControlModel, EditorType.editorUpdate);

                    LOGGER.info("Created new AspectEditorPanel: {}", aspectEditorPanel);

                    return new HierarchicalPanel(aspectEditorPanel, BorderFactory.createEmptyBorder());
                }
                return null;
            }

            @Override
            public void destroyChildComponent(HierarchicalTable table, Component component, int row) {

                Component childComponent = component;
                LOGGER.info("Destroy current childComponent: {}", childComponent);

                AspectEditorPanel aspectEditorPanel = null;
                if (childComponent instanceof HierarchicalPanel) {
                    Component comp = ((HierarchicalPanel) childComponent).getComponent(0);
                    if (comp instanceof AspectEditorPanel) {
                        aspectEditorPanel = (AspectEditorPanel) comp;

                        if (aspectEditorPanel.isDirty()) {
                            LOGGER.warn("The current aspectEditorPanel is dirty: {}", aspectEditorPanel);
                        }

                        aspectEditorPanel.cleanup();
                    }
                }
            }
        });

    }

    @Override
    public void changeSelection(int rowIndex, int columnIndex, boolean toggle, boolean extend) {
        // Member is being edited and they've clicked on a DIFFERENT row (this
        // method gets called even when the selection isn't actually changing)

        int selectedRow = getSelectedRow();
        LOGGER
            .info("changeSelection, selectedRow: {}, rowIndex: {}, columnIndex: {}, toggle: {}, extend: {}",
                selectedRow, rowIndex, columnIndex, toggle, extend);

        // check if the editor panel is dirty
        boolean editModeIsActive = isAnyExpanded();

        if (editModeIsActive && selectedRow != rowIndex) {
            if (!checkAspectEditorIsDirty(selectedRow)) {
                LOGGER.info("Check if aspect editor is dirty failed. Do not change selection.");
                return;
            }
        }

        // make the selection change
        super.changeSelection(rowIndex, columnIndex, toggle, extend);
    }

    public boolean checkAspectEditorIsDirty(int selectedRow) {

        // we must use the actual row from the sorted model
        int actualRow = getSortableTableModel().getActualRowAt(selectedRow);

        Component childComponent = getChildComponentAt(actualRow);
        LOGGER.info("The selected row has been changed. Current childComponent: {}", childComponent);

        boolean isDirty = false;
        AspectEditorPanel aspectEditorPanel = null;
        if (childComponent instanceof HierarchicalPanel) {
            Component comp = ((HierarchicalPanel) childComponent).getComponent(0);
            if (comp instanceof AspectEditorPanel) {
                aspectEditorPanel = (AspectEditorPanel) comp;

                isDirty = aspectEditorPanel.isDirty();
            }
        }

        if (isDirty) {
            // User was editing, now they're trying to move away without saving

            String okButtonText = Resources.getString(AspectTable.class, "save");
            String discardButtonText = Resources.getString(AspectTable.class, "discard");
            String cancelButtonText = UIManager.getString("OptionPane.cancelButtonText");

            Object[] options = { okButtonText, discardButtonText, cancelButtonText };
            int n =
                JOptionPane
                    .showOptionDialog(this, Resources.getString(AspectTable.class, "save_changes.message"),
                        Resources.getString(AspectTable.class, "save_changes.title"), JOptionPane.YES_NO_CANCEL_OPTION,
                        JOptionPane.WARNING_MESSAGE, null, options, options[0]);

            try {
                if (n == JOptionPane.YES_OPTION) {

                    aspectEditorPanel.applyChanges();
                }
                else if (n == JOptionPane.NO_OPTION) {
                    callbackListener.discardChanges(aspectEditorPanel::flushBuffer);

                    aspectEditorPanel.cleanup();
                }
                else {
                    // Exit without passing call on to super
                    return false;
                }
            }
            catch (VetoChangeException ex) {
                LOGGER.warn("Save changes was vetoed. Do not change the selection.");
                return false;
            }
            catch (Exception ex) {
                LOGGER.warn("Save changes failed.", ex);
                return false;
                // TODO: handle exception
            }
        }
        else if (aspectEditorPanel != null) {
            aspectEditorPanel.cleanup();
        }

        return true;
    }

    public void scrollToTop() {
        if (getParent() instanceof JViewport) {
            JViewport scrollPane = (JViewport) getParent();
            scrollPane.setViewPosition(new Point(0, 0));
        }
    }

    @Override
    public void collapseRow(int param1) {
        // prevent collapse row
    }

    @Override
    public void collapseAllRows() {
        // prevent collapse all rows
    }
}
