package org.bidib.wizard.mvc.main.controller;

import java.awt.Point;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.AddressTypeEnum;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.EnrailmentDirectionEnum;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.core.LocoService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.PomProgrammerControllerFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.common.DialogRegistry;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.main.controller.listener.GlobalDetectorPanelListener;
import org.bidib.wizard.mvc.main.model.GlobalDetectorModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.GlobalDetectorPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.pom.controller.PomProgrammerController;
import org.bidib.wizard.mvc.pom.controller.listener.PomProgrammerControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import com.vlsolutions.swing.docking.DockingDesktop;

public class GlobalDetectorPanelController implements GlobalDetectorPanelListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalDetectorPanelController.class);

    private final MainModel mainModel;

    private final GlobalDetectorModel globalDetectorModel;

    private final NodeListListener nodeListListener;

    private GlobalDetectorPanel globalDetectorPanel;

    private final CommandStationService commandStationService;

    private final LocoService locoService;

    @Autowired
    private ApplicationContext applicationContext;

    public GlobalDetectorPanelController(final MainModel mainModel, final CommandStationService commandStationService,
        final LocoService locoService) {
        this.mainModel = mainModel;
        this.commandStationService = commandStationService;
        this.locoService = locoService;

        globalDetectorModel = new GlobalDetectorModel();

        nodeListListener = new DefaultNodeListListener() {
            @Override
            public void nodeChanged(NodeInterface node) {
                LOGGER.debug("The node has changed.");

                NodeInterface selectedNode = mainModel.getSelectedNode();
                // update the selected node
                globalDetectorModel.setSelectedNode(selectedNode);

                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        LOGGER.debug("Notify that the selected node was changed.");
                        if (globalDetectorPanel != null) {
                            globalDetectorPanel.listChanged();
                        }
                        else {
                            LOGGER.warn("The globalDetectorPanel is not available.");
                        }
                    }
                });
            }
        };
        this.mainModel.addNodeListListener(nodeListListener);

    }

    @Override
    public NodeInterface getSelectedNode() {
        return globalDetectorModel.getSelectedNode();
    }

    public GlobalDetectorPanel createGlobalDetectorPanel(final TabVisibilityListener tabVisibilityListener) {
        LOGGER.debug("Create new GlobalDetectorPanel.");
        globalDetectorPanel = new GlobalDetectorPanel(this, globalDetectorModel, mainModel, tabVisibilityListener);

        return globalDetectorPanel;
    }

    @Override
    public void openLocoDialog(FeedbackAddressData addressData) {
        LOGGER.info("Open the loco controller dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node = findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            final DialogRegistry dialogRegistry = applicationContext.getBean(DialogRegistry.class);
            final LocoControllerFactory locoControllerFactory = applicationContext.getBean(LocoControllerFactory.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node,
                        (JFrame) JOptionPane.getFrameForComponent(globalDetectorPanel.getComponent()),
                        mainModel.getNodeProvider(), dialogRegistry);

            AddressData initialAddress = null;
            if (addressData != null) {
                final EnrailmentDirectionEnum enrailmentDirection = addressData.getType();
                AddressTypeEnum addressType = null;
                switch (enrailmentDirection) {
                    case LOCOMOTIVE_LEFT:
                    case LOCOMOTIVE_RIGHT:
                        addressType = AddressTypeEnum.LOCOMOTIVE_FORWARD;
                        break;
                    case BASIC_ACCESSORY:
                        addressType = AddressTypeEnum.ACCESSORY;
                        break;
                    case EXTENDED_ACCESSORY:
                        addressType = AddressTypeEnum.EXTENDED_ACCESSORY;
                        break;
                    default:
                        break;
                }
                initialAddress = new AddressData(addressData.getAddress(), addressType);
            }

            LocoListModel selectedLoco = null;
            SpeedSteps speedSteps = null;
            if (initialAddress != null) {
                int locoAddr = initialAddress.getAddress();
                final List<LocoListModel> locoList =
                    this.locoService.getLocoList(ConnectionRegistry.CONNECTION_ID_MAIN, node);

                selectedLoco = locoList.stream().filter(llm -> llm.getAddress() == locoAddr).findFirst().orElse(null);
                LOGGER.info("Found selected loco in locolist: {}", selectedLoco);

                speedSteps = selectedLoco.getSpeedSteps();
            }

            locoController.start(initialAddress, speedSteps, selectedLoco);
        }
    }

    @Override
    public void openPomDialog(FeedbackAddressData addressData) {
        LOGGER.info("Open the POM dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node = findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            final DialogRegistry dialogRegistry = applicationContext.getBean(DialogRegistry.class);
            final PomProgrammerControllerFactory pomProgrammerControllerFactory =
                applicationContext.getBean(PomProgrammerControllerFactory.class);
            final PomProgrammerController pomProgrammerController =
                pomProgrammerControllerFactory
                    .createPomProgrammerController(node, dialogRegistry,
                        (JFrame) JOptionPane.getFrameForComponent(globalDetectorPanel.getComponent()), new Point(0, 0));

            pomProgrammerController.addPomProgrammerControllerListener(new PomProgrammerControllerListener() {

                @Override
                public void sendRequest(
                    CommandStationNodeInterface node, PomAddressData locoAddress, CommandStationPom opCode,
                    int cvNumber, int cvValue) {
                    LOGGER.info("Send POM request.");

                    PomAcknowledge pomAck =
                        commandStationService
                            .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node, locoAddress, opCode,
                                cvNumber, cvValue);
                    LOGGER.info("Received pomAck: {}", pomAck);
                }

                @Override
                public void close() {
                }
            });

            AddressData initialAddress = null;
            if (addressData != null) {
                final EnrailmentDirectionEnum enrailmentDirection = addressData.getType();
                AddressTypeEnum addressType = null;
                switch (enrailmentDirection) {
                    case LOCOMOTIVE_LEFT:
                    case LOCOMOTIVE_RIGHT:
                        addressType = AddressTypeEnum.LOCOMOTIVE_FORWARD;
                        break;
                    case BASIC_ACCESSORY:
                        addressType = AddressTypeEnum.ACCESSORY;
                        break;
                    case EXTENDED_ACCESSORY:
                        addressType = AddressTypeEnum.EXTENDED_ACCESSORY;
                        break;
                    default:
                        break;
                }
                initialAddress = new AddressData(addressData.getAddress(), addressType);
            }

            final DockingDesktop desktop =
                DefaultApplicationContext
                    .getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);
            // open the dialog
            pomProgrammerController.start(desktop, initialAddress);
        }
    }

    private CommandStationNodeInterface findFirstCommandStationNode(Iterable<NodeInterface> nodes) {
        NodeInterface node = IterableUtils.find(nodes, new Predicate<NodeInterface>() {

            @Override
            public boolean evaluate(final NodeInterface node) {
                if (node.getCommandStationNode() != null && NodeUtils.hasCommandStationFunctions(node.getUniqueId())
                    && !(ProductUtils.isRFBasisNode(node.getUniqueId())
                        || ProductUtils.isSpeedometer(node.getUniqueId()))) {
                    LOGGER.debug("Found command station node: {}", node);
                    return true;
                }
                return false;
            }
        });
        return node != null ? node.getCommandStationNode() : null;
    }
}
