package org.bidib.wizard.mvc.stepcontrol.controller;

import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.mvc.stepcontrol.view.CvConsoleView;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

public class CvConsoleController {
    private static final Logger LOGGER = LoggerFactory.getLogger(CvConsoleController.class);

    private static String SEARCHKEY = "CvConsoleView";

    private CvConsoleView view;

    private DockableStateChangeListener dockableStateChangeListener;

    public void start(final DockingDesktop desktop) {

        // check if the console is already opened
        LOGGER.info("Search for view with key: {}", SEARCHKEY);
        Dockable consoleView = desktop.getContext().getDockableByKey(SEARCHKEY);
        if (consoleView != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return;
        }

        LOGGER.info("Create new CvConsoleView.");

        view = new CvConsoleView();

        // add the log panel next to the booster panel
        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState boosterTableView = null;
            // search the booster table view
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_BOOSTER_TABLE_VIEW.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the booster table view dockable.");
                    boosterTableView = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (boosterTableView != null) {
                LOGGER.info("Add the CV console view to the booster table view panel.");
                dock = boosterTableView.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new log panel at order: {}", order);

                desktop.createTab(dock, view, order, true);
            }
            else {
                desktop.split(dock, view, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(view, RelativeDockablePosition.RIGHT);
        }

        this.dockableStateChangeListener = new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(view) && event.getNewState().isClosed()) {
                    LOGGER.info("CvConsoleView was closed, free resources.");

                    try {
                        desktop.removeDockableStateChangeListener(dockableStateChangeListener);
                    }
                    catch (Exception ex) {
                        LOGGER
                            .warn("Remove dockableStateChangeListener from desktop failed: "
                                + dockableStateChangeListener, ex);
                    }
                    finally {
                        dockableStateChangeListener = null;
                    }

                    view.close();
                }
            }
        };
        desktop.addDockableStateChangeListener(this.dockableStateChangeListener);
    }

    public static synchronized void ensureConsoleVisible() {
        // check if the console is already opened
        LOGGER.info("Search for view with key: {}", SEARCHKEY);

        DockingDesktop desktop =
            DefaultApplicationContext.getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);

        Dockable consoleView = desktop.getContext().getDockableByKey(SEARCHKEY);
        if (consoleView != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return;
        }
        else {
            LOGGER.info("Create new controller to open the console.");
            new CvConsoleController().start(desktop);
        }
    }
}
