package org.bidib.wizard.mvc.stepcontrol.view;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.util.Map;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.WindowUtils;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.model.stepcontrol.TurnTableType;
import org.bidib.wizard.mvc.common.view.wizard.CancelAction;
import org.bidib.wizard.mvc.common.view.wizard.FinishAction;
import org.bidib.wizard.mvc.common.view.wizard.JWizardPanel;
import org.bidib.wizard.mvc.stepcontrol.model.AccelarationScaleEnum;
import org.bidib.wizard.mvc.stepcontrol.model.ConfigurationWizardModel;
import org.bidib.wizard.mvc.stepcontrol.model.ConfigurationWizardModel.WizardStatus;
import org.bidib.wizard.mvc.stepcontrol.model.Gearing;
import org.bidib.wizard.mvc.stepcontrol.model.MicroStepsEnum;
import org.bidib.wizard.mvc.stepcontrol.model.MotorSizeType;
import org.bidib.wizard.mvc.stepcontrol.model.MovementScaleEnum;
import org.bidib.wizard.mvc.stepcontrol.view.wizard.CustomLogoJWizardDialog;
import org.bidib.wizard.mvc.stepcontrol.view.wizard.StepMotorCharacteristicsPanel;
import org.bidib.wizard.mvc.stepcontrol.view.wizard.SummaryPanel;
import org.bidib.wizard.mvc.stepcontrol.view.wizard.TableTypePanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationWizard {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigurationWizard.class);

    private final ConfigurationWizardModel configurationWizardModel;

    public ConfigurationWizard(final ConfigurationWizardModel configurationWizardModel) {
        this.configurationWizardModel = configurationWizardModel;
    }

    public void showWizard(final Component parent) {
        Frame frame = JOptionPane.getFrameForComponent(parent);
        ImageIcon turntableLogo = ImageUtils.createImageIcon(getClass(), "/icons/stepcontrol/turntable.png");
        ImageIcon transferTableLogo = ImageUtils.createImageIcon(getClass(), "/icons/stepcontrol/transfertable.png");

        // create the modal wizard: the constructor takes the parent frame
        final CustomLogoJWizardDialog wizardDialog =
            new CustomLogoJWizardDialog(frame, new ImageIcon[] { transferTableLogo, turntableLogo }, true);
        wizardDialog.getWizardComponents().setFinishAction(new FinishAction(wizardDialog.getWizardComponents()) {

            @Override
            public void performAction() {
                configurationWizardModel.setWizardStatus(WizardStatus.finished);
                wizardDialog.dispose();
            }
        });
        wizardDialog.getWizardComponents().setCancelAction(new CancelAction(wizardDialog.getWizardComponents()) {

            @Override
            public void performAction() {
                configurationWizardModel.setWizardStatus(WizardStatus.aborted);
                wizardDialog.dispose();
            }
        });

        wizardDialog.getWizardComponents().getFinishButton().setText(Resources.getString(getClass(), "L_FinishButton"));
        wizardDialog
            .getWizardComponents().getFinishButton()
            .setMnemonic(Resources.getString(getClass(), "L_FinishButtonMnem").charAt(0));

        SwingUtilities.updateComponentTreeUI(wizardDialog);

        // set the dialog title
        wizardDialog.setTitle(Resources.getString(getClass(), "title"));

        // add panels to the wizard
        wizardDialog
            .getWizardComponents()
            .addWizardPanel(new TableTypePanel(wizardDialog.getWizardComponents(), configurationWizardModel));
        wizardDialog
            .getWizardComponents().addWizardPanel(
                new StepMotorCharacteristicsPanel(wizardDialog.getWizardComponents(), configurationWizardModel));
        wizardDialog
            .getWizardComponents()
            .addWizardPanel(new SummaryPanel(wizardDialog.getWizardComponents(), configurationWizardModel));

        // find max size
        Dimension prefDimension = null;
        for (JWizardPanel panel : wizardDialog.getWizardComponents().getWizardPanelList()) {
            if (prefDimension == null || prefDimension.getWidth() < panel.getPanel().getPreferredSize().getWidth()) {
                prefDimension = panel.getPanel().getPreferredSize();
            }
        }
        LOGGER.info("Current prefDimension: {}", prefDimension);
        wizardDialog.setSize(prefDimension.width + 300, 480);

        wizardDialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        WindowUtils.centerOnCurrentScreen(parent, wizardDialog);

        // show the wizard
        wizardDialog.setVisible(true);

        LOGGER.info("The wizard has finished, configurationWizardModel: {}", configurationWizardModel);
    }

    public ConfigurationWizardModel getConfigurationWizardModel() {
        return configurationWizardModel;
    }

    public static ConfigurationWizardModel prepareConfigurationModel(
        final NodeInterface selectedNode, TurnTableType turnTableType, final Map<String, CvNode> cvNumberToJideNodeMap,
        final Map<String, CvNode> mapKeywordToNode) {

        final ConfigurationWizardModel configurationWizardModel = new ConfigurationWizardModel(selectedNode);

        configurationWizardModel.setTurnTableType(turnTableType);

        Integer nemaType =
            StepControlCvUtils
                .getConfigVarByteValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_NEMATYPE);
        try {
            MotorSizeType motorSizeType = MotorSizeType.fromValue(ByteUtils.getLowByte(nemaType));
            configurationWizardModel.setMotorSizeType(motorSizeType);
        }
        catch (IllegalArgumentException ex) {
            LOGGER
                .warn("Invalid motor size in configuration. Fallback to NEMA13. Provided error message: {}",
                    ex.getMessage());
            configurationWizardModel.setMotorSizeType(MotorSizeType.nema13);
        }

        Integer stepCount =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_STEPCOUNT);
        configurationWizardModel.setStepCount(stepCount);

        Gearing gearing = null;
        Integer gearPrimary =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap,
                    StepControlKeywords.KEYWORD_GEAR_PRIMARY);
        Integer gearSecondary =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap,
                    StepControlKeywords.KEYWORD_GEAR_SECONDARY);
        if (gearPrimary == null || gearPrimary.intValue() < 1) {
            LOGGER.info("No gearing defined because primary value = 0.");
        }
        else if ((gearPrimary != null && gearPrimary.intValue() > 0)
            || (gearSecondary != null && gearSecondary.intValue() > 0)) {
            gearing = new Gearing(Gearing.YES);

            gearing.setGearRatioPrimary(gearPrimary != null ? gearPrimary : 1);
            gearing.setGearRatioSecondary(gearSecondary != null ? gearSecondary : 1);
            gearing.setBackLash(0);
        }

        Integer backlash =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_BACKLASH);
        if (backlash != null && backlash.intValue() > 0) {
            if (gearing != null) {
                LOGGER.info("Set the backlash value.");
                gearing.setBackLash(backlash);
            }
            else {
                LOGGER.info("No gearing available to set the backlash value.");
            }
        }

        if (gearing == null) {
            LOGGER.info("No gearing defined.");
            gearing = new Gearing(Gearing.NO);
        }
        configurationWizardModel.setGearing(gearing);

        // Integer microSteppingValue = StepControlCvUtils.getConfigVarByteValue(mapKeywordToNode,
        // cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_MICROSTEPPING);
        // MicroStepsEnum microStepsEnum = null;
        // if (microSteppingValue != null) {
        // try {
        // microStepsEnum = MicroStepsEnum.fromValue(microSteppingValue);
        // }
        // catch (Exception ex) {
        // LOGGER.warn("Parse microsteps value to supported enum failed.", ex);
        // }
        // }
        //
        // if (microStepsEnum == null) {
        // LOGGER.warn("Set the default microstepping value of 32!");
        // microStepsEnum = MicroStepsEnum.steps64;
        // }
        MicroStepsEnum microStepsEnum =
            StepControlCvUtils.getValidMicroStepping(mapKeywordToNode, cvNumberToJideNodeMap);
        configurationWizardModel.setMicroStepping(microStepsEnum);

        // unit system
        Integer unitSystemValue =
            StepControlCvUtils
                .getConfigVarByteValue(mapKeywordToNode, cvNumberToJideNodeMap,
                    StepControlKeywords.KEYWORD_UNIT_SYSTEM);
        if (unitSystemValue != null) {
            int bitSpeed = ByteUtils.getBit(unitSystemValue, 0);
            configurationWizardModel
                .setSpeedScale(bitSpeed > 0 ? MovementScaleEnum.scale0_1 : MovementScaleEnum.scale1);

            int bitAccelaration = ByteUtils.getBit(unitSystemValue, 4);
            configurationWizardModel
                .setAccelerationScale(
                    bitAccelaration > 0 ? AccelarationScaleEnum.scale0_1 : AccelarationScaleEnum.scale1);
        }
        else {
            configurationWizardModel.setSpeedScale(MovementScaleEnum.scale1);
            configurationWizardModel.setAccelerationScale(AccelarationScaleEnum.scale1);
        }

        // speed
        Integer speedValue =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_SPEED);
        if (speedValue != null) {
            configurationWizardModel.setSpeed(speedValue);
        }
        else {
            configurationWizardModel.setSpeed(null);
        }

        // accel
        Integer accelValue =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_ACCEL);
        if (accelValue != null) {
            configurationWizardModel.setAccel(accelValue);
        }
        else {
            configurationWizardModel.setAccel(null);
        }

        // decel
        Integer decelValue =
            StepControlCvUtils
                .getConfigVarIntValue(mapKeywordToNode, cvNumberToJideNodeMap, StepControlKeywords.KEYWORD_DECEL);
        if (decelValue != null) {
            configurationWizardModel.setDecel(decelValue);
        }
        else {
            configurationWizardModel.setDecel(null);
        }

        return configurationWizardModel;
    }

}
