package org.bidib.wizard.mvc.worklist.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.bidib.wizard.api.event.WorkListItemEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class WorkItemListModel extends Model {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkItemListModel.class);

    private static final long serialVersionUID = 1L;

    private ArrayListModel<WorkItemModel> workItemList = new ArrayListModel<>();

    public WorkItemListModel() {
    }

    public ArrayListModel<WorkItemModel> getWorkItemListModel() {
        return this.workItemList;
    }

    public List<WorkItemModel> getWorkItems() {
        return Collections.unmodifiableList(workItemList);
    }

    public boolean contains(final WorkListItemEvent item) {
        long count = 0;
        if (item.isDistinctItem()) {
            count =
                this.workItemList
                    .stream().filter(wi -> wi.getWorkListItemEvent().getComponent().equals(item.getComponent())
                        && wi.getWorkListItemEvent().getClass() == item.getClass())
                    .count();
        }
        else {
            // compare additional attributes
            count =
                this.workItemList
                    .stream()
                    .filter(wi -> wi.getWorkListItemEvent().getComponent().equals(item.getComponent())
                        && wi.getWorkListItemEvent().getClass() == item.getClass()
                        && item.compareAdditonalProperties(wi.getWorkListItemEvent()))
                    .count();
        }

        return count > 0;
    }

    public boolean addWorkListItem(final WorkListItemEvent item) {

        // check if an item exists already
        if (!contains(item)) {
            LOGGER.info("The new item is not in the work item list.");
            final WorkItemModel workItemModel = new WorkItemModel(item);
            this.addWorkListItem(workItemModel);

            return true;
        }
        else if (item.isRemoveDuplicates()) {
            // remove duplicates
            List<WorkItemModel> models = new ArrayList<>(this.workItemList);
            List<WorkItemModel> toRemove = new ArrayList<>();
            for (WorkItemModel workItemModel : models) {
                if (workItemModel.getWorkListItemEvent().getClass() == item.getClass()) {
                    LOGGER.info("Prepare remove duplicate: {}", workItemModel);
                    toRemove.add(workItemModel);
                }
            }

            boolean keepItem = false;
            if (toRemove.size() > 1) {
                // legacy processing: we remove all ... for the last time
                for (WorkItemModel workItemModel : toRemove) {
                    removeWorkListItem(workItemModel);
                }
            }
            else if (toRemove.size() == 1) {
                // check if the item to remove has the status 'done'
                WorkItemModel workItemModel = toRemove.get(0);
                if (workItemModel.getWorkListItemEvent().isIgnored(item)) {
                    // keep the item
                    keepItem = true;
                    LOGGER.info("Keep the existing workItem with state 'ignore'.");
                }
                else {
                    removeWorkListItem(workItemModel);
                }
            }

            if (!keepItem) {
                // add the new work item
                final WorkItemModel workItemModel = new WorkItemModel(item);
                this.addWorkListItem(workItemModel);
            }

            return !keepItem;
        }
        else {
            LOGGER.info("Skip adding duplicate item: {}", item);
        }

        return false;
    }

    public void addWorkListItem(final WorkItemModel workItemModel) {
        this.workItemList.add(workItemModel);
    }

    public void removeWorkListItem(final WorkItemModel workItemModel) {
        this.workItemList.remove(workItemModel);
    }

    public void addWorkListItems(final List<WorkListItemEvent> items) {
        for (WorkListItemEvent item : items) {
            this.addWorkListItem(item);
        }
    }

    public void clear() {
        this.workItemList.clear();
    }
}
