package org.bidib.wizard.mvc.main.controller;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.MapUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.IoBehaviourInputEnum;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.InputPortListener;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.common.labels.BidibLabelUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.mvc.main.model.InputPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.InputPortModelListener;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.InputPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class InputPortPanelController {

    private static final Logger LOGGER = LoggerFactory.getLogger(InputPortPanelController.class);

    private final MainModel mainModel;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    private InputPortListPanel inputPortListPanel;

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject = PublishSubject.create();

    public InputPortPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;
    }

    public InputPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final InputPortTableModel tableModel = new InputPortTableModel();

        tableModel.setPortListener(new InputPortModelListener() {

            @Override
            public void labelChanged(InputPort port, String label) {
                port.setLabel(label);

                try {
                    NodeLabels nodeLabels = getNodeLabels();

                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.inputPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save input port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void configChanged(InputPort port, PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port config value are changed for port: {}, portConfigKeys: {}", port, portConfigKeys);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                // we must get the configured port
                final List<InputPort> inputPorts = mainModel.getSelectedNode().getInputPorts();
                final InputPort inputPort = PortListUtils.findPortByPortNumber(inputPorts, port.getId());

                for (PortConfigKeys key : portConfigKeys) {

                    if (!inputPort.isPortConfigKeySupported(key)) {
                        LOGGER.info("Unsupported port config key detected: {}", key);
                        continue;
                    }

                    switch (key) {
                        case BIDIB_PCFG_INPUT_CTRL:
                            IoBehaviourInputEnum inputBehaviour = port.getInputBehaviour();
                            if (inputBehaviour != null && inputBehaviour != IoBehaviourInputEnum.UNKNOWN) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_INPUT_CTRL,
                                        new BytePortConfigValue(inputBehaviour.getType()));
                            }
                            break;
                        case BIDIB_PCFG_TICKS:
                            Integer switchOffTime = port.getSwitchOffTime();
                            if (switchOffTime != null) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_TICKS,
                                        new BytePortConfigValue(ByteUtils.getLowByte(switchOffTime)));
                            }
                            break;
                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the inputport parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }
            }

            @Override
            public void changePortType(LcOutputType portType, InputPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                final Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);
            }
        });

        this.inputPortListPanel =
            new InputPortListPanel(this, tableModel, mainModel, tabVisibilityListener, portConfigChangeEventSubject);

        // add an input port listener that handles some method calls and delegates labelChanged to the provided listener
        inputPortListPanel.setPortListener(new InputPortListener() {
            @Override
            public Class<?> getPortClass() {
                return InputPort.class;
            }

            @Override
            public void labelChanged(final InputPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                NodeLabels nodeLabels = getNodeLabels();

                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.inputPort, port.getId(), port.getLabel());
                saveLabels();

                inputPortListPanel.repaint();
            }

            @Override
            public void statusChanged(final NodeInterface node, final InputPort port) {
                LOGGER.info("The port status has changed: {}", node, port);
                // update port status
                SwingUtilities.invokeLater(() -> tableModel.notifyPortStatusChanged(port));
            }

            @Override
            public void configChanged(final NodeInterface node, final InputPort port) {
                // update port config
                SwingUtilities.invokeLater(() -> tableModel.notifyPortConfigChanged(port));
            }

            @Override
            public void changePortType(LcOutputType portType, InputPort port) {

            }

            @Override
            public void valuesChanged(InputPort port, PortConfigKeys... portConfigKeys) {

            }
        });

        return inputPortListPanel;
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(Long.valueOf(mainModel.getSelectedNode().getUniqueId()));
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            Long uniqueId = Long.valueOf(mainModel.getSelectedNode().getUniqueId());
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.inputPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.inputPortListPanel.refreshView());
        }
    }

}
