package org.bidib.wizard.mvc.main.view.table;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.IoBehaviourInputEnum;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.HintTextField;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardBindings;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.mvc.main.model.InputPortTableModel;
import org.bidib.wizard.mvc.main.view.panel.renderer.IoBehaviourInputCellRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.collect.ArrayListModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.Severity;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class InputPortEditorPanel extends AbstractPortEditorPanel<InputPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(InputPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private ValueModel switchOffTimeConverterModel;

    private ValueModel selectionHolderIoBehaviour;

    private JTextField portName;

    private HintTextField switchOffTimeText;

    private JComboBox<IoBehaviourInputEnum> comboIoBehaviour;

    public InputPortEditorPanel(InputPort port, Consumer<InputPort> saveCallback,
        final Consumer<InputPort> refreshCallback,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject,
        final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, null, refreshCallback, portConfigChangeEventSubject, nodeSelectionProvider);
    }

    @Override
    protected InputPort clonePort(final InputPort port) {
        // create a clone of the input port
        final InputPort clone =
            InputPort
                .builder() //
                .withInputBehaviour(port.getInputBehaviour()) //
                .withSwitchOffTime(port.getSwitchOffTime()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final InputPort port) {
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        final List<Component> order = new ArrayList<>();

        int row = 1;

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(InputPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(InputPortTableModel.class, "label") + ":").xy(1, row);
        this.portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        portName.setEnabled(port.isEnabled());
        dialogBuilder.add(this.portName).xyw(3, row, 3);

        order.add(this.portName);

        row += 2;

        // switchoff time
        this.switchOffTimeConverterModel =
            new ConverterValueModel(getPresentationModel().getBufferedModel(InputPort.PROPERTYNAME_SWITCHOFFTIME),
                new StringToIntegerConverter());

        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TICKS)) {
            this.switchOffTimeText = new HintTextField(() -> triggerValidation());
            final IntegerInputValidationDocument switchOffTimeDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            switchOffTimeDocument
                .setRangeValidationCallback(
                    new DefaultRangeValidationCallback(0, 255, showHint -> switchOffTimeText.showHint(showHint)));
            this.switchOffTimeText.setDocument(switchOffTimeDocument);
            this.switchOffTimeText.setColumns(2);
            this.switchOffTimeText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            WizardBindings.bind(this.switchOffTimeText, switchOffTimeConverterModel, true);
            this.switchOffTimeText.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(InputPortTableModel.class, "switchOffTime")).xy(1, row);
            dialogBuilder.add(this.switchOffTimeText).xy(3, row);

            order.add(this.switchOffTimeText);

            ValidationComponentUtils.setMandatory(this.switchOffTimeText, true);
            ValidationComponentUtils.setMessageKeys(this.switchOffTimeText, "validation.switchOffTime_key");

            row += 2;
        }

        // ioBehaviour
        this.selectionHolderIoBehaviour =
            getPresentationModel().getBufferedModel(InputPort.PROPERTYNAME_INPUTBEHAVIOUR);

        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_INPUT_CTRL)) {
            final ArrayListModel<IoBehaviourInputEnum> ioBehaviourList = new ArrayListModel<>();

            for (IoBehaviourInputEnum value : IoBehaviourInputEnum.values()) {
                ioBehaviourList.add(value);
            }

            final SelectionInList<IoBehaviourInputEnum> ioBehaviourSelection =
                new SelectionInList<>((ListModel<IoBehaviourInputEnum>) ioBehaviourList);

            final ComboBoxAdapter<IoBehaviourInputEnum> comboBoxAdapterIoBehaviour =
                new ComboBoxAdapter<IoBehaviourInputEnum>(ioBehaviourSelection, selectionHolderIoBehaviour);
            this.comboIoBehaviour = new JComboBox<>();
            this.comboIoBehaviour.setModel(comboBoxAdapterIoBehaviour);
            this.comboIoBehaviour.setRenderer(new IoBehaviourInputCellRenderer());
            this.comboIoBehaviour.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(InputPortTableModel.class, "ioBehaviour") + ":").xy(1, row);
            dialogBuilder.add(this.comboIoBehaviour).xy(3, row);

            order.add(this.comboIoBehaviour);

            row += 2;
        }

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 5);
        addButtons(buttonPanel, order);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        this.switchOffTimeConverterModel.addValueChangeListener(evt -> triggerValidation());
        this.selectionHolderIoBehaviour.addValueChangeListener(evt -> triggerValidation());

        enableComponents();

        triggerValidation();

        getPanel().setFocusCycleRoot(true);
        getPanel().setFocusTraversalPolicy(createFocusTransversalPolicy(order));

        return getPanel();
    }

    @Override
    public void requestDefaultFocus() {
        this.portName.requestFocusInWindow(); // default focus
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        if (SwingUtilities.isEventDispatchThread()) {
            processPropertyChanged(evt);
        }
        else {
            SwingUtilities.invokeLater(() -> processPropertyChanged(evt));
        }
    }

    private void processPropertyChanged(final PropertyChangeEvent evt) {

        try {
            // the status of the port was changed
            switch (evt.getPropertyName()) {
                case InputPort.PROPERTY_STATUS:
                    LOGGER.info("The port status has changed: {}", evt.getNewValue());
                    break;
                case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                    break;
                case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                    LOGGER.info("The port type has changed: {}", evt.getNewValue());
                    enableComponents();
                    break;

                default:
                    break;
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Update the status failed.", ex);
        }
    }

    @Override
    protected ValidationResult validate(final InputPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (switchOffTimeText != null && this.switchOffTimeConverterModel.getValue() == null) {
            support.addError("switchOffTime_key", "not_empty_for_write");
        }
        else if (this.switchOffTimeText != null && this.switchOffTimeText.isShowHint()) {
            support.add(Severity.INFO, "switchOffTime_key", "validRange;0..255");
        }

        if (this.comboIoBehaviour != null && this.selectionHolderIoBehaviour.getValue() == null) {
            support.addError("ioBehaviour_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    @Override
    protected void doEnableComponents(final InputPort port) {

        boolean enabled = port.isEnabled();
        this.portName.setEnabled(enabled);

        if (this.comboIoBehaviour != null) {
            this.comboIoBehaviour
                .setEnabled(enabled && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_INPUT_CTRL));
        }

        if (switchOffTimeText != null) {
            switchOffTimeText.setEnabled(enabled && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_TICKS));
        }

        super.doEnableComponents(port);
    }

}
