package org.bidib.wizard.mvc.features.view.panel;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.WordUtils;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.bidib2.DocumentationType;
import org.bidib.jbidibc.core.schema.bidib2.FeatureCode;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.utils.ConversionUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.CellStyle;
import com.jidesoft.grid.StyleModel;

public class FeatureTableModel extends AbstractTableModel implements StyleModel {
    private static final Logger LOGGER = LoggerFactory.getLogger(FeatureTableModel.class);

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_FEATURE_NAME = 0;

    public static final int COLUMN_FEATURE_ID = 1;

    public static final int COLUMN_FEATURE_VALUE = 2;

    public static final int COLUMN_FEATURE_UNIT = 3;

    private List<FeatureCode> featureCodes;

    private static final String[] _COLUMN_NAMES_ =
        { Resources.getString(FeatureTableModel.class, "featureName"),
            Resources.getString(FeatureTableModel.class, "featureId"),
            Resources.getString(FeatureTableModel.class, "value"),
            Resources.getString(FeatureTableModel.class, "unit") };

    private List<Feature> features = new LinkedList<Feature>();

    private String lang;

    public FeatureTableModel() {
        // load the feature codes
        featureCodes = BidibFactory.getFeatureCodes();

        lang = XmlLocaleUtils.getXmlLocale();
        LOGGER.info("Fetched the lang for feature description: {}", lang);
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        if (column == COLUMN_FEATURE_VALUE) {
            return true;
        }
        return false;
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        if (columnIndex == COLUMN_FEATURE_VALUE) {
            return Integer.class;
        }
        return super.getColumnClass(columnIndex);
    }

    public void updateRow(Feature feature) {

        // check if the feature is already displayed, if yes: update the row, otherwise: add new row
        if (features.contains(feature)) {
            LOGGER.debug("Feature is already in list and will be removed: {}", feature);
            features.remove(feature);
        }
        // create a copy of the feature to store in the feature list
        Feature copy = new Feature(feature.getType(), feature.getValue());
        features.add(copy);

        fireTableDataChanged();
    }

    @Override
    public int getColumnCount() {
        return _COLUMN_NAMES_.length;
    }

    @Override
    public String getColumnName(int column) {
        return _COLUMN_NAMES_[column];
    }

    @Override
    public int getRowCount() {
        return features.size();
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        Feature feature = features.get(rowIndex);
        switch (columnIndex) {
            case COLUMN_FEATURE_NAME:
                return getFeatureName(feature);
            case COLUMN_FEATURE_ID:
                return feature.getFeatureName();
            case COLUMN_FEATURE_UNIT:
                return getFeatureUnit(feature);
            default:
                return feature;
        }
    }

    private String getFeatureName(Feature feature) {

        FeatureCode featureCode = IterableUtils.find(featureCodes, new Predicate<FeatureCode>() {

            @Override
            public boolean evaluate(FeatureCode featureCode) {
                return featureCode.getId() == feature.getType();
            }
        });
        if (featureCode != null) {
            String featureName = getDocumentationText(featureCode.getDocumentation(), lang);
            if (StringUtils.isNotEmpty(featureName)) {
                return featureName;
            }
        }
        return feature.getFeatureName();
    }

    private String getFeatureDescription(Feature feature) {

        FeatureCode featureCode = IterableUtils.find(featureCodes, new Predicate<FeatureCode>() {

            @Override
            public boolean evaluate(FeatureCode featureCode) {
                return featureCode.getId() == feature.getType();
            }
        });
        if (featureCode != null) {
            String featureName = getDocumentationDescription(featureCode.getDocumentation(), lang);
            if (StringUtils.isNotEmpty(featureName)) {
                return featureName;
            }
        }
        return feature.getFeatureName();
    }

    private String getFeatureUnit(Feature feature) {

        if (feature.getFeatureEnum() != null) {
            switch (feature.getFeatureEnum()) {
                case FEATURE_BST_AMPERE:
                    // calculate the current value
                    return String.format("%d mA", ConversionUtils.convertCurrent(feature.getValue()));
                default:
                    break;
            }
        }

        FeatureCode featureCode = IterableUtils.find(featureCodes, new Predicate<FeatureCode>() {

            @Override
            public boolean evaluate(FeatureCode featureCode) {
                return featureCode.getId() == feature.getType();
            }
        });
        if (featureCode != null) {
            String featureUnit = featureCode.getUnit();
            if (StringUtils.isNotEmpty(featureUnit)) {
                return featureUnit;
            }
        }
        return null;
    }

    private String getDocumentationText(List<DocumentationType> descriptions, String lang) {

        if (descriptions != null) {
            for (DocumentationType description : descriptions) {
                if (lang.equals(description.getLanguage())) {
                    return description.getText();
                }
            }
        }
        return null;
    }

    private String getDocumentationDescription(List<DocumentationType> descriptions, String lang) {

        if (descriptions != null) {
            for (DocumentationType description : descriptions) {
                if (lang.equals(XmlLocaleUtils.stripMainLanguage(description.getLanguage()))) {
                    return description.getDescription();
                }
            }
        }
        return null;
    }

    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {
        Feature feature = features.get(rowIndex);
        switch (columnIndex) {
            case COLUMN_FEATURE_VALUE:
                setFeatureValue(feature, value);
                break;
            default:
                break;
        }
    }

    private void setFeatureValue(Feature feature, Object value) {
        if (value instanceof String) {
            try {
                feature.setValue(Integer.parseInt((String) value));
            }
            catch (Exception ex) {
                LOGGER.warn("Set the new feature value from string failed.", ex);
            }
        }
        else if (value instanceof Integer) {
            feature.setValue(((Integer) value).intValue());
        }
    }

    public Feature getRowAt(int rowIndex) {
        Feature feature = features.get(rowIndex);
        return feature;
    }

    public void clear() {
        features.clear();
        fireTableDataChanged();
    }

    public List<Feature> getFeatures() {
        return Collections.unmodifiableList(features);
    }

    private final CellStyle cellStyle = new CellStyle();

    @Override
    public CellStyle getCellStyleAt(int rowIndex, int columnIndex) {

        if (columnIndex < 2) {

            Feature feature = getRowAt(rowIndex);
            String toolTipText = getFeatureDescription(feature);
            toolTipText = WordUtils.wrap(toolTipText, 100, "<br/>", true);
            this.cellStyle.setToolTipText("<html>" + toolTipText + "</html>");
            return cellStyle;
        }
        return null;
    }

    @Override
    public boolean isCellStyleOn() {
        return true;
    }
}
