package org.bidib.wizard.mvc.main.model;

import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.MotorPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.MotorPortStatus;
import org.bidib.wizard.mvc.main.model.listener.MotorPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MotorPortTableModel
    extends SimpleHierarchicalPortTableModel<MotorPortStatus, MotorPort, MotorPortModelListener> {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(MotorPortTableModel.class);

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_VALUE = 1;

    public static final int COLUMN_STATUS = 2;

    public static final int COLUMN_TEST = 3;

    public static final int COLUMN_PORT_INSTANCE = 4;

    public MotorPortTableModel() {
        super();
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Integer getTestColumnIndex() {
        return COLUMN_TEST;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(MotorPortTableModel.class, "label"),
                Resources.getString(MotorPortTableModel.class, "value"),
                Resources.getString(MotorPortTableModel.class, "status"),
                Resources.getString(MotorPortTableModel.class, "test"), null };
    }

    @Override
    public void addRow(MotorPort motorPort) {
        if (motorPort != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = motorPort.getLabel();
            rowData[COLUMN_VALUE] = motorPort.getValue();
            rowData[COLUMN_STATUS] = motorPort.getStatus();
            MotorPortStatus oppositeStatus = PortUtils.getOppositeStatus(motorPort.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;

            rowData[COLUMN_PORT_INSTANCE] = motorPort;
            addRow(rowData);
        }
        else {
            LOGGER.warn("Performed addRow without servo port instance.");
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        MotorPort motorPort = (MotorPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_TEST:
                // the test can be changed.
                if (Port.getConfiguredPortType(motorPort) == LcOutputType.MOTORPORT) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            default:
                return Integer.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final MotorPort port = (MotorPort) getValueAt(row, COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_STATUS:
                port.setStatus((MotorPortStatus) value);
                super.setValueAt(value, row, column);
                break;
            case COLUMN_VALUE:
                // set the value for the target position
                int intValue = ((Integer) value);
                LOGGER.info("Set the value, new value: {}, port.value: {}", value, port.getValue());

                port.setValue(intValue);
                super.setValueAt(intValue, row, column);

                // set the value on the temporary port
                final MotorPort motorPort = new MotorPort();
                motorPort.setId(port.getId());
                motorPort.setValue(intValue);

                fireTestButtonPressed(motorPort);

                break;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        final MotorPort port = (MotorPort) super.getValueAt(row, MotorPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_STATUS:
                return port.getStatus();
            case COLUMN_VALUE:
                // return the value for the target position
                Integer portValue = port.getValue();
                LOGGER.trace("Get the port value: {}, row: {}", portValue, row);
                return portValue;
            default:
                column = COLUMN_PORT_INSTANCE;
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortStatusChanged(final MotorPort port) {
        LOGGER.info("Update the port status: {}", port);
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.info("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);
                MotorPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
                LOGGER.info("Set the opposite state: {}", oppositeStatus);
                super.setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(MotorPort port) {
        LOGGER.debug("The port config was changed for port: {}", port);

        // update the port status
        notifyPortStatusChanged(port);
    }

    @Override
    public void notifyPortLabelChanged(final MotorPort port) {
        LOGGER.info("The port label was changed for port: {}", port.getDebugString());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, MotorPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, MotorPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final MotorPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final MotorPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            MotorPort current = (MotorPort) getValueAt(row, MotorPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(final MotorPort port) {

        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(MotorPort port, PortConfigKeys... portConfigKeys) {
        LOGGER.info("The values of the port have changed: {}", port);

        portListener.configChanged(port, portConfigKeys);
    }

    @Override
    public void changePortType(LcOutputType portType, MotorPort port) {

        portListener.changePortType(portType, port);
    }

    @Override
    protected MotorPort getRow(int rowIndex) {
        MotorPort motorPort = (MotorPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return motorPort;
    }

    public void fireTestButtonPressed(final MotorPort port) {
        LOGGER.info("The value has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }

}
