package org.bidib.wizard.mvc.main.model;

import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.status.SoundPortStatus;
import org.bidib.wizard.mvc.main.model.listener.SoundPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SoundPortTableModel
    extends SimpleHierarchicalPortTableModel<SoundPortStatus, SoundPort, SoundPortModelListener> {
    private static final Logger LOGGER = LoggerFactory.getLogger(SoundPortTableModel.class);

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_PULSE_TIME = 1;

    public static final int COLUMN_PORT_IDENTIFIER = 2;

    public static final int COLUMN_STATUS = 3;

    public static final int COLUMN_TEST = 4;

    public static final int COLUMN_PORT_INSTANCE = 5;

    public SoundPortTableModel() {
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Integer getTestColumnIndex() {
        return COLUMN_TEST;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(getClass(), "label"), Resources.getString(getClass(), "pulseTime"),
                Resources.getString(getClass(), "portIdentifier"), Resources.getString(getClass(), "status"),
                Resources.getString(getClass(), "test"), null };
    }

    @Override
    public void addRow(SoundPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.getLabel();
            rowData[COLUMN_PULSE_TIME] = port.getPulseTime();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();

            SoundPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;
            rowData[COLUMN_PORT_INSTANCE] = port;
            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        SoundPort soundPort = (SoundPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_LABEL:
                isEditable = true;
                break;
            case COLUMN_PULSE_TIME:
                if (soundPort.isEnabled() && soundPort.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TICKS)) {
                    isEditable = true;
                }
                break;
            case COLUMN_STATUS:
                // the status can never be changed.
                isEditable = false;
            case COLUMN_TEST:
                if (soundPort.isEnabled()) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_PORT_INSTANCE:
            case COLUMN_PORT_IDENTIFIER:
                return Object.class;
            case COLUMN_STATUS:
                return Object.class;
            case COLUMN_TEST:
                return Object.class;

            default:
                return Object.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final Object o = getValueAt(row, COLUMN_PORT_INSTANCE);

        if (o instanceof SoundPort) {
            final SoundPort port = (SoundPort) o;

            switch (column) {
                case COLUMN_LABEL:
                    // if (value instanceof String) {
                    port.setLabel((String) value);
                    super.setValueAt(port.toString(), row, column);
                    fireLabelChanged(port, port.getLabel());
                    // }
                    // else {
                    // super.setValueAt(value, row, column);
                    // }
                    break;
                case COLUMN_PULSE_TIME:
                    int pulseTime = (Integer) value;
                    if (port.getPulseTime() != pulseTime) {
                        port.setPulseTime(pulseTime);
                        super.setValueAt(value, row, column);
                        fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TICKS);
                    }
                    else {
                        LOGGER.debug("The pulse time has not been changed.");
                    }
                    break;
                case COLUMN_STATUS:
                    port.setStatus((SoundPortStatus) value);
                    super.setValueAt(value, row, column);
                    break;
                case COLUMN_TEST:
                    SoundPortStatus portStatus = (SoundPortStatus) value;

                    LOGGER.info("Set the test port status: {}", portStatus);

                    super.setValueAt(portStatus, row, column);

                    // set the value on the temporary port
                    final SoundPort soundPort = new SoundPort();
                    soundPort.setId(port.getId());
                    soundPort.setStatus(portStatus);
                    fireTestButtonPressed(soundPort);

                    break;
                default:
                    super.setValueAt(value, row, column);
                    break;
            }
        }
        else {
            super.setValueAt(value, row, column);
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
            case COLUMN_PULSE_TIME:
            case COLUMN_TEST:
                column = COLUMN_PORT_INSTANCE;
                break;
            default:
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortStatusChanged(final SoundPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.debug("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);

                // get the opposite status and set it
                SoundPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());

                LOGGER.info("Set the port status, oppositeStatus: {}", oppositeStatus);

                setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(SoundPort port) {
        // TODO Auto-generated method stub

    }

    @Override
    public void notifyPortLabelChanged(final SoundPort port) {
        LOGGER.info("The port label was changed for port: {}", port);

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, SoundPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, SoundPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final SoundPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final SoundPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            SoundPort current = (SoundPort) getValueAt(row, SoundPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(final SoundPort port) {

        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(SoundPort port, PortConfigKeys... portConfigKeys) {

        LOGGER.info("The values of the port have changed: {}", port);

        portListener.configChanged(port, portConfigKeys);
    }

    @Override
    public void changePortType(LcOutputType portType, SoundPort port) {

        portListener.changePortType(portType, port);
    }

    @Override
    protected SoundPort getRow(int rowIndex) {
        SoundPort port = (SoundPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return port;
    }

    public void fireTestButtonPressed(SoundPort port) {
        LOGGER.info("Port status has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }
}
