package org.bidib.wizard.mvc.main.view.table;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.util.Optional;
import java.util.function.IntSupplier;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.TicksAware;

public class PortTicksRenderer extends DefaultTableCellRenderer {

    private static final long serialVersionUID = 1L;

    private JPanel comp;

    private JLabel value;

    private int portInstanceColumn;

    private final IntSupplier ticksTimeSupplier;

    public PortTicksRenderer(int portInstanceColumn) {
        this(portInstanceColumn, null);
    }

    public PortTicksRenderer(int portInstanceColumn, final IntSupplier ticksTimeSupplier) {
        this.portInstanceColumn = portInstanceColumn;
        this.ticksTimeSupplier = ticksTimeSupplier;

        comp = new JPanel();
        comp.setLayout(new BorderLayout());
        value = new JLabel();
        value.setMinimumSize(new Dimension(60, 12));
        value.setPreferredSize(new Dimension(70, 12));

        comp.add(value, BorderLayout.EAST);

    }

    @Override
    public Component getTableCellRendererComponent(
        JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        // The value must be a port. Return the port value from the tableModel instead of the field value.
        Port<?> port = (Port<?>) table.getModel().getValueAt(row, portInstanceColumn);

        // call super to set the correct color if selected
        Component superComp = super.getTableCellRendererComponent(table, null, isSelected, hasFocus, row, column);

        this.value.setText(null);
        // renderer only handles ports and only show a value if the port is TicksAware
        if (port instanceof Port<?>) {
            boolean enabled = port.isEnabled();
            setEnabled(enabled);
            this.value.setEnabled(enabled);
            setIcon(null);
            if (port instanceof TicksAware) {
                Integer ticks = ((TicksAware) port).getTicks();
                setText(ticks != null ? ticks.toString() : null);

                if (ticks != null) {
                    int factor = Optional.ofNullable(ticksTimeSupplier).orElse(() -> 1).getAsInt();
                    if (ticks.intValue() == 0) {
                        this.value.setText("(permanent)");
                    }
                    else {
                        this.value.setText(String.format("(%d ms)", ticks * 10 * factor));
                    }
                }
                else {
                    this.value.setText(null);
                }
            }
        }
        else {
            setEnabled(false);
            this.value.setEnabled(false);
            setIcon(null);
            setText(null);
        }

        ((JComponent) superComp).setOpaque(false);
        comp.add(superComp, BorderLayout.CENTER);

        if (isSelected) {
            comp.setBackground(table.getSelectionBackground());
            comp.setForeground(table.getSelectionForeground());
            this.value.setForeground(table.getSelectionForeground());
        }
        else {
            comp.setBackground(table.getBackground());
            comp.setForeground(table.getForeground());
            this.value.setForeground(table.getForeground());
        }
        return comp;
    }

}
