package org.bidib.wizard.mvc.stepcontrol.view.excel;

import java.awt.Color;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Row.MissingCellPolicy;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.utils.NodeUtils;
import org.bidib.wizard.model.stepcontrol.TurnTableType;
import org.bidib.wizard.mvc.stepcontrol.view.converter.AngleDegreesConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExcelAspectReader {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExcelAspectReader.class);

    private static final Color EXCEL_COLOR_GREEN = new Color(0x00, 0xB0, 0x50);

    private static final Color EXCEL_COLOR_RED = new Color(0xFF, 0x00, 0x00);

    public static class MotorData {
        private final int motorSteps;

        private final int microSteps;

        private final int totalSteps;

        private TurnTableType turnTableType;

        public MotorData(int motorSteps, int microSteps, int totalSteps, TurnTableType turnTableType) {
            this.motorSteps = motorSteps;
            this.microSteps = microSteps;
            this.totalSteps = totalSteps;
            this.turnTableType = turnTableType;
        }

        public int getMotorSteps() {
            return motorSteps;
        }

        public int getMicroSteps() {
            return microSteps;
        }

        public int getTotalSteps() {
            return totalSteps;
        }

        public TurnTableType getTurnTableType() {
            return turnTableType;
        }
    }

    public List<ImportAspect> readAspects(final InputStream excelFile) {

        final List<ImportAspect> importAspects = new LinkedList<>();
        Workbook wb = null;
        try {
            // create the workbook
            wb = new XSSFWorkbook(excelFile);

            final Sheet firstSheet = wb.getSheetAt(0);

            int rowNum = firstSheet.getFirstRowNum();
            LOGGER.info("First row num: {}", rowNum);

            // search the row with text 'Gleisanschluss'
            for (; rowNum < 20; rowNum++) {
                Row row = firstSheet.getRow(rowNum);
                if (row == null) {
                    LOGGER.info("Emtpy row: {}", rowNum);
                    continue;
                }
                int firstCellNum = row.getFirstCellNum();

                if (firstCellNum == -1) {
                    LOGGER.info("Emtpy row: {}", rowNum);
                    continue;
                }

                // LOGGER
                // .debug("Column width: {} : {} : {}", firstSheet.getColumnWidth(0), firstSheet.getColumnWidth(1),
                // firstSheet.getColumnWidth(2));

                Cell firstCell = row.getCell(firstCellNum, MissingCellPolicy.RETURN_NULL_AND_BLANK);
                if (firstCell != null && "Gleisanschluss".equals(firstCell.getStringCellValue())) {
                    LOGGER.info("Found 'Gleisanschluss' in row: {}", rowNum);
                    rowNum++;
                    parseAspects(importAspects, firstSheet, rowNum, firstCellNum);
                    break;
                }
            }

        }
        catch (IOException ex) {
            LOGGER.warn("Import aspects from Excel file failed.", ex);

            throw new DataExchangeException("Import aspects from Excel file failed.", ex);
        }
        finally {
            if (wb != null) {
                try {
                    wb.close();
                }
                catch (Exception ex) {
                    LOGGER.warn("Close Excel workbook failed.", ex);
                }
            }
        }
        return importAspects;
    }

    public void writeAspects(
        final NodeInterface node, final File excelFile, List<ImportAspect> aspects, final MotorData motorData) {

        if (excelFile.exists()) {
            try {
                excelFile.renameTo(new File(excelFile.getPath() + ".bak"));
            }
            catch (Exception ex) {
                LOGGER.warn("Create backup of existing file failed.", ex);
                throw new DataExchangeException("Create backup of Excel file failed.", ex);
            }
        }

        LOGGER.info("Write workbook to: {}", excelFile);

        // create the workbook
        try (OutputStream fileOut = new FileOutputStream(excelFile); Workbook wb = new XSSFWorkbook()) {

            String sheetName = NodeUtils.getNodeName(node);

            CellStyle h1Style = prepareBoldCellStyle(wb);

            LOGGER.info("Create summary sheet: {}", sheetName);
            final Sheet sheetSummary = wb.createSheet(sheetName);

            short rowNumber = 0;
            // Create a row and put some cells in it. Rows are 0 based.
            Row row = sheetSummary.createRow(rowNumber);
            // Create a cell and put a value in it.
            Cell cell = row.createCell(0);
            cell.setCellValue(sheetName);
            cell.setCellStyle(h1Style);
            h1Style.setAlignment(HorizontalAlignment.CENTER);
            sheetSummary.addMergedRegion(new CellRangeAddress(rowNumber, rowNumber, 0, 3));

            rowNumber++;

            final CellStyle defaultStyle = prepareDefaultCellStyle(wb);
            defaultStyle.setAlignment(HorizontalAlignment.RIGHT);

            row = addMotorSpecRow(sheetSummary, rowNumber++, defaultStyle, "Motorschritte", motorData.getMotorSteps());
            row = addMotorSpecRow(sheetSummary, rowNumber++, defaultStyle, "Mikroschritte", motorData.getMicroSteps());
            row = addMotorSpecRow(sheetSummary, rowNumber++, defaultStyle, "Gesamtschritte", motorData.getTotalSteps());

            rowNumber += 5;

            CellStyle boldStyle = prepareBoldCellStyle(wb);

            final TurnTableType turnTableType = motorData.getTurnTableType();
            if (turnTableType == TurnTableType.round) {
                addRow(sheetSummary, rowNumber++, new String[] { "Gleisanschluss", "Position", "Winkel" }, boldStyle);
            }
            else {
                addRow(sheetSummary, rowNumber++, new String[] { "Gleisanschluss", "Position" }, boldStyle);
            }

            for (ImportAspect aspect : aspects) {
                CellStyle defaultStyleAspects = prepareDefaultCellStyle(wb);

                addRow(wb, sheetSummary, rowNumber++, aspect, defaultStyleAspects, turnTableType,
                    Long.valueOf(motorData.getTotalSteps()));
            }

            sheetSummary.setColumnWidth(0, 3584);
            sheetSummary.setColumnWidth(1, 3986);
            sheetSummary.setColumnWidth(2, 3840);

            wb.write(fileOut);
        }
        catch (IOException ex) {
            LOGGER.warn("Export aspects to Excel file failed.", ex);

            throw new DataExchangeException("Export aspects to Excel file failed.", ex);
        }
        catch (Exception ex) {
            LOGGER.warn("Export aspects to Excel file failed.", ex);

            throw new DataExchangeException("Export aspects to Excel file failed.", ex);
        }

    }

    private void parseAspects(
        final List<ImportAspect> importAspects, final Sheet firstSheet, int rowNumStart, int firstCellNum) {

        for (int rowNum = rowNumStart; rowNum < rowNumStart + 48; rowNum++) {

            Row row = firstSheet.getRow(rowNum);
            if (row == null) {
                continue;
            }
            Cell indexCell = row.getCell(firstCellNum, MissingCellPolicy.RETURN_NULL_AND_BLANK);
            Cell positionCell = row.getCell(firstCellNum + 1, MissingCellPolicy.RETURN_NULL_AND_BLANK);

            LOGGER.debug("Current cell, index: {}, position: {}", indexCell, positionCell);

            double index = indexCell.getNumericCellValue();
            double position = positionCell.getNumericCellValue();

            int intIndex = (int) index;
            int roundedPos = (int) Math.round(position);

            LOGGER.info("Current cell values, index: {}, position: {}", intIndex, roundedPos);

            // get the color of the indexCell
            if (indexCell.getCellStyle() != null && indexCell.getCellStyle().getFillForegroundColorColor() != null) {

                XSSFCellStyle cellStyle = (XSSFCellStyle) indexCell.getCellStyle();
                XSSFColor bgColor = cellStyle.getFillForegroundXSSFColor();

                LOGGER.debug("Current bgColor: {}, val: {}", bgColor, ByteUtils.bytesToHex(bgColor.getRGB()));

                if ((EXCEL_COLOR_GREEN.getRGB() & 0xFFFFFF) == ByteUtils.getRGB(bgColor.getRGB())) {
                    LOGGER.info("Current color is green.");

                    importAspects.add(new ImportAspect(intIndex, roundedPos, false));
                }
                else if ((EXCEL_COLOR_RED.getRGB() & 0xFFFFFF) == ByteUtils.getRGB(bgColor.getRGB())) {
                    LOGGER.info("Current color is red.");

                    importAspects.add(new ImportAspect(intIndex, roundedPos, true));
                }
            }
        }
    }

    private CellStyle prepareH1CellStyle(final Workbook wb) {
        // Create a new font and alter it.
        Font font = wb.createFont();
        font.setFontHeightInPoints((short) 24);
        // font.setFontName("Courier New");
        font.setItalic(true);
        font.setBold(true);
        // font.setStrikeout(true);
        font.setColor(IndexedColors.BLUE_GREY.getIndex());

        // Fonts are set into a style so create a new one to use.
        CellStyle style = wb.createCellStyle();
        style.setFont(font);
        return style;
    }

    private CellStyle prepareBoldCellStyle(final Workbook wb) {
        // Create a new font and alter it.
        Font font = wb.createFont();
        font.setBold(true);

        // Fonts are set into a style so create a new one to use.
        CellStyle style = wb.createCellStyle();
        style.setFont(font);
        return style;
    }

    private CellStyle prepareDefaultCellStyle(final Workbook wb) {
        // Fonts are set into a style so create a new one to use.
        CellStyle style = wb.createCellStyle();
        return style;
    }

    // private void addRow(final Sheet sheet, int rowNumber, String[] cellValues) {
    // addRow(sheet, rowNumber, cellValues, (CellStyle) null);
    // }
    //
    // private void addRow(final Workbook wb, final Sheet sheet, int rowNumber, final ImportAspect aspect) {
    // addRow(wb, sheet, rowNumber, aspect, (CellStyle) null, null, null);
    // }

    private void addRow(final Sheet sheet, int rowNumber, String[] cellValues, CellStyle cellStyle) {
        Row row = sheet.createRow(rowNumber);
        // int column = 0;
        if (cellValues != null) {
            for (int column = 0; column < cellValues.length; column++) {
                String cellValue = cellValues[column];
                Cell cell = row.createCell(column);
                cell.setCellValue(cellValue);
                if (cellStyle != null) {
                    cell.setCellStyle(cellStyle);
                }
            }
        }
    }

    private void addRow(
        final Workbook wb, final Sheet sheet, int rowNumber, final ImportAspect aspect, final CellStyle cellStyle,
        final TurnTableType turnTableType, final Long totalSteps) {
        Row row = sheet.createRow(rowNumber);
        // int column = 0;

        double[] cellValues;
        if (TurnTableType.round == turnTableType) {
            // add the angle
            double angleValue = AngleDegreesConverter.getPositionAsAngle(aspect.getPosition(), totalSteps);
            cellValues = new double[] { aspect.getIndex(), aspect.getPosition(), angleValue };
        }
        else {
            cellValues = new double[] { aspect.getIndex(), aspect.getPosition() };
        }

        CellStyle defaultStyleAspects = prepareDefaultCellStyle(wb);

        cellStyle
            .setFillForegroundColor(
                aspect.isInverse() ? new XSSFColor(EXCEL_COLOR_RED, null) : new XSSFColor(EXCEL_COLOR_GREEN, null));
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);

        if (cellValues != null) {
            final DecimalFormat df = new DecimalFormat("#,###.#", DecimalFormatSymbols.getInstance(Locale.ENGLISH));

            for (int column = 0; column < cellValues.length; column++) {
                double cellValue = cellValues[column];

                Cell cell = row.createCell(column);
                cell.setCellValue(cellValue);
                if (column == 0 && cellStyle != null) {
                    cell.setCellStyle(cellStyle);
                }
                else if (column == 2) {
                    DataFormat format = wb.createDataFormat();
                    defaultStyleAspects.setDataFormat(format.getFormat("0.0"));

                    cell.setCellStyle(defaultStyleAspects);

                    cell.setCellValue(Double.valueOf(df.format(cellValue)));
                }
            }
        }
    }

    private Row addMotorSpecRow(final Sheet sheet, int rowNumber, CellStyle cellStyle, String title, int value) {

        Row row = sheet.createRow(rowNumber);

        Cell cell = row.createCell(0);
        cell.setCellValue(title);
        cell.setCellStyle(cellStyle);
        sheet.addMergedRegion(new CellRangeAddress(rowNumber, rowNumber, 0, 1));

        cell = row.createCell(2);
        cell.setCellValue(value);

        return row;
    }

}
