package org.bidib.wizard.mvc.ifnet.view;

import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.client.common.dialog.EscapeDialog;
import org.bidib.wizard.client.common.model.IpAddress;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.mvc.ifnet.model.IFnetConfigurationModel;
import org.bidib.wizard.mvc.ifnet.model.IFnetConfigurationModel.IpMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.PresentationModel;
import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.Trigger;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.base.Objects;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.field.IPTextField;

public class IFnetConfigurationDialog extends EscapeDialog {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(IFnetConfigurationDialog.class);

    private static final String ENCODED_COLUMN_SPECS = "pref, 3dlu, max(pref;100dlu)";

    private static final String ENCODED_ROW_SPECS = "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private final JButton cancelButton = new JButton(Resources.getString(IFnetConfigurationDialog.class, "cancel"));

    private final JButton applyButton = new JButton(Resources.getString(IFnetConfigurationDialog.class, "apply"));

    private final NodeService nodeService;

    private final NodeInterface selectedNode;

    private Trigger trigger;

    private PresentationModel<IFnetConfigurationModel> presentationModel;

    private ValueModel ipModeModel;

    public IFnetConfigurationDialog(final Frame frame, String title, final NodeService nodeService,
        final NodeInterface selectedNode) {
        super(frame, title, true);
        this.nodeService = nodeService;
        this.selectedNode = selectedNode;
    }

    private JPanel createPanel(final IFnetConfigurationModel iFnetConfigurationModel) {
        FormBuilder builder = null;
        boolean debug = false;
        if (debug) {
            builder =
                FormBuilder.create().columns(ENCODED_COLUMN_SPECS).rows(ENCODED_ROW_SPECS).panel(new FormDebugPanel());
        }
        else {
            builder = FormBuilder.create().columns(ENCODED_COLUMN_SPECS).rows(ENCODED_ROW_SPECS).panel(new JPanel());
        }
        builder.border(Paddings.TABBED_DIALOG);

        this.trigger = new Trigger();
        this.presentationModel = new PresentationModel<>(iFnetConfigurationModel, this.trigger);

        // TODO add content
        int row = 1;

        builder.add(Resources.getString(IFnetConfigurationDialog.class, "ipMode")).xy(1, row);
        // row += 2;

        final FormBuilder ipModeBuilder = FormBuilder.create().columns("pref, 3dlu, pref").rows("p, 3dlu, p");
        ipModeModel =
            new PropertyAdapter<IFnetConfigurationModel>(iFnetConfigurationModel,
                IFnetConfigurationModel.PROPERTYNAME_IP_MODE, true);

        int column = 1;
        for (int i = 0; i < 2; i++) {
            IpMode ipMode = null;
            switch (i) {
                case 0:
                    ipMode = IpMode.DHCP;
                    break;
                default:
                    ipMode = IpMode.STATIC;
                    break;
            }

            JRadioButton radio =
                WizardComponentFactory
                    .createRadioButton(ipModeModel, ipMode,
                        Resources.getString(IFnetConfigurationDialog.class, ipMode.getKey()));

            // add radio button
            ipModeBuilder.add(radio).xy(column, 1);
            column += 2;
        }

        builder.add(ipModeBuilder.build()).xy(3, row);

        row += 2;

        final BufferedValueModel staticAddressModel =
            presentationModel.getBufferedModel(IFnetConfigurationModel.PROPERTYNAME_STATIC_ADDRESS);

        final IPTextField staticAddress = WizardComponentFactory.createIPTextField(staticAddressModel, false);

        builder.add(Resources.getString(getClass(), "staticAddress") + ":").xy(1, row);
        builder.add(staticAddress).xyw(3, row, 1);

        staticAddress.setEnabled(false);

        row += 2;

        final BufferedValueModel subnetMaskModel =
            presentationModel.getBufferedModel(IFnetConfigurationModel.PROPERTYNAME_SUBNET_MASK);

        final IPTextField subnetMask = WizardComponentFactory.createIPTextField(subnetMaskModel, false);

        builder.add(Resources.getString(getClass(), "subnetMask") + ":").xy(1, row);
        builder.add(subnetMask).xyw(3, row, 1);

        subnetMask.setEnabled(false);

        row += 2;

        final BufferedValueModel gatewayModel =
            presentationModel.getBufferedModel(IFnetConfigurationModel.PROPERTYNAME_GATEWAY);

        final IPTextField gateway = WizardComponentFactory.createIPTextField(gatewayModel, false);

        builder.add(Resources.getString(getClass(), "gateway") + ":").xy(1, row);
        builder.add(gateway).xyw(3, row, 1);

        gateway.setEnabled(false);

        row += 2;

        // prepare the close button
        JPanel buttons = new ButtonBarBuilder().addGlue().addButton(applyButton, cancelButton).build();
        builder.add(buttons).xyw(1, row, 3);

        applyButton.setEnabled(false);

        final JPanel contentPanel = builder.build();

        // add property change listener for the ipMode
        this.ipModeModel.addValueChangeListener(pce -> {
            updateIpModeComponents(enabled -> {
                staticAddress.setEnabled(enabled);
                subnetMask.setEnabled(enabled);
                gateway.setEnabled(enabled);
            });
        });

        return contentPanel;
    }

    private void updateIpModeComponents(Consumer<Boolean> enableCallback) {
        LOGGER.info("The ipMode has been changed: {}", this.ipModeModel.getValue());
        IpMode ipMode = IpMode.DHCP;
        if (this.ipModeModel.getValue() instanceof IpMode) {
            ipMode = (IpMode) this.ipModeModel.getValue();
        }
        if (IpMode.STATIC == ipMode) {
            enableCallback.accept(true);
        }
        else {
            enableCallback.accept(false);
        }
    }

    /**
     * @return the presentationModel
     */
    protected PresentationModel<IFnetConfigurationModel> getPresentationModel() {
        return presentationModel;
    }

    private void close() {
        setVisible(false);

        dispose();
    }

    private static Integer findCvValue(final List<ConfigurationVariable> cvValues, String cvNumber) {
        Integer cvValue =
            cvValues
                .stream().filter(cv -> Objects.equals(cv.getName(), cvNumber)).findFirst()
                .map(cv -> Integer.valueOf(cv.getValue())).orElse(null);
        return cvValue;
    }

    public void showDialog() {

        final IFnetConfigurationModel iFnetConfigurationModel = new IFnetConfigurationModel();

        final List<ConfigurationVariable> configVariables = new ArrayList<>();
        configVariables.add(new ConfigurationVariable("3073", null));
        // static address
        configVariables.add(new ConfigurationVariable("3081", null));
        configVariables.add(new ConfigurationVariable("3082", null));
        configVariables.add(new ConfigurationVariable("3083", null));
        configVariables.add(new ConfigurationVariable("3084", null));
        // subnet mask address
        configVariables.add(new ConfigurationVariable("3085", null));
        configVariables.add(new ConfigurationVariable("3086", null));
        configVariables.add(new ConfigurationVariable("3087", null));
        configVariables.add(new ConfigurationVariable("3088", null));
        // gateway address
        configVariables.add(new ConfigurationVariable("3089", null));
        configVariables.add(new ConfigurationVariable("3090", null));
        configVariables.add(new ConfigurationVariable("3091", null));
        configVariables.add(new ConfigurationVariable("3092", null));

        // read the values from the node
        final List<ConfigurationVariable> cvValues =
            this.nodeService.queryConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, configVariables);

        // update the node model
        selectedNode.updateConfigVariableValues(cvValues, true);

        Integer ipModeValue = findCvValue(cvValues, "3073");
        if (ipModeValue != null && ipModeValue.intValue() == 1) {
            iFnetConfigurationModel.setIpMode(IpMode.STATIC);
        }
        else {
            iFnetConfigurationModel.setIpMode(IpMode.DHCP);
        }

        Integer staticIpByte1Value = findCvValue(cvValues, "3081");
        Integer staticIpByte2Value = findCvValue(cvValues, "3082");
        Integer staticIpByte3Value = findCvValue(cvValues, "3083");
        Integer staticIpByte4Value = findCvValue(cvValues, "3084");
        String staticAddress = "";
        if (staticIpByte1Value != null) {
            staticAddress += staticIpByte1Value.toString();
        }
        staticAddress += ".";
        if (staticIpByte2Value != null) {
            staticAddress += staticIpByte2Value.toString();
        }
        staticAddress += ".";
        if (staticIpByte3Value != null) {
            staticAddress += staticIpByte3Value.toString();
        }
        staticAddress += ".";
        if (staticIpByte4Value != null) {
            staticAddress += staticIpByte4Value.toString();
        }
        LOGGER.info("Prepared staticAddress: {}", staticAddress);
        iFnetConfigurationModel.setStaticAddress(staticAddress);

        // subnet mask
        Integer subnetMaskByte1Value = findCvValue(cvValues, "3085");
        Integer subnetMaskByte2Value = findCvValue(cvValues, "3086");
        Integer subnetMaskByte3Value = findCvValue(cvValues, "3087");
        Integer subnetMaskByte4Value = findCvValue(cvValues, "3088");
        String subnetMask = "";
        if (subnetMaskByte1Value != null) {
            subnetMask += subnetMaskByte1Value.toString();
        }
        subnetMask += ".";
        if (subnetMaskByte2Value != null) {
            subnetMask += subnetMaskByte2Value.toString();
        }
        subnetMask += ".";
        if (subnetMaskByte3Value != null) {
            subnetMask += subnetMaskByte3Value.toString();
        }
        subnetMask += ".";
        if (subnetMaskByte4Value != null) {
            subnetMask += subnetMaskByte4Value.toString();
        }
        LOGGER.info("Prepared subnetMask: {}", subnetMask);
        iFnetConfigurationModel.setSubnetMask(subnetMask);

        // gateway
        Integer gatewayByte1Value = findCvValue(cvValues, "3089");
        Integer gatewayByte2Value = findCvValue(cvValues, "3090");
        Integer gatewayByte3Value = findCvValue(cvValues, "3091");
        Integer gatewayByte4Value = findCvValue(cvValues, "3092");
        String gateway = "";
        if (gatewayByte1Value != null) {
            gateway += gatewayByte1Value.toString();
        }
        gateway += ".";
        if (gatewayByte2Value != null) {
            gateway += gatewayByte2Value.toString();
        }
        gateway += ".";
        if (gatewayByte3Value != null) {
            gateway += gatewayByte3Value.toString();
        }
        gateway += ".";
        if (gatewayByte4Value != null) {
            gateway += gatewayByte4Value.toString();
        }
        LOGGER.info("Prepared gateway: {}", gateway);
        iFnetConfigurationModel.setGateway(gateway);

        this.add(createPanel(iFnetConfigurationModel), BorderLayout.CENTER);

        final Consumer<IFnetConfigurationModel> saveConsumer = model -> writeCvValueToNode(model);
        applyButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                // update the model
                if (trigger != null) {
                    trigger.triggerCommit();
                }

                saveConsumer.accept(iFnetConfigurationModel);

                close();
            }
        });
        cancelButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                // close the dialog
                close();
            }
        });

        PropertyConnector.connect(this.presentationModel, PresentationModel.PROPERTY_CHANGED, applyButton, "enabled");

        pack();

        setLocationRelativeTo(getParent());
        setVisible(true);
    }

    private void writeCvValueToNode(final IFnetConfigurationModel model) {

        final List<ConfigurationVariable> configVariables = new ArrayList<>();
        configVariables.add(new ConfigurationVariable("3073", model.getIpMode().getValue()));

        // TODO add the static ip address, subnet mask and gateway
        IpAddress staticAddress = IpAddress.parseValue(model.getStaticAddress());
        configVariables.add(new ConfigurationVariable("3081", Integer.toString(staticAddress.getByte1())));
        configVariables.add(new ConfigurationVariable("3082", Integer.toString(staticAddress.getByte2())));
        configVariables.add(new ConfigurationVariable("3083", Integer.toString(staticAddress.getByte3())));
        configVariables.add(new ConfigurationVariable("3084", Integer.toString(staticAddress.getByte4())));

        IpAddress subnetMask = IpAddress.parseValue(model.getSubnetMask());
        configVariables.add(new ConfigurationVariable("3085", Integer.toString(subnetMask.getByte1())));
        configVariables.add(new ConfigurationVariable("3086", Integer.toString(subnetMask.getByte2())));
        configVariables.add(new ConfigurationVariable("3087", Integer.toString(subnetMask.getByte3())));
        configVariables.add(new ConfigurationVariable("3088", Integer.toString(subnetMask.getByte4())));

        IpAddress gateway = IpAddress.parseValue(model.getGateway());
        configVariables.add(new ConfigurationVariable("3089", Integer.toString(gateway.getByte1())));
        configVariables.add(new ConfigurationVariable("3090", Integer.toString(gateway.getByte2())));
        configVariables.add(new ConfigurationVariable("3091", Integer.toString(gateway.getByte3())));
        configVariables.add(new ConfigurationVariable("3092", Integer.toString(gateway.getByte4())));

        final List<ConfigurationVariable> cvValues =
            this.nodeService.setConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, configVariables);

        // update the node model
        selectedNode.updateConfigVariableValues(cvValues, true);
    }

}
