package org.bidib.wizard.mvc.main.view.table;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.mvc.main.model.InputPortTableModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class SoundPortEditorPanel extends AbstractPortEditorPanel<SoundPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(SoundPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private JTextField portName;

    public SoundPortEditorPanel(SoundPort port, Consumer<SoundPort> saveCallback,
        final Consumer<SoundPort> valueCallback, final Consumer<SoundPort> refreshCallback,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject,
        final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, portConfigChangeEventSubject, nodeSelectionProvider);
    }

    @Override
    protected SoundPort clonePort(final SoundPort port) {
        // create a clone of the sound port
        final SoundPort clone =
            SoundPort
                .builder() //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final SoundPort port) {
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        final List<Component> order = new ArrayList<>();

        int row = 1;

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(SoundPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(InputPortTableModel.class, "label") + ":").xy(1, 1);
        this.portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        portName.setEnabled(port.isEnabled());
        dialogBuilder.add(this.portName).xyw(3, row, 3);

        order.add(this.portName);

        row += 2;

        // add buttons
        final JPanel buttonPanel = getButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 5);
        addButtons(buttonPanel, order);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());

        enableComponents();

        triggerValidation();

        getPanel().setFocusCycleRoot(true);
        getPanel().setFocusTraversalPolicy(createFocusTransversalPolicy(order));

        return getPanel();
    }

    @Override
    public void requestDefaultFocus() {
        this.portName.requestFocusInWindow(); // default focus
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case SoundPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
                        LOGGER.info("The port status has changed: {}", evt.getNewValue());
                        // updatePortStatusFromOriginalPort();
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        enableComponents();
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    @Override
    protected ValidationResult validate(final SoundPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    @Override
    protected void doEnableComponents(final SoundPort port) {

        boolean enabled = port.isEnabled();
        this.portName.setEnabled(enabled);

        super.doEnableComponents(port);
    }

}
