package org.bidib.wizard.mvc.stepcontrol.view.converter;

import java.text.DecimalFormat;
import java.text.Format;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.common.base.Preconditions;

/**
 * Converts Values to Strings and vice-versa using a given Format.
 */
public final class AngleDegreesConverter implements BindingConverter<Number, String> {
    private static final Logger LOGGER = LoggerFactory.getLogger(AngleDegreesConverter.class);

    /**
     * Holds the {@code Format} used to format and parse.
     */
    private final Format format;

    private final long totalSteps;

    // Instance Creation **************************************************

    /**
     * Constructs a {@code StringConverter} on the given subject using the specified {@code Format}.
     * 
     * @param format
     *            the {@code Format} used to format and parse
     * @throws NullPointerException
     *             if the subject or the format is null.
     */
    public AngleDegreesConverter(Format format, long totalSteps) {
        this.format = Preconditions.checkNotNull(format, "The format must not be null.");
        this.totalSteps = totalSteps;
    }

    /**
     * Constructs a {@code StringConverter} on the given subject using the default DecimalFormat.
     */
    public AngleDegreesConverter(long totalSteps) {
        this(new DecimalFormat("0.0"), totalSteps);
    }

    // Implementing Abstract Behavior *************************************

    /**
     * Formats the source value and returns a String representation.
     * 
     * @param sourceValue
     *            the source value
     * @return the formatted sourceValue
     */
    @Override
    public String targetValue(Number sourceValue) {
        if (sourceValue != null) {
            long value = sourceValue.longValue();
            long lValue = (value) & 0xffffffffL;

            double angle = getPositionAsAngle(lValue, totalSteps);
            return format.format(angle);
        }

        return null;
    }

    /**
     * @return the position
     */
    public static double getPositionAsAngle(long position, long totalSteps) {
        return (position * 360.0f) / totalSteps;
    }

    /**
     * Parses the given String encoding and sets it as the subject's new value. Silently catches {@code ParseException}.
     * 
     * @param targetValue
     *            the value to be converted and set as new subject value
     */
    @Override
    public Number sourceValue(String targetValue) {
        try {
            if (StringUtils.isNotBlank(targetValue)) {
                // return format.parseObject((String) targetValue);
                return Long.valueOf(targetValue);
            }
        }
        catch (NumberFormatException e) {
            LOGGER.warn("Cannot parse the target value {}", targetValue);
        }
        return null;
    }
}
