package org.bidib.wizard.mvc.console.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.beans.IndexedPropertyChangeEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.UIManager;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleModel;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.client.common.component.ColorPane;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.mvc.console.model.DefaultConsoleModel;
import org.bidib.wizard.mvc.console.model.DefaultConsoleModel.ConsoleLine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vlsolutions.swing.docking.DockKey;
import com.vlsolutions.swing.docking.Dockable;

public class ConsoleView implements Dockable {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConsoleView.class);

    private final JPanel contentPanel;

    private final ColorPane coloredTextPane = new ColorPane();

    private final ConsoleModel consoleModel;

    private final SimpleDateFormat dateFormat = new SimpleDateFormat("HH:mm:ss.SSS");

    private final PropertyChangeListener pcsContent;

    private final PropertyChangeListener pcsContentSize;

    private Color colorRed;

    private Color colorBlue;

    private Color colorGreen;

    private Color colorBlack;

    private Color colorGray;

    public ConsoleView(final ConsoleService consoleService) {
        DockKeys.DOCKKEY_CONSOLE_VIEW.setName(Resources.getString(getClass(), "title"));
        DockKeys.DOCKKEY_CONSOLE_VIEW.setFloatEnabled(true);
        DockKeys.DOCKKEY_CONSOLE_VIEW.setAutoHideEnabled(false);

        LOGGER.info("Create new ConsoleView");
        contentPanel = new JPanel();
        contentPanel.setLayout(new BorderLayout());

        coloredTextPane.setEditable(false);

        JScrollPane scrollPane = new JScrollPane(coloredTextPane);
        contentPanel.add(scrollPane);

        this.colorRed = UIManager.getColor("Console.red");
        this.colorBlue = UIManager.getColor("Console.blue");
        this.colorGreen = UIManager.getColor("Console.green");
        this.colorBlack = UIManager.getColor("Console.black");
        this.colorGray = UIManager.getColor("Console.gray");

        consoleModel = consoleService.getConsoleModel();

        pcsContent = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.trace("Property was changed: {}", evt);
                if (evt instanceof IndexedPropertyChangeEvent) {
                    IndexedPropertyChangeEvent ipce = (IndexedPropertyChangeEvent) evt;

                    ConsoleLine newLine = (ConsoleLine) ipce.getNewValue();

                    if (newLine != null) {
                        String now = dateFormat.format(new Date());
                        coloredTextPane.append(getColor(newLine.getColor()), now + " - " + newLine.getMessage() + "\n");
                    }
                }
            }
        };

        consoleModel.addPropertyChangeListener(DefaultConsoleModel.PROPERTY_CONSOLE_CONTENT, pcsContent);

        pcsContentSize = new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.trace("Property was changed: {}", evt);

                if (evt.getNewValue() != null) {
                    Integer value = (Integer) evt.getNewValue();
                    if (value == 0) {
                        // clear the console
                        coloredTextPane.clear();
                    }
                }
            }
        };

        consoleModel.addPropertyChangeListener(DefaultConsoleModel.PROPERTY_CONSOLE_CONTENT_SIZE, pcsContentSize);

        JPopupMenu popupMenu = new BasicPopupMenu();
        JMenuItem clearConsole = new JMenuItem(Resources.getString(getClass(), "clear_console"));
        clearConsole.addActionListener(evt -> fireClearConsole());
        popupMenu.add(clearConsole);
        coloredTextPane.setComponentPopupMenu(popupMenu);
    }

    private Color getColor(ConsoleColor consoleColor) {
        Color color = this.colorBlack;
        switch (consoleColor) {
            case red:
                color = this.colorRed;
                break;
            case blue:
                color = this.colorBlue;
                break;
            case green:
                color = this.colorGreen;
                break;
            case gray:
                color = this.colorGray;
                break;
            default:
                break;
        }
        return color;
    }

    @Override
    public Component getComponent() {
        return contentPanel;
    }

    @Override
    public DockKey getDockKey() {
        return DockKeys.DOCKKEY_CONSOLE_VIEW;
    }

    public void close() {

        consoleModel.removePropertyChangeListener(DefaultConsoleModel.PROPERTY_CONSOLE_CONTENT, pcsContent);
        consoleModel.removePropertyChangeListener(DefaultConsoleModel.PROPERTY_CONSOLE_CONTENT_SIZE, pcsContentSize);
    }

    private void fireClearConsole() {
        LOGGER.info("clear the console.");

        consoleModel.clear();
        coloredTextPane.clear();
    }
}
