package org.bidib.wizard.mvc.debug.controller;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import javax.swing.JOptionPane;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.api.json.types.ConnectionPhase;
import org.bidib.jbidibc.messages.helpers.DefaultContext;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.CommPort;
import org.bidib.wizard.api.model.common.LineEnding;
import org.bidib.wizard.api.model.common.PreferencesPortType;
import org.bidib.wizard.api.model.common.PreferencesPortType.ConnectionPortType;
import org.bidib.wizard.api.model.debug.DebugConnection;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.exception.ConnectionException;
import org.bidib.wizard.common.model.settings.DebugConnectionConfiguration;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.logger.BidibLogsAppender;
import org.bidib.wizard.core.model.debug.DebugConnectionRegistry;
import org.bidib.wizard.core.model.settings.DebugToolsSettings;
import org.bidib.wizard.core.service.DebugConnectionService;
import org.bidib.wizard.mvc.debug.controller.listener.DebugInterfaceControllerListener;
import org.bidib.wizard.mvc.debug.model.DebugInterfaceModel;
import org.bidib.wizard.mvc.debug.view.DebugInterfaceView;
import org.bidib.wizard.mvc.debug.view.listener.DebugInterfaceViewListener;
import org.bidib.wizard.mvc.debug.view.listener.ProgressStatusCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.status.Status;
import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;

public class DebugInterfaceController implements DebugInterfaceControllerListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(DebugInterfaceController.class);

    private static final String APPENDER_NAME = DebugInterfaceController.class.getName();

    private final DockingDesktop desktop;

    private DockableStateChangeListener dockableStateChangeListener;

    private DebugInterfaceModel debugInterfaceModel;

    private DebugInterfaceView debugInterfaceView;

    // private DebugInterface debugReader;
    private DebugConnection connection;

    // private DebugMessageReceiver messageReceiver;

    // private DebugMessageListener messageListener;

    private List<byte[]> firmwareContent;

    private ScheduledExecutorService sendFileWorker;

    private final SettingsService settingsService;

    private final DebugConnectionService debugConnectionService;

    private CompositeDisposable compDisposable;

    private final Supplier<String> connectionId;

    /**
     * Create new instance of {@code DebugInterfaceController}.
     * 
     * @param desktop
     *            the docking desktop
     */
    public DebugInterfaceController(final DockingDesktop desktop, final SettingsService settingsService,
        final DebugConnectionService debugConnectionService) {
        this.desktop = desktop;
        this.settingsService = settingsService;
        this.debugConnectionService = debugConnectionService;

        this.connectionId = () -> DebugConnectionRegistry.CONNECTION_ID_DEBUG_MAIN;
    }

    /**
     * Extract the portName to access the serial ports.
     * 
     * @param connectionUri
     *            the connection uri
     * @return the port name to access the serial port
     */
    private String extractPortName(String connectionUri) {

        if (connectionUri != null && connectionUri.indexOf(" - ") > 0) {
            connectionUri = connectionUri.substring(0, connectionUri.indexOf(" - "));
        }
        return connectionUri;
    }

    public void start() {
        // check if the debug interface view is already opened
        String searchKey = DockKeys.DEBUG_INTERFACE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing debug interface view instead of open a new one.");
            DockUtils.selectWindow(view);
            return;
        }

        LOGGER.info("Create new DebugInterfaceView.");
        debugInterfaceModel = new DebugInterfaceModel();
        final List<DebugConnectionConfiguration> connectionConfigurations =
            settingsService.getDebugToolsSettings().getDebugConnectionConfigurations();
        if (connectionConfigurations != null && !connectionConfigurations.isEmpty()) {
            final DebugConnectionConfiguration config =
                connectionConfigurations
                    .stream().filter(cc -> this.connectionId.get().equals(cc.getId())).findFirst().orElse(null);
            if (config != null) {
                try {
                    debugInterfaceModel.setSelectedPort(new CommPort(extractPortName(config.getConnectionUri())));
                }
                catch (Exception ex) {
                    LOGGER
                        .warn("Extract com port identifier from connectionUri failed: {}", config.getConnectionUri(),
                            ex);
                }
            }
        }

        debugInterfaceView = new DebugInterfaceView(desktop, this, debugInterfaceModel, settingsService);

        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState tabPanelNodeDetails = null;
            // search the node details tab panel
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_TAB_PANEL.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the tab panel dockable.");
                    tabPanelNodeDetails = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (tabPanelNodeDetails != null) {
                LOGGER.info("Add the debug interface view next to the node details panel.");
                dock = tabPanelNodeDetails.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new debugInterfaceView at order: {}", order);

                desktop.createTab(dock, debugInterfaceView, order, true);
            }
            else {
                desktop.split(dock, debugInterfaceView, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(debugInterfaceView, RelativeDockablePosition.RIGHT);
        }

        debugInterfaceView.addDebugInterfaceViewListener(new DebugInterfaceViewListener() {

            @Override
            public void openConnection() {

                Integer baudRate = debugInterfaceModel.getBaudRate();

                if (baudRate == null) {
                    LOGGER.warn("No baudrate selected!");

                    JOptionPane
                        .showMessageDialog(debugInterfaceView.getComponent(),
                            Resources.getString(DebugInterfaceController.class, "select-baudrate"), "Debug Interface",
                            JOptionPane.ERROR_MESSAGE);
                    return;
                }
                // PreferencesPortType debugPort = Preferences.getInstance().getPreviousSelectedDebugPort();
                CommPort debugPort = debugInterfaceModel.getSelectedPort();
                if (debugPort == null || StringUtils.isBlank(debugPort.getName())) {
                    LOGGER.warn("No debugPort selected!");

                    JOptionPane
                        .showMessageDialog(debugInterfaceView.getComponent(),
                            Resources.getString(DebugInterfaceController.class, "select-port"), "Debug Interface",
                            JOptionPane.ERROR_MESSAGE);

                    return;
                }
                LOGGER.info("Current debugPort: {}", debugPort);

                if (compDisposable == null) {
                    compDisposable = new CompositeDisposable();
                }

                try {
                    Disposable dispConnectionStatusChanges =
                        debugConnectionService.subscribeConnectionStatusChanges(ci -> {

                            if (DebugInterfaceController.this.connectionId.get().equals(ci.getConnectionId())) {
                                LOGGER.info("The connection status has been changed: {}", ci);
                                if (ci.getConnectionState().isConnected()
                                    && ConnectionPhase.CONNECTED == ci.getConnectionState().getActualPhase()) {
                                    LOGGER.info("Set the status flag connected.");
                                    SwingUtils.executeInEDT(() -> debugInterfaceModel.setConnected(true));
                                }
                                else {
                                    LOGGER.info("Set the status flag disconnected.");
                                    SwingUtils.executeInEDT(() -> debugInterfaceModel.setConnected(false));
                                }
                            }

                        }, th -> {
                            LOGGER.warn("The connection status changes subject signalled an error.", th);
                        }, () -> {
                            LOGGER.info("The connection status changes subject has completed.");
                        });

                    compDisposable.add(dispConnectionStatusChanges);

                    DebugInterfaceController.this.connection =
                        debugConnectionService
                            .connect(DebugInterfaceController.this.connectionId.get(), null, baudRate,
                                new DefaultContext());

                    Disposable dispSubscribeDataEvents =
                        DebugInterfaceController.this.connection.subscribeDataEvents(dde -> {

                            LOGGER.info("debug message received size: {}", dde.getData());
                            SwingUtils.executeInEDT(() -> debugInterfaceView.addLog(new String(dde.getData())));

                        }, t -> {
                            LOGGER.warn("The data events subject signalled an error.", t);
                        }, () -> {
                            LOGGER.info("The data events subject has completed.");
                        });

                    compDisposable.add(dispSubscribeDataEvents);
                }
                catch (ConnectionException ex) {
                    LOGGER.warn("Open debug port failed.", ex);

                    JOptionPane
                        .showMessageDialog(debugInterfaceView.getComponent(),
                            "Open debug port failed. Reason: " + ex.getMessage(), "Debug Interface",
                            JOptionPane.ERROR_MESSAGE);
                }
                catch (Exception ex) {
                    LOGGER.warn("Open debug port failed.", ex);
                    JOptionPane
                        .showMessageDialog(debugInterfaceView.getComponent(),
                            "Open debug port failed. Reason: " + ex.getMessage(), "Debug Interface",
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void closeConnection() {
                LOGGER.info("Close the debug connection.");

                fireCloseConnection();
            }

            @Override
            public void transmit() {
                String sendText = debugInterfaceModel.getSendText();

                if (StringUtils.isEmpty(sendText)) {
                    LOGGER.info("No data to send!");
                    return;
                }

                LOGGER
                    .info("Send text to debugReader: {}, lineEnding: {}", sendText,
                        debugInterfaceModel.getLineEnding());

                if (DebugInterfaceController.this.connection != null) {
                    DebugInterfaceController.this.connection
                        .transmit(sendText, LineEnding.fromKey(debugInterfaceModel.getLineEnding().getKey()));
                }
            }

            @Override
            public void transmitFile(
                final AtomicBoolean continueTransmitHolder, final ProgressStatusCallback callback) {
                fireTransmitFile(continueTransmitHolder, callback);
            }
        });

        this.dockableStateChangeListener = new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(debugInterfaceView) && event.getNewState().isClosed()) {
                    LOGGER.info("LogPanelView was closed, free resources.");

                    try {
                        desktop.removeDockableStateChangeListener(dockableStateChangeListener);
                    }
                    catch (Exception ex) {
                        LOGGER
                            .warn("Remove dockableStateChangeListener from desktop failed: "
                                + dockableStateChangeListener, ex);
                    }
                    finally {
                        dockableStateChangeListener = null;
                    }

                    try {
                        // assume SLF4J is bound to logback in the current environment
                        LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();

                        ch.qos.logback.classic.Logger logTX = context.getLogger("TX");
                        ch.qos.logback.classic.Logger logRX = context.getLogger("RX");
                        Appender<ILoggingEvent> appender = logTX.getAppender(BidibLogsAppender.APPENDER_NAME);
                        LOGGER.info("Current BidibLogsAppender: {}", appender);

                        if (appender != null) {
                            LOGGER.info("Detach and stop the appender.");
                            logTX.detachAppender(appender);
                            logRX.detachAppender(appender);

                            appender.stop();
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.info("Remove BidibLogsAppender failed.");
                    }
                }
            }
        };
        desktop.addDockableStateChangeListener(this.dockableStateChangeListener);

        debugInterfaceModel
            .addPropertyChangeListener(DebugInterfaceModel.PROPERTY_LOG_TO_FILE, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    fireLogToFileChanged();
                }
            });

        debugInterfaceModel
            .addPropertyChangeListener(DebugInterfaceModel.PROPERTY_LOGFILE_NAME, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    fireLogToFileChanged();
                }
            });

        debugInterfaceModel.addPropertyChangeListener(DebugInterfaceModel.PROPERTY_SELECTED_PORT, evt -> {
            LOGGER.info("The selected port has been changed: {}", debugInterfaceModel.getSelectedPort());

            if (connectionConfigurations != null) {
                final DebugConnectionConfiguration config =
                    connectionConfigurations
                        .stream().filter(cc -> DebugInterfaceController.this.connectionId.get().equals(cc.getId()))
                        .findFirst().orElse(null);
                if (config != null) {
                    LOGGER.info("Update the existing config entry.");
                    final PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialPort.name());
                    config.setConnectionUri(debugInterfaceModel.getSelectedPort().getName());
                    config.setBaudRate(debugInterfaceModel.getBaudRate());
                    config.setConnectionProvider(portType.getConnectionPortType().name());
                }
                else {
                    LOGGER.info("Create new config entry.");
                    final PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialPort.name());
                    final DebugConnectionConfiguration newConfig =
                        new DebugConnectionConfiguration()
                            .withConnectionUri(debugInterfaceModel.getSelectedPort().getName())
                            .withBaudRate(debugInterfaceModel.getBaudRate())
                            .withConnectionProvider(portType.getConnectionPortType().name())
                            .withId(DebugInterfaceController.this.connectionId.get())
                            .withName(DebugInterfaceController.this.connectionId.get());
                    newConfig.setProviderKey(DebugToolsSettings.PROVIDER_KEY_DEBUG_TOOLS);
                    connectionConfigurations.add(newConfig);
                }
            }
        });
    }

    private void fireTransmitFile(final AtomicBoolean continueTransmit, final ProgressStatusCallback callback) {
        File sendFile = debugInterfaceModel.getSendFile();
        LOGGER.info("Send file content to debugReader: {}", sendFile);

        if (this.connection != null) {
            // prepare the data to send
            this.firmwareContent = null;

            InputStream input = null;
            List<byte[]> firmwareContent = new ArrayList<>();
            try {
                byte[] buffer = new byte[256];
                input = new FileInputStream(sendFile);

                int readBytes = IOUtils.read(input, buffer);
                while (readBytes > 0) {
                    LOGGER.info("Number of bytes read: {}", readBytes);
                    byte[] packet = Arrays.copyOf(buffer, readBytes);
                    firmwareContent.add(packet);

                    readBytes = IOUtils.read(input, buffer);
                }

                // keep the firmware content
                this.firmwareContent = firmwareContent;
            }
            catch (IOException ex) {
                LOGGER.info("No firmware content file found.");
            }
            finally {
                if (input != null) {
                    try {
                        input.close();
                    }
                    catch (Exception e) {
                        LOGGER.warn("Close input stream failed.", e);
                    }
                    input = null;
                }
            }

            if (CollectionUtils.isNotEmpty(this.firmwareContent)) {
                LOGGER.info("Send first packet to debug reader.");

                debugInterfaceModel.setTransferInProgress(true);

                sendFirmwarePackets(continueTransmit, callback);
            }
        }
    }

    private void sendFirmwarePackets(
        final AtomicBoolean continueTransmitHolder, final ProgressStatusCallback callback) {

        final Runnable runnable = () -> {
            LOGGER.info("Start sending firmware packets.");

            int packetsSent = 0;
            int totalPackets = firmwareContent.size();
            try {
                for (byte[] content : firmwareContent) {

                    if (!continueTransmitHolder.get()) {
                        LOGGER.warn("Transfer firmware was aborted by user.");
                        break;
                    }

                    this.connection.transmit(content);

                    packetsSent++;
                    if (callback != null) {
                        callback.statusChanged((packetsSent * 100) / totalPackets);
                    }
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Send firmware content to debug reader failed.", ex);
            }

            debugInterfaceModel.setTransferInProgress(false);

            if (callback != null) {
                callback.transferFinished();
            }
        };

        if (sendFileWorker == null) {
            sendFileWorker =
                Executors
                    .newScheduledThreadPool(1,
                        new ThreadFactoryBuilder().setNameFormat("sendFileWorkers-thread-%d").build());
        }
        // Start the send firmware files process
        sendFileWorker.execute(runnable);

    }

    private void fireCloseConnection() {
        if (this.connection != null) {
            LOGGER.info("Close the debug reader.");
            this.connection.disconnect();

            debugInterfaceModel.setConnected(false);

            if (compDisposable != null) {
                LOGGER.info("Dispose and release the compDisposable.");
                compDisposable.dispose();
                compDisposable = null;
            }

            // messageReceiver.removeMessageListener(messageListener);
            // messageListener = null;
            // messageReceiver = null;

            this.connection = null;
        }
        else {
            LOGGER.info("debug reader not available.");
        }
    }

    @Override
    public void viewClosed() {
        LOGGER.info("The view is closed.");

        fireCloseConnection();
    }

    protected void fireLogToFileChanged() {

        if (debugInterfaceModel.isLogToFile() && StringUtils.isNotBlank(debugInterfaceModel.getLogFileName())) {
            LOGGER.info("Add logger for debug interface: {}", debugInterfaceModel.getLogFileName());

            try {
                // assume SLF4J is bound to logback in the current environment
                LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();

                ch.qos.logback.classic.Logger logTX = context.getLogger(DebugInterfaceView.LOGGER_PANE_NAME);
                Appender<ILoggingEvent> appender = logTX.getAppender(APPENDER_NAME);

                if (appender == null) {

                    // clear all status messages
                    context.getStatusManager().clear();

                    PatternLayoutEncoder ple = new PatternLayoutEncoder();

                    ple.setPattern("%msg%n");
                    ple.setContext(context);
                    ple.start();

                    LOGGER.info("Create new appender.");
                    FileAppender<ILoggingEvent> fileAppender = new FileAppender<>();
                    fileAppender.setFile(debugInterfaceModel.getLogFileName());
                    fileAppender.setAppend(false);
                    fileAppender.setEncoder(ple);
                    fileAppender.setContext(context);
                    fileAppender.setName(APPENDER_NAME);
                    fileAppender.start();

                    logTX.setAdditive(false);
                    logTX.setLevel(Level.INFO);
                    logTX.addAppender(fileAppender);

                    List<Status> statusList = context.getStatusManager().getCopyOfStatusList();
                    if (CollectionUtils.isNotEmpty(statusList)) {
                        for (Status status : statusList) {
                            LOGGER.warn("Current status entry: {}", status);
                        }
                    }
                }
                else {
                    LOGGER.info("{} appender is already assigned.", APPENDER_NAME);
                }
            }
            catch (Exception ex) {
                LOGGER.info("Add debug interface logger failed.");
            }
        }
        else if (!debugInterfaceModel.isLogToFile() || StringUtils.isBlank(debugInterfaceModel.getLogFileName())) {
            LOGGER.info("Check if we must stop the logger.");
            try {
                // assume SLF4J is bound to logback in the current environment
                LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();

                // ch.qos.logback.classic.Logger log = context.getLogger("org.bidib");
                ch.qos.logback.classic.Logger logTX = context.getLogger(DebugInterfaceView.LOGGER_PANE_NAME);
                Appender<ILoggingEvent> appender = logTX.getAppender(APPENDER_NAME);
                LOGGER.info("Current debug interface appender: {}", appender);

                if (appender != null) {
                    // clear all status messages
                    context.getStatusManager().clear();

                    LOGGER.info("Detach and stop the appender.");
                    logTX.detachAppender(appender);

                    ((FileAppender<ILoggingEvent>) appender).stop();

                    // this is an ugly hack ... :-(
                    Map<String, String> map =
                        (Map<String, String>) context.getObject(CoreConstants.RFA_FILENAME_PATTERN_COLLISION_MAP);
                    map.remove(APPENDER_NAME);

                    List<Status> statusList = context.getStatusManager().getCopyOfStatusList();
                    if (CollectionUtils.isNotEmpty(statusList)) {
                        for (Status status : statusList) {
                            LOGGER.warn("Current status entry: {}", status);
                        }
                    }
                }
                else {
                    LOGGER.warn("Appender was not attached.");
                }
            }
            catch (Exception ex) {
                LOGGER.info("Remove debug interface logger failed.");
            }
        }
    }

}
