package org.bidib.wizard.mvc.worklist.controller.actions;

import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import javax.swing.JOptionPane;

import org.bidib.api.json.types.firmware.FirmwareVersionType.FirmwareRepoOriginType;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.event.FirmwareUpdateWorkListItemEvent;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.exception.UserActionAbortedException;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.firmwarerepo.core.NodeFirmwareRepoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import com.vlsolutions.swing.docking.DockingDesktop;

public class DownloadFirmwareAction implements WorkListAction<FirmwareUpdateWorkListItemEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(DownloadFirmwareAction.class);

    private final ApplicationContext applicationContext;

    private final StatusBar statusBar;

    private final ScheduledExecutorService firmwareDownloadWorker =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("firmwareDownloadWorkers-thread-%d").build());

    public DownloadFirmwareAction(final ApplicationContext applicationContext, final StatusBar statusBar) {
        this.applicationContext = applicationContext;
        this.statusBar = statusBar;
    }

    @Override
    public void apply(final DockingDesktop desktop, final FirmwareUpdateWorkListItemEvent actionEvent)
        throws UserActionAbortedException {

        LOGGER.info("Let the node firmware repo service download the new firmware.");

        int result =
            JOptionPane
                .showConfirmDialog(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                    Resources.getString(DownloadFirmwareAction.class, "download-new-firmware.text"),
                    Resources.getString(DownloadFirmwareAction.class, "download-new-firmware.title"),
                    JOptionPane.YES_OPTION | JOptionPane.CANCEL_OPTION);

        if (result == JOptionPane.YES_OPTION) {
            LOGGER
                .info("Download the new firmware to the local repo. Current repoOrigin: {}",
                    List.of(actionEvent.getRepoOrigin()));

            if (actionEvent.getRepoOrigin() != null && actionEvent.getRepoOrigin().length > 0) {
                for (FirmwareRepoOriginType repoOrigin : actionEvent.getRepoOrigin()) {
                    downloadFirmware(repoOrigin);
                }
            }
            else {
                // fallback to public repo
                downloadFirmware(FirmwareRepoOriginType.publicRepo);
            }
        }
        else {
            LOGGER.info("Download the new firmware to the local repo dialog was cancelled.");
            throw new UserActionAbortedException("Download the new firmware to the local repo dialog was cancelled.");
        }
    }

    private void downloadFirmware(final FirmwareRepoOriginType repoOrigin) {

        this.statusBar.setStatusText(Resources.getString(DownloadFirmwareAction.class, "download-firmware-start.text"));

        firmwareDownloadWorker.submit(() -> {

            try {
                final NodeFirmwareRepoService nodeFirmwareRepoService =
                    applicationContext.getBean(NodeFirmwareRepoService.class);
                nodeFirmwareRepoService.downloadNewRemoteFirmwareAvailable(repoOrigin, path -> {
                    // show progress in status bar
                    this.statusBar
                        .setProgress(Resources.getString(DownloadFirmwareAction.class, "download-successful", path),
                            true);
                });

                this.statusBar
                    .setStatusText(Resources.getString(DownloadFirmwareAction.class, "download-firmware-passed.text"));
            }
            catch (Exception ex) {
                LOGGER.warn("Download firmware has failed.", ex);
                this.statusBar
                    .setStatusText(Resources
                        .getString(DownloadFirmwareAction.class, "download-firmware-failed.text", ex.getMessage()));
            }

        });

    }

}
