package org.bidib.wizard.mvc.loco.view;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.WindowConstants;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.common.view.RegisteredDialog;
import org.bidib.wizard.mvc.common.view.ViewCloseListener;
import org.bidib.wizard.mvc.loco.model.LocoModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocoDialog extends JDialog implements RegisteredDialog {

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoDialog.class);

    private static final long serialVersionUID = 1L;

    private LocoView locoView;

    private final LocoModel locoModel;

    public LocoDialog(JFrame parent, final LocoModel locoModel, final SettingsService settingsService) {
        super(parent);
        this.locoModel = locoModel;

        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        if (locoModel.isCarControlEnabled()) {
            setTitle(Resources.getString(LocoView.class, "title.car"));
        }
        else {
            setTitle(Resources.getString(LocoView.class, "title.loco"));
        }

        addContentPanel(locoModel, settingsService);
        setLocationRelativeTo(parent);
    }

    public static String prepareKey(Integer address) {
        if (address != null) {
            return LocoDialog.class.getSimpleName() + "-" + address;
        }
        return LocoDialog.class.getSimpleName() + "-";
    }

    @Override
    public String getKey() {
        LOGGER.info("Prepare the key: {}", locoModel);
        Integer address = locoModel.getAddress();
        return prepareKey(address);
    }

    @Override
    public Window getWindow() {
        return this;
    }

    private void addContentPanel(final LocoModel locoModel, final SettingsService settingsService) {

        locoView = new LocoView(locoModel, settingsService);

        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                LOGGER.info("The window is closing: {}", e.getWindow());

                cleanup(settingsService);

                super.windowClosing(e);
            }

            @Override
            public void windowClosed(WindowEvent e) {
                LOGGER.info("The window is closed: {}", e.getWindow());

                super.windowClosed(e);
            }
        });

        getContentPane().setLayout(new BorderLayout());
        getContentPane().add(locoView.getComponent(), BorderLayout.CENTER);

        pack();

        setMinimumSize(new Dimension(locoView.getComponent().getPreferredSize().width + 20,
            locoView.getComponent().getPreferredSize().height + 20));

        setVisible(true);
    }

    public void addViewCloseListener(ViewCloseListener listener) {
        locoView.addViewCloseListener(listener);
    }

    public void close(final SettingsService settingsService) {
        LOGGER.info("The LocoView should be closed.");

        cleanup(settingsService);

        LOGGER.info("Cleanup has passed, set visible false.");
        setVisible(false);
    }

    private void cleanup(final SettingsService settingsService) {
        LOGGER.info("The LocoDialog is disposed.");

        if (locoView != null) {
            locoView.cleanup(settingsService);
        }
    }
}
