package org.bidib.wizard.mvc.stepcontrol.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.function.BiConsumer;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.converter.StringToUnsignedLongConverter;
import org.bidib.wizard.client.common.table.PortValidationResultModel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.common.exception.VetoChangeException;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlAspect;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlAspect.AspectPersistanceStatus;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlModel;
import org.bidib.wizard.mvc.stepcontrol.model.TurnTableType;
import org.bidib.wizard.mvc.stepcontrol.view.converter.AngleDegreesConverter;
import org.bidib.wizard.mvc.stepcontrol.view.converter.BooleanToPolarityConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.PresentationModel;
import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.Trigger;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.forms.layout.Sizes;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;
import com.jidesoft.plaf.UIDefaultsLookup;
import com.jidesoft.swing.NullButton;

public class AspectEditorPanel extends JPanel {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(AspectEditorPanel.class);

    public enum EditorType {
        editorNew, editorUpdate;
    }

    private final EditorType editorType;

    private final AspectCallbackListener<StepControlAspect> callbackListener;

    private final StepControlAspect stepControlAspect;

    private final StepControlAspect originalAspect;

    private final ImageIcon selectedIcon;

    private final ImageIcon unselectedIcon;

    private final ImageIcon selectedRolloverIcon;

    private final ImageIcon unselectedRolloverIcon;

    private final long totalSteps;

    private final TurnTableType turnTableType;

    private final boolean useSelectionBackground = false;

    private Trigger trigger;

    private PresentationModel<StepControlAspect> presentationModel;

    private ValueModel activateButtonEnabled = new ValueHolder();

    private ValueModel applyButtonEnabled = new ValueHolder();

    private PropertyChangeListener listener;

    private JTextField currentAngle;

    private JLabel angleLabel;

    private JTextField currentOppositeAngle;

    private JLabel oppositeAngleLabel;

    private NullButton oppositeActivateButton;

    private BufferedValueModel bufferedPositionModel;

    private ValueModel currentPositionConverterModel;

    private BufferedValueModel bufferedOppositePositionModel;

    private ValueModel oppositePositionConverterModel;

    private PortValidationResultModel validationResultModel;

    public AspectEditorPanel(final AspectCallbackListener<StepControlAspect> callbackListener,
        final StepControlAspect stepControlAspect, final ImageIcon selectedIcon, final ImageIcon unselectedIcon,
        final ImageIcon selectedRolloverIcon, final ImageIcon unselectedRolloverIcon,
        final StepControlModel stepControlModel, final EditorType editorType) {

        LOGGER.info("Create new editor, editorType: {}, stepControlAspect: {}", editorType, stepControlAspect);

        this.callbackListener = callbackListener;
        this.editorType = editorType;

        this.originalAspect = stepControlAspect;
        // create a copy of the aspect
        this.stepControlAspect =
            new StepControlAspect(stepControlAspect.getAspectNumber(), stepControlAspect.getPosition(),
                stepControlAspect.getPolarity(), stepControlAspect.getOppositePosition(),
                stepControlAspect.getOppositePolarity());
        this.stepControlAspect.setStatus(stepControlAspect.getStatus());

        this.selectedIcon = selectedIcon;
        this.unselectedIcon = unselectedIcon;

        this.selectedRolloverIcon = selectedRolloverIcon;
        this.unselectedRolloverIcon = unselectedRolloverIcon;

        this.totalSteps = stepControlModel.getTotalSteps() != null ? stepControlModel.getTotalSteps() : 1;
        this.turnTableType = stepControlModel.getTurnTableType();

        setLayout(new BorderLayout());

        add(createTextPanel(stepControlModel));
        add(createControlPanel(), BorderLayout.SOUTH);

        if (useSelectionBackground) {
            setBackground(UIDefaultsLookup.getColor("Table.selectionBackground"));
            setForeground(UIDefaultsLookup.getColor("Table.selectionForeground"));
        }

        // set the initial value
        triggerValidation();
    }

    private JComponent createTextPanel(final StepControlModel stepControlModel) {

        this.trigger = new Trigger();
        this.presentationModel = new PresentationModel<>(this.stepControlAspect, this.trigger);

        this.listener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateActivateButtonEnabled();
                updateApplyButtonEnabled();
            }
        };

        this.presentationModel.addPropertyChangeListener(PresentationModel.PROPERTY_BUFFERING, this.listener);
        this.originalAspect.addPropertyChangeListener(StepControlAspect.PROPERTYNAME_STATUS, this.listener);

        // add a validation model that can trigger a button state with the validState property
        this.validationResultModel = new PortValidationResultModel();

        // create the 'detail panel'
        boolean debug = false;
        FormBuilder detailFormBuilder =
            FormBuilder
                .create()
                .columns("p, 3dlu, max(30dlu;p), 3dlu, p, 3dlu, p, 6dlu, p, 3dlu, max(30dlu;p), 3dlu, p, 3dlu, p")
                .rows("p").panel(debug ? new FormDebugPanel() : new JPanel());

        detailFormBuilder.border(Paddings.DLU4);

        this.bufferedPositionModel = presentationModel.getBufferedModel(StepControlAspect.PROPERTYNAME_POSITION);

        this.currentPositionConverterModel =
            new ConverterValueModel(bufferedPositionModel, new StringToUnsignedLongConverter());

        JTextField currentPosition = new JTextField();
        InputValidationDocument currentPositionDocument =
            new InputValidationDocument(6, InputValidationDocument.NUMERIC);
        currentPosition.setDocument(currentPositionDocument);
        currentPosition.setColumns(6);

        // bind manually because we changed the document of the textfield
        Bindings.bind(currentPosition, currentPositionConverterModel, false);

        final JLabel positionLabel = new JLabel(Resources.getString(StepControlPanel.class, "position"));
        positionLabel.setToolTipText(Resources.getString(StepControlPanel.class, "position.tooltip"));
        detailFormBuilder.add(positionLabel).xy(1, 1);
        detailFormBuilder.add(currentPosition).xy(3, 1);
        if (useSelectionBackground) {
            clearAttribute(positionLabel);
        }

        ValidationComponentUtils.setMandatory(currentPosition, true);
        ValidationComponentUtils.setMessageKeys(currentPosition, "validation.currentPosition_key");

        bufferedPositionModel.addValueChangeListener(evt -> triggerValidation());

        // button use current position
        final ImageIcon wrenchIcon = ImageUtils.createImageIcon(AspectEditorPanel.class, "/icons/16x16/wrench.png");
        NullButton useCurrentPositionButton = new NullButton(wrenchIcon);
        useCurrentPositionButton.setToolTipText(Resources.getString(AspectEditorPanel.class, "use_current_position"));
        useCurrentPositionButton.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
        useCurrentPositionButton
            .addActionListener(new AbstractClickAction(
                Resources.getString(AspectEditorPanel.class, "use_current_position"), useCurrentPositionButton) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    LOGGER.info("Use the current position to set the aspect position.");

                    if (callbackListener != null) {
                        Long currentTurntablePosition = callbackListener.getCurrentTurntablePosition();

                        if (currentTurntablePosition != null && isRoundTurnTable(stepControlModel)) {
                            // check if the current position is >180 degrees position
                            LOGGER
                                .info(
                                    "The current position is greater than the 180 degrees position. Use the current position as opposite position: {}",
                                    currentTurntablePosition);

                            bufferedOppositePositionModel.setValue(currentTurntablePosition);

                            currentTurntablePosition =
                                currentTurntablePosition - (AspectEditorPanel.this.totalSteps / 2);
                            LOGGER
                                .info("Calculated new position for the current position: {}", currentTurntablePosition);
                        }

                        bufferedPositionModel.setValue(currentTurntablePosition);
                    }
                }
            });
        detailFormBuilder.add(useCurrentPositionButton).xy(5, 1);

        // angle
        final ValueModel currentAngleConverterModel =
            new ConverterValueModel(bufferedPositionModel, new AngleDegreesConverter(totalSteps));
        currentAngle = WizardComponentFactory.createTextField(currentAngleConverterModel);
        currentAngle.setEditable(false);
        angleLabel = new JLabel(Resources.getString(StepControlPanel.class, "angle"));
        detailFormBuilder.add(angleLabel).xy(9, 1);
        detailFormBuilder.add(currentAngle).xy(11, 1);
        if (useSelectionBackground) {
            clearAttribute(angleLabel);
        }

        // polarity
        final ValueModel polarityConverterModel =
            new ConverterValueModel(presentationModel.getBufferedModel(StepControlAspect.PROPERTYNAME_POLARITY),
                new BooleanToPolarityConverter());

        final JCheckBox currentPolarity = WizardComponentFactory.createCheckBox(polarityConverterModel, null);
        currentPolarity.setIcon(unselectedIcon);
        currentPolarity.setSelectedIcon(selectedIcon);
        currentPolarity.setRolloverIcon(unselectedRolloverIcon);
        currentPolarity.setRolloverSelectedIcon(selectedRolloverIcon);

        final JLabel polarityLabel = new JLabel(Resources.getString(StepControlPanel.class, "polarity"));
        detailFormBuilder.add(currentPolarity).xy(13, 1);
        if (useSelectionBackground) {
            clearAttribute(polarityLabel);
        }

        final NullButton activateButton = new NullButton(Resources.getString(AspectEditorPanel.class, "activate"));
        activateButton
            .addActionListener(
                new AbstractClickAction(Resources.getString(AspectEditorPanel.class, "activate"), activateButton) {
                    @Override
                    public void actionPerformed(ActionEvent e) {

                        if (callbackListener != null) {
                            callbackListener.activateAspect(AspectEditorPanel.this.originalAspect);
                        }
                    }
                });
        detailFormBuilder.add(activateButton).xyw(15, 1, 1);

        // check if round turntable because round turntable must have opposite position defined
        if (isRoundTurnTable(stepControlModel)) {

            detailFormBuilder.appendRows("3dlu, p");

            // opposite position
            this.bufferedOppositePositionModel =
                presentationModel.getBufferedModel(StepControlAspect.PROPERTYNAME_OPPOSITE_POSITION);

            this.oppositePositionConverterModel =
                new ConverterValueModel(bufferedOppositePositionModel, new StringToUnsignedLongConverter());

            JTextField oppositePosition = new JTextField();
            InputValidationDocument oppositePositionDocument =
                new InputValidationDocument(6, InputValidationDocument.NUMERIC);
            oppositePosition.setDocument(oppositePositionDocument);
            oppositePosition.setColumns(6);

            // bind manually because we changed the document of the textfield
            Bindings.bind(oppositePosition, oppositePositionConverterModel, false);

            final JLabel oppositePositionLabel =
                new JLabel(Resources.getString(StepControlPanel.class, "opposite_position"));
            oppositePositionLabel
                .setToolTipText(Resources.getString(StepControlPanel.class, "opposite_position.tooltip"));

            detailFormBuilder.add(oppositePositionLabel).xy(1, 3);
            detailFormBuilder.add(oppositePosition).xy(3, 3);
            if (useSelectionBackground) {
                clearAttribute(oppositePositionLabel);
            }

            ValidationComponentUtils.setMandatory(oppositePosition, true);
            ValidationComponentUtils.setMessageKeys(oppositePosition, "validation.oppositePosition_key");

            bufferedOppositePositionModel.addValueChangeListener(evt -> triggerValidation());

            NullButton useCurrentOppositePositionButton = new NullButton(wrenchIcon);
            useCurrentOppositePositionButton
                .setToolTipText(Resources.getString(AspectEditorPanel.class, "use_current_opposite_position"));
            useCurrentOppositePositionButton.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            useCurrentOppositePositionButton
                .addActionListener(new AbstractClickAction(
                    Resources.getString(AspectEditorPanel.class, "use_current_opposite_position"),
                    useCurrentOppositePositionButton) {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        LOGGER.info("Use the current position to set the aspect opposite position.");

                        if (callbackListener != null) {
                            Long currentTurntablePosition = callbackListener.getCurrentTurntablePosition();

                            bufferedOppositePositionModel.setValue(currentTurntablePosition);
                        }
                    }
                });
            detailFormBuilder.add(useCurrentOppositePositionButton).xy(5, 3);

            final ImageIcon calculatorIcon =
                ImageUtils.createImageIcon(AspectEditorPanel.class, "/icons/16x16/calculator.png");
            NullButton calculateOppositePositionButton = new NullButton(calculatorIcon);
            calculateOppositePositionButton
                .setToolTipText(Resources.getString(AspectEditorPanel.class, "calculate_opposite_position"));
            calculateOppositePositionButton.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            calculateOppositePositionButton
                .addActionListener(
                    new AbstractClickAction(Resources.getString(AspectEditorPanel.class, "calculate_opposite_position"),
                        calculateOppositePositionButton) {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            LOGGER.info("Calculate the position to set the aspect opposite position.");

                            if (callbackListener != null) {
                                try {
                                    Object configuredPositionValue = bufferedPositionModel.getValue();
                                    if (configuredPositionValue != null) {
                                        long configuredPosition = ((Long) configuredPositionValue).longValue();
                                        long totalSteps = stepControlModel.getTotalSteps();

                                        long calculatedOppositePosition = configuredPosition + totalSteps / 2;
                                        LOGGER
                                            .info("Set the calculatedOppositePosition: {}", calculatedOppositePosition);

                                        if (calculatedOppositePosition > totalSteps) {
                                            LOGGER
                                                .warn("The calculated opposite position is out of range: {}",
                                                    calculatedOppositePosition);
                                            throw new PositionOutOfRangeException(calculatedOppositePosition);
                                        }

                                        bufferedOppositePositionModel.setValue(calculatedOppositePosition);
                                    }
                                }
                                catch (PositionOutOfRangeException ex) {
                                    long position = ex.getPosition();
                                    LOGGER.info("Set the calculated position failed: {}", position);

                                    JOptionPane
                                        .showMessageDialog(AspectEditorPanel.this,
                                            Resources
                                                .getString(AspectEditorPanel.class,
                                                    "set_calculated_position_failed_message", position),
                                            Resources
                                                .getString(AspectEditorPanel.class, "set_calculated_position_title"),
                                            JOptionPane.ERROR_MESSAGE);
                                }
                                catch (Exception ex) {
                                    LOGGER.warn("Calculate opposite value failed.", ex);
                                    // TODO: handle exception
                                }

                            }
                        }
                    });
            detailFormBuilder.add(calculateOppositePositionButton).xy(7, 3);

            // angle
            final ValueModel currentOppositeAngleConverterModel =
                new ConverterValueModel(bufferedOppositePositionModel, new AngleDegreesConverter(totalSteps));
            currentOppositeAngle = WizardComponentFactory.createTextField(currentOppositeAngleConverterModel);
            currentOppositeAngle.setEditable(false);
            oppositeAngleLabel = new JLabel(Resources.getString(StepControlPanel.class, "angle"));
            detailFormBuilder.add(oppositeAngleLabel).xy(9, 3);
            detailFormBuilder.add(currentOppositeAngle).xy(11, 3);
            if (useSelectionBackground) {
                clearAttribute(oppositeAngleLabel);
            }

            // polarity
            final ValueModel oppositePolarityConverterModel =
                new ConverterValueModel(
                    presentationModel.getBufferedModel(StepControlAspect.PROPERTYNAME_OPPOSITE_POLARITY),
                    new BooleanToPolarityConverter());

            final JCheckBox currentOppositePolarity =
                WizardComponentFactory.createCheckBox(oppositePolarityConverterModel, null);
            currentOppositePolarity.setIcon(unselectedIcon);
            currentOppositePolarity.setSelectedIcon(selectedIcon);
            currentOppositePolarity.setRolloverIcon(unselectedRolloverIcon);
            currentOppositePolarity.setRolloverSelectedIcon(selectedRolloverIcon);

            final JLabel oppositePolarityLabel = new JLabel(Resources.getString(StepControlPanel.class, "polarity"));
            detailFormBuilder.add(currentOppositePolarity).xy(13, 3);
            if (useSelectionBackground) {
                clearAttribute(oppositePolarityLabel);
            }

            oppositeActivateButton = new NullButton(Resources.getString(AspectEditorPanel.class, "activate"));
            oppositeActivateButton
                .addActionListener(new AbstractClickAction(Resources.getString(AspectEditorPanel.class, "activate"),
                    oppositeActivateButton) {
                    @Override
                    public void actionPerformed(ActionEvent e) {

                        if (callbackListener != null) {

                            // activate the opposite aspect
                            callbackListener.activateOppositeAspect(AspectEditorPanel.this.originalAspect);
                        }
                    }
                });
            detailFormBuilder.add(oppositeActivateButton).xyw(15, 3, 1);
        }

        stepControlModel.addPropertyChangeListener(StepControlModel.PROPERTYNAME_TURNTABLE_TYPE, evt -> {
            boolean showAngle = stepControlModel.getTurnTableType() == TurnTableType.round;
            setComponentsVisible(showAngle);
        });

        boolean showAngle = stepControlModel.getTurnTableType() == TurnTableType.round;
        setComponentsVisible(showAngle);

        JPanel panel = detailFormBuilder.build();
        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), detailFormBuilder.build());
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(new JPanel());

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            panel = feedbackBuilder.build();
        }
        else {
            panel = detailFormBuilder.build();
        }

        if (useSelectionBackground) {
            clearAttribute(panel);
        }

        PropertyConnector.connect(activateButtonEnabled, "value", activateButton, "enabled");
        if (oppositeActivateButton != null) {
            PropertyConnector.connect(activateButtonEnabled, "value", oppositeActivateButton, "enabled");
        }

        return panel;
    }

    private boolean isRoundTurnTable(final StepControlModel stepControlModel) {
        return TurnTableType.round == stepControlModel.getTurnTableType();
    }

    private void updateActivateButtonEnabled() {
        LOGGER
            .info("Current validState: {}",
                validationResultModel != null ? validationResultModel.getValidState() : true);
        activateButtonEnabled
            .setValue(!presentationModel.isBuffering()
                && (validationResultModel != null ? validationResultModel.getValidState() : true)
                && AspectPersistanceStatus.statusPersistent == originalAspect.getStatus());
    }

    protected void triggerValidation() {
        LOGGER.info("Validation is triggered.");

        ValidationResult validationResult = validateModel();
        validationResultModel.setResult(validationResult);

        // enable or disable the buttons
        updateActivateButtonEnabled();
        updateApplyButtonEnabled();
    }

    protected ValidationResult validateModel() {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(presentationModel, "validation");

        Long position = null;
        if (this.currentPositionConverterModel.getValue() == null) {
            support.addError("currentPosition_key", "not_empty_for_write");
        }
        else {
            // get the position from the buffered value to perform range exceeded validation
            position = (Long) this.bufferedPositionModel.getValue();

            // check if the position is not above half of total steps if round table
            if (TurnTableType.round == this.turnTableType) {
                if (position.longValue() > (this.totalSteps / 2)) {
                    support.addError("currentPosition_key", "position_range_exceeded_half");
                }
            }
            else if (position.longValue() >= this.totalSteps) {
                support.addError("currentPosition_key", "position_range_exceeded");
            }
        }

        if (TurnTableType.round == this.turnTableType) {
            if (this.oppositePositionConverterModel.getValue() == null) {
                support.addError("oppositePosition_key", "not_empty_for_write");
            }
            else if (position != null) {
                // get the opposite position from the buffered value to perform position is smaller than opposite
                // position validation
                Long oppositePosition = (Long) this.bufferedOppositePositionModel.getValue();
                if (position.longValue() >= oppositePosition.longValue()) {
                    support.addError("oppositePosition_key", "opposite_position_greater_than_position");
                }

                if (oppositePosition.longValue() >= this.totalSteps) {
                    support.addError("oppositePosition_key", "position_range_exceeded");
                }
            }
        }

        ValidationResult validationResult = support.getResult();
        LOGGER.debug("Prepared validationResult: {}", validationResult);
        return validationResult;
    }

    /**
     * @return the validationResultModel
     */
    protected PortValidationResultModel getValidationResultModel() {
        return validationResultModel;
    }

    private void setComponentsVisible(boolean showAngle) {
        currentAngle.setVisible(showAngle);
        angleLabel.setVisible(showAngle);
        if (currentOppositeAngle != null) {
            currentOppositeAngle.setVisible(showAngle);
        }
        if (oppositeAngleLabel != null) {
            oppositeAngleLabel.setVisible(showAngle);
        }
    }

    public boolean isDirty() {
        if (this.presentationModel != null) {
            return this.presentationModel.isBuffering();
        }
        return false;
    }

    public StepControlAspect commitBuffer(BiConsumer<Long, Long> validationCallback) {
        LOGGER.info("Commit the buffer.");

        // get the position from the buffered value to perform unique position validation
        Long position = (Long) this.presentationModel.getBufferedValue(StepControlAspect.PROPERTYNAME_POSITION);

        // TODO check if the position is not above half of total steps if round table
        if (TurnTableType.round == this.turnTableType) {
            if (position > (this.totalSteps / 2)) {
                throw new IllegalArgumentException("Provided position value is not in the first half of total steps.");
            }
        }

        Long positionOpposite =
            (Long) this.presentationModel.getBufferedValue(StepControlAspect.PROPERTYNAME_OPPOSITE_POSITION);

        LOGGER.info("Call the validation callback with position: {}, positionOpposite: {}", position, positionOpposite);
        validationCallback.accept(position, positionOpposite);

        if (trigger != null) {
            trigger.triggerCommit();
        }

        LOGGER.info("Commited the buffer, stepControlAspect: {}", stepControlAspect);
        return stepControlAspect;
    }

    public StepControlAspect flushBuffer() {
        LOGGER.info("Flush the buffer.");

        if (trigger != null) {
            trigger.triggerFlush();
        }

        return stepControlAspect;
    }

    private void clearAttribute(final JComponent comp) {
        comp.setFont((Font) null);
        comp.setBackground((Color) null);
        comp.setForeground((Color) null);
    }

    private static final Border DLU3TOP_DLU4OTHERS =
        Paddings.createPadding(Sizes.DLUY3, Sizes.DLUX4, Sizes.DLUY4, Sizes.DLUX4);

    private JComponent createControlPanel() {

        boolean debug = false;
        FormBuilder detailFormBuilder =
            FormBuilder.create().columns("p, 3dlu, p").rows("p").panel(debug ? new FormDebugPanel() : new JPanel());

        detailFormBuilder.border(DLU3TOP_DLU4OTHERS);

        NullButton applyButton = new NullButton(Resources.getString(AspectEditorPanel.class, "apply"));
        applyButton.addActionListener(new AbstractClickAction("Apply", applyButton) {
            @Override
            public void actionPerformed(ActionEvent e) {
                try {
                    applyChanges();
                }
                catch (VetoChangeException ex) {
                    LOGGER.warn("Save changes was vetoed.");
                }
            }
        });

        detailFormBuilder.add(applyButton).xy(1, 1);
        NullButton revertButton =
            new NullButton(
                EditorType.editorUpdate == editorType ? Resources.getString(AspectEditorPanel.class, "revert")
                    : Resources.getString(AspectEditorPanel.class, "cancel"));
        revertButton.addActionListener(new AbstractClickAction("Revert", revertButton) {
            @Override
            public void actionPerformed(ActionEvent e) {

                if (callbackListener != null) {
                    try {
                        callbackListener.discardChanges(() -> flushBuffer());
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Discard pending changes failed.", ex);
                    }
                }
            }
        });
        detailFormBuilder.add(revertButton).xy(3, 1);

        applyButton.setEnabled(presentationModel.isBuffering());
        revertButton.setEnabled(presentationModel.isBuffering());

        PropertyConnector.connect(applyButtonEnabled, "value", applyButton, "enabled");
        PropertyConnector.connect(presentationModel, "buffering", revertButton, "enabled");

        return detailFormBuilder.build();
    }

    private void updateApplyButtonEnabled() {
        LOGGER
            .info("Current validState: {}",
                validationResultModel != null ? validationResultModel.getValidState() : true);
        applyButtonEnabled
            .setValue(presentationModel.isBuffering()
                && (validationResultModel != null ? validationResultModel.getValidState() : true));
    }

    private abstract static class AbstractClickAction implements ActionListener {

        String buttonName;

        AbstractButton button;

        public AbstractClickAction(String buttonName, AbstractButton button) {
            this.buttonName = buttonName;
            this.button = button;
        }
    }

    public void applyChanges() {

        if (callbackListener != null) {
            LOGGER.info("Apply the changes.");
            try {
                callbackListener
                    .saveChanges(originalAspect, (originalAspect) -> commitBuffer(
                        (pos, posOpp) -> callbackListener.verifyUniquePosition(originalAspect, pos, posOpp)));
            }
            catch (IllegalArgumentException ex) {
                LOGGER.warn("Save changes failed: {}", ex.getMessage());

                JOptionPane
                    .showMessageDialog(AspectEditorPanel.this,
                        Resources.getString(StepControlPanel.class, "apply-changes-failed", ex.getMessage()),
                        Resources.getString(StepControlPanel.class, "apply"), JOptionPane.ERROR_MESSAGE);

                throw new VetoChangeException();
            }
            catch (Exception ex) {
                LOGGER.warn("Save pending changes failed.", ex);

                JOptionPane
                    .showMessageDialog(AspectEditorPanel.this,
                        Resources.getString(StepControlPanel.class, "apply-changes-failed", "unspecified"),
                        Resources.getString(StepControlPanel.class, "apply"), JOptionPane.ERROR_MESSAGE);

                throw new VetoChangeException();
            }
        }

        cleanup();
    }

    public void cleanup() {
        LOGGER.info("Cleanup the editor: {}", this);

        if (this.listener != null) {
            this.presentationModel.removePropertyChangeListener(PresentationModel.PROPERTY_BUFFERING, this.listener);
            this.originalAspect.removePropertyChangeListener(StepControlAspect.PROPERTYNAME_STATUS, this.listener);
        }

    }

}
