package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;

import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.table.DefaultTableCellRenderer;

import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.FeedbackPosition;
import org.bidib.jbidibc.messages.enums.FeatureEnum;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.mvc.main.controller.FeedbackPositionPanelController;
import org.bidib.wizard.mvc.main.model.FeedbackPositionModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.glazed.DecodersSelect;
import org.bidib.wizard.mvc.main.view.panel.glazed.FeedbackPositionComparator;
import org.bidib.wizard.mvc.main.view.panel.glazed.FeedbackPositionTableFormat;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.grid.CategorizedTable;

import ca.odell.glazedlists.EventList;
import ca.odell.glazedlists.FilterList;
import ca.odell.glazedlists.SortedList;
import ca.odell.glazedlists.matchers.ThreadedMatcherEditor;
import ca.odell.glazedlists.swing.AdvancedTableModel;
import ca.odell.glazedlists.swing.GlazedListsSwing;
import ca.odell.glazedlists.swing.TableComparatorChooser;

public class FeedbackPositionListPanel implements TabVisibilityProvider, TabPanelProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPositionListPanel.class);

    private static final int EXTRA_VERTICAL_SPACE = 4;

    private final FeedbackPositionPanelController controller;

    private final FeedbackPositionModel feedbackPositionModel;

    private final TabVisibilityListener tabVisibilityListener;

    private DecodersSelect decodersSelect;

    private static final String ENCODED_FILTER_PANEL_COLUMN_SPECS = "70dlu:grow";

    private static final String ENCODED_FILTER_PANEL_ROW_SPECS = "pref, $lg, fill:50dlu:grow, $lg, pref, $lg, pref";

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:50dlu:grow";

    private JPanel contentPanel;

    private final AdvancedTableModel<FeedbackPosition> positionsTableModel;

    private JEditorPane editor;

    private static class TimestampCellRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        private final DateFormat formatter = new SimpleDateFormat("HH:mm:ss.SSS");

        public TimestampCellRenderer() {
        }

        @Override
        public void setValue(Object value) {
            setText((value == null) ? "" : formatter.format(value));
        }

    }

    public FeedbackPositionListPanel(final FeedbackPositionPanelController controller,
        final FeedbackPositionModel feedbackPositionModel, MainModel model,
        final TabVisibilityListener tabVisibilityListener) {

        this.controller = controller;
        this.feedbackPositionModel = feedbackPositionModel;
        this.tabVisibilityListener = tabVisibilityListener;
        LOGGER.debug("Create new FeedbackPositionListPanel.");

        FormBuilder dialogBuilderFilter = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilderFilter =
                FormBuilder
                    .create().columns(ENCODED_FILTER_PANEL_COLUMN_SPECS).rows(ENCODED_FILTER_PANEL_ROW_SPECS)
                    .panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilderFilter =
                FormBuilder
                    .create().columns(ENCODED_FILTER_PANEL_COLUMN_SPECS).rows(ENCODED_FILTER_PANEL_ROW_SPECS)
                    .panel(panel);
        }

        FormBuilder dialogBuilder = null;
        // boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        EventList<FeedbackPosition> feedbackPositionEventList = feedbackPositionModel.getPositionsEventList();
        SortedList<FeedbackPosition> sortedFeedbackPositions =
            new SortedList<FeedbackPosition>(feedbackPositionEventList, new FeedbackPositionComparator());

        decodersSelect = new DecodersSelect(sortedFeedbackPositions);

        // create the filter list
        FilterList<FeedbackPosition> filteredFeedbackPositions =
            new FilterList<FeedbackPosition>(sortedFeedbackPositions,
                new ThreadedMatcherEditor<FeedbackPosition>(decodersSelect));

        positionsTableModel =
            GlazedListsSwing
                .eventTableModelWithThreadProxyList(filteredFeedbackPositions, new FeedbackPositionTableFormat());

        CategorizedTable positionsJTable = new CategorizedTable(positionsTableModel) {
            private static final long serialVersionUID = 1L;

            @Override
            public Dimension getPreferredScrollableViewportSize() {
                return getPreferredSize();
            }

            @Override
            public boolean getScrollableTracksViewportHeight() {
                return getPreferredSize().height < getParent().getHeight();
            }

        };

        positionsJTable
            .getColumnModel().getColumn(FeedbackPositionTableFormat.COLUMN_LASTSEEN)
            .setCellRenderer(new TimestampCellRenderer());
        positionsJTable.setRowAutoResizes(false);
        positionsJTable.setRowHeight(positionsJTable.getRowHeight() + EXTRA_VERTICAL_SPACE);

        editor = new JEditorPane("text/html", getEmptyTableText());
        editor.setOpaque(false);
        editor.setEditable(false);
        editor.putClientProperty(JEditorPane.HONOR_DISPLAY_PROPERTIES, Boolean.TRUE);

        positionsJTable.setLayout(new GridBagLayout());
        positionsJTable.add(editor);

        positionsTableModel.addTableModelListener(e -> {
            boolean visible = positionsTableModel.getRowCount() == 0;
            editor.setVisible(visible);
        });

        // do not remove sorter, otherwise sorting will no longer work
        TableComparatorChooser<FeedbackPosition> tableSorter =
            TableComparatorChooser
                .install(positionsJTable, sortedFeedbackPositions, TableComparatorChooser.MULTIPLE_COLUMN_MOUSE);
        // set the initial sorting
        tableSorter.fromString("column " + FeedbackPositionTableFormat.COLUMN_LASTSEEN + " reversed");

        JScrollPane positionsTableScrollPane = new JScrollPane(positionsJTable);

        // create the decoders list
        JList<String> decodersJList = decodersSelect.getJList();
        JScrollPane decodersListScrollPane = new JScrollPane(decodersJList);

        final JButton clearButton = new JButton(Resources.getString(FeedbackPositionListPanel.class, "label.clear"));
        clearButton.setToolTipText(Resources.getString(FeedbackPositionListPanel.class, "tooltip.clear"));
        clearButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                decodersJList.clearSelection();
            }
        });

        dialogBuilderFilter
            .add(new JLabel(Resources.getString(FeedbackPositionListPanel.class, "label.decoder"))).xy(1, 1);
        dialogBuilderFilter.add(decodersListScrollPane).xy(1, 3);
        dialogBuilderFilter.add(clearButton).xy(1, 5);

        dialogBuilder.add(dialogBuilderFilter.build()).xy(1, 1);

        dialogBuilder.add(positionsTableScrollPane).xy(3, 1);

        contentPanel = dialogBuilder.build();

        JPopupMenu popupMenu = new BasicPopupMenu();
        JMenuItem clearPositionTable = new JMenuItem(Resources.getString(getClass(), "clear_table"));
        clearPositionTable.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireClearPositionTable();
            }
        });
        popupMenu.add(clearPositionTable);

        positionsJTable.setComponentPopupMenu(popupMenu);

        // the name is displayed on the tab
        contentPanel.setName(getName());
    }

    protected void fireClearPositionTable() {

        this.feedbackPositionModel.getPositionsEventList().clear();
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    private String getEmptyTableText() {
        return Resources.getString(FeedbackPositionListPanel.class, "emptyTable");
    }

    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = feedbackPositionModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = hasFeedbackPositions(node.getNode().getFeatures());
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    public void listChanged() {
        LOGGER.info("List has changed, remove all rows and add rows again.");

        tabVisibilityListener.setTabVisible(contentPanel, isTabVisible());
    }

    private boolean hasFeedbackPositions(Collection<Feature> features) {

        Feature feedbackPositions = Feature.findFeature(features, FeatureEnum.FEATURE_BM_POSITION_ON.getNumber());
        return (feedbackPositions != null && feedbackPositions.getValue() > 0);
    }
}
