package org.bidib.wizard.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);

    public static final String EXCEL_EXTENSION = "xlsx";

    public static boolean isPathValid(String path) {

        try {
            File dir = org.apache.commons.io.FileUtils.getFile(path);
            dir.getCanonicalPath();
            // Paths.get(path);
        }
        catch (/* InvalidPathException | */ IOException ex) {
            return false;
        }

        return true;
    }

    public static final String FILENAME_REGEX_PATTERN = "^[^\\\\/:*?\"<>|\\$\\s]{1,255}$";

    public static boolean isFilenameValid(String filename) {
        if (filename == null) {
            return false;
        }
        return filename.matches(FILENAME_REGEX_PATTERN);
    }

    public static String escapeInvalidFilenameCharacters(String path, String replacement) {

        // return path.replaceAll("[^a-zA-Z0-9_\\.\\-]", replacement);
        return path.replaceAll("[\\\\/:*?\"<>|\\$\\s]", replacement);
    }

    public static void copyFile(final File srcFile, final File destFile, final CopyOption... copyOptions)
        throws IOException {
        requireFileCopy(srcFile, destFile);
        requireFile(srcFile, "srcFile");
        requireCanonicalPathsNotEquals(srcFile, destFile);
        org.apache.commons.io.FileUtils.createParentDirectories(destFile);
        requireFileIfExists(destFile, "destFile");
        if (destFile.exists()) {
            requireCanWrite(destFile, "destFile");
        }

        // On Windows, the last modified time is copied by default.
        Files.copy(srcFile.toPath(), destFile.toPath(), copyOptions);

        // do not compare file size
    }

    private static void requireFileCopy(final File source, final File destination) throws FileNotFoundException {
        requireExistsChecked(source, "source");
        Objects.requireNonNull(destination, "destination");
    }

    private static File requireExistsChecked(final File file, final String fileParamName) throws FileNotFoundException {
        Objects.requireNonNull(file, fileParamName);
        if (!file.exists()) {
            throw new FileNotFoundException(
                "File system element for parameter '" + fileParamName + "' does not exist: '" + file + "'");
        }
        return file;
    }

    private static File requireFile(final File file, final String name) {
        Objects.requireNonNull(file, name);
        if (!file.isFile()) {
            throw new IllegalArgumentException("Parameter '" + name + "' is not a file: " + file);
        }
        return file;
    }

    private static void requireCanonicalPathsNotEquals(final File file1, final File file2) throws IOException {
        final String canonicalPath = file1.getCanonicalPath();
        if (canonicalPath.equals(file2.getCanonicalPath())) {
            throw new IllegalArgumentException(String
                .format("File canonical paths are equal: '%s' (file1='%s', file2='%s')", canonicalPath, file1, file2));
        }
    }

    private static File requireFileIfExists(final File file, final String name) {
        Objects.requireNonNull(file, name);
        return file.exists() ? requireFile(file, name) : file;
    }

    private static void requireCanWrite(final File file, final String name) {
        Objects.requireNonNull(file, "file");
        if (!file.canWrite()) {
            throw new IllegalArgumentException("File parameter '" + name + " is not writable: '" + file + "'");
        }
    }

    public static boolean silentlyDeleteFile(final File file, final Long waitAfterDelete) {
        boolean deleted = true;
        if (file.exists()) {
            deleted = file.delete();

            if (waitAfterDelete != null) {
                try {
                    Thread.sleep(waitAfterDelete.longValue());
                }
                catch (Exception ex) {
                    LOGGER.warn("Sleep after delete file silently failed.", ex);
                }
            }
        }
        return deleted;
    }
}
