package org.bidib.wizard.mvc.nodedebug.controller;

import javax.annotation.PostConstruct;
import javax.swing.SwingUtilities;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.connection.BidibConnection;
import org.bidib.wizard.api.model.event.DebugActionEvent;
import org.bidib.wizard.api.model.listener.NodeSelectionListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.config.DebugConsoleControllerFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.node.DebugService;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.nodedebug.model.DebugConsoleModel;
import org.bidib.wizard.mvc.nodedebug.view.DebugConsoleView;
import org.bidib.wizard.mvc.nodedebug.view.listener.DebugConsoleViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

public class DebugConsoleController {
    private static final Logger LOGGER = LoggerFactory.getLogger(DebugConsoleController.class);

    // private static String SEARCHKEY = "DebugConsoleView";

    @Autowired
    private DockingDesktop desktop;

    private DockableStateChangeListener dockableStateChangeListener;

    private DebugConsoleView debugConsoleView;

    @Autowired
    private MainModel mainModel;

    @Autowired
    private DebugConsoleModel debugConsoleModel;

    @Autowired
    private ConnectionRegistry connectionRegistry;

    @Autowired
    private DebugService debugService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private ApplicationContext applicationContext;

    public DebugConsoleController() {
    }

    @PostConstruct
    public void initialize() {

        debugService.getSubjectDebugEvents().subscribe(evt -> {
            LOGGER.debug("Received DebugActionEvent: {}", evt);

            SwingUtilities.invokeLater(() -> {
                ensureConsoleVisible();

                publishToConsole(evt);
            });
        });
    }

    public void start() {

        // check if the console is already opened
        String searchKey = DockKeys.DEBUG_CONSOLE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable consoleView = desktop.getContext().getDockableByKey(searchKey);
        if (consoleView != null) {
            LOGGER.info("Select the existing console view.");
            DockUtils.selectWindow(consoleView);
            return;
        }

        LOGGER.info("Create new DebugConsoleView.");

        debugConsoleModel.setSelectedNode(mainModel.getSelectedNode());

        debugConsoleView = new DebugConsoleView(debugConsoleModel);

        // add the log panel next to the booster panel
        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState boosterTableView = null;
            // search the booster table view
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_BOOSTER_TABLE_VIEW.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the booster table view dockable.");
                    boosterTableView = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (boosterTableView != null) {
                LOGGER.info("Add the Debug console view to the booster table view panel.");
                dock = boosterTableView.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new log panel at order: {}", order);

                desktop.createTab(dock, debugConsoleView, order, true);
            }
            else if (desktop.getDockables().length > 1) {

                desktop.split(dock, debugConsoleView, DockingConstants.SPLIT_BOTTOM);
                desktop.setDockableHeight(debugConsoleView, 0.2d);
            }
            else {
                desktop.split(dock, debugConsoleView, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(debugConsoleView, RelativeDockablePosition.RIGHT);
        }

        this.dockableStateChangeListener = new DockableStateChangeListener() {

            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                if (event.getNewState().getDockable().equals(debugConsoleView) && event.getNewState().isClosed()) {
                    LOGGER.info("DebugConsoleView was closed, free resources.");

                    try {
                        desktop.removeDockableStateChangeListener(dockableStateChangeListener);
                    }
                    catch (Exception ex) {
                        LOGGER
                            .warn("Remove dockableStateChangeListener from desktop failed: "
                                + dockableStateChangeListener, ex);
                    }
                    finally {
                        dockableStateChangeListener = null;
                    }

                    debugConsoleView.close();
                }
            }
        };
        desktop.addDockableStateChangeListener(this.dockableStateChangeListener);

        debugConsoleView.addDebugConsoleViewListener(new DebugConsoleViewListener() {

            @Override
            public void transmit() {

                String sendText = debugConsoleModel.getSendText();

                if (StringUtils.isEmpty(sendText)) {
                    LOGGER.info("No data to send!");
                    return;
                }

                // TODO maximum size of text to send?
                if (sendText.length() > 32) {
                    LOGGER.info("The data to send is too long!");
                    return;
                }

                final NodeInterface selectedNode = debugConsoleModel.getSelectedNode();
                if (selectedNode != null) {

                    LOGGER.info("Transmit the text: {}", sendText);

                    if (!sendText.endsWith("\n")) {
                        sendText += "\n";
                    }

                    final BidibConnection connection =
                        connectionRegistry.getConnection(ConnectionRegistry.CONNECTION_ID_MAIN);

                    connection
                        .setString(selectedNode, StringData.NAMESPACE_DEBUG, StringData.INDEX_DEBUG_STDIN, sendText);
                }
                else {
                    LOGGER.warn("No node selected to send the STRING for debug purposes.");
                }
            }
        });

        this.mainModel.addNodeSelectionListener(new NodeSelectionListener() {

            @Override
            public void selectedNodeChanged(NodeInterface selectedNode) {
                LOGGER.info("Set the selected node in the debugConsoleModel: {}", selectedNode);
                debugConsoleModel.setSelectedNode(selectedNode);
            }

        });
    }

    public synchronized void ensureConsoleVisible() {
        // check if the console is already opened
        String searchKey = DockKeys.DEBUG_CONSOLE_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);

        DockingDesktop desktop =
            DefaultApplicationContext.getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);

        Dockable consoleView = desktop.getContext().getDockableByKey(searchKey);
        if (consoleView != null) {
            LOGGER.info("Use the existing console view.");
            return;
        }
        else {
            LOGGER.info("Create new controller to open the console.");

            DebugConsoleControllerFactory debugConsoleControllerFactory =
                applicationContext.getBean(DebugConsoleControllerFactory.class);

            final DebugConsoleController debugConsoleController = debugConsoleControllerFactory.createController();
            debugConsoleController.start();
        }
    }

    protected void publishToConsole(final DebugActionEvent evt) {

        int stringId = evt.getStringId();
        String nodeLabel = getNodeLabel(evt.getNode());
        String value = evt.getValue();

        ConsoleColor textColor = ConsoleColor.black;
        switch (stringId) {
            case StringData.INDEX_DEBUG_TRACE:
                textColor = ConsoleColor.gray;
                break;
            case StringData.INDEX_DEBUG_DEBUG:
                textColor = ConsoleColor.gray;
                break;
            case StringData.INDEX_DEBUG_INFO:
                textColor = ConsoleColor.blue;
                break;
            case StringData.INDEX_DEBUG_WARN:
            case StringData.INDEX_DEBUG_STDERR:
                textColor = ConsoleColor.red;
                break;
            default:
                break;
        }

        // add line
        debugConsoleModel
            .addConsoleLine(textColor,
                String.format("%s : %s - %d : %s", evt.getConnectionId(), nodeLabel, stringId, value));
    }

    /**
     * @return the node label
     */
    private String getNodeLabel(final NodeInterface node) {

        final NodeLabels nodeLabels = this.wizardLabelWrapper.loadLabels(node.getUniqueId());
        if (nodeLabels != null) {
            BidibNodeNameUtils.NodeLabelData labelData =
                BidibNodeNameUtils.prepareLabel(node, nodeLabels, false, false);
            String userName = labelData.getNodeLabel();
            if (StringUtils.isNotBlank(userName)) {
                return userName;
            }
        }

        return ByteUtils.getUniqueIdAsString(node.getUniqueId());
    }

}
