package org.bidib.wizard.mvc.main.controller;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.MapUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.SwitchingNodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.PortValueListener;
import org.bidib.wizard.api.script.ScriptCommand;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.common.labels.BidibLabelUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.script.DefaultScriptContext;
import org.bidib.wizard.common.script.common.WaitCommand;
import org.bidib.wizard.common.script.engine.ScriptEngine;
import org.bidib.wizard.common.script.switching.MotorPortCommand;
import org.bidib.wizard.common.script.switching.PortScripting;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.MotorPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.PortTypeAware;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.MotorPortStatus;
import org.bidib.wizard.mvc.main.controller.wrapper.NodePortWrapper;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.MotorPortTableModel;
import org.bidib.wizard.mvc.main.model.listener.MotorPortModelListener;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.MotorPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;
import io.reactivex.rxjava3.subjects.Subject;

public class MotorPortPanelController implements PortScripting {

    private static final Logger LOGGER = LoggerFactory.getLogger(MotorPortPanelController.class);

    private final MainModel mainModel;

    private MotorPortListPanel motorPortListPanel;

    private final Map<NodeInterface, NodePortWrapper> testToggleRegistry = new LinkedHashMap<>();

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private ConsoleService consoleService;

    private final Subject<PortConfigChangeEvent> portConfigChangeEventSubject;

    private CompositeDisposable compDisp;

    public MotorPortPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        this.compDisp = new CompositeDisposable();

        this.portConfigChangeEventSubject = PublishSubject.<PortConfigChangeEvent> create().toSerialized();
    }

    public MotorPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final MotorPortTableModel tableModel = new MotorPortTableModel();

        tableModel.setPortListener(new MotorPortModelListener() {

            @Override
            public void labelChanged(MotorPort port, String label) {
                port.setLabel(label);

                try {
                    final NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.motorPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save motor port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void configChanged(MotorPort port, PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port config value are changed for port: {}", port);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the motorPort parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }
            }

            @Override
            public void changePortType(LcOutputType portType, MotorPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);
            }

            @Override
            public void testButtonPressed(MotorPort port, MotorPortStatus newStatus) {
                LOGGER.info("The test button was pressed for port: {}", port);

                NodeInterface node = mainModel.getSelectedNode();
                if (MotorPortStatus.TEST != port.getStatus()) {
                    stopTestToggleTask(node, port);

                    MotorPort motorPort = port;
                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), motorPort);
                }
                else {
                    addTestToggleTask(node, port);
                }
            }

        });

        final MotorPortListPanel motorPortListPanel =
            new MotorPortListPanel(this, tableModel, mainModel, tabVisibilityListener, portConfigChangeEventSubject);

        motorPortListPanel.setPortListener(new PortValueListener<MotorPort>() {
            @Override
            public Class<?> getPortClass() {
                return MotorPort.class;
            }

            @Override
            public void labelChanged(final MotorPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                final NodeLabels nodeLabels = getNodeLabels();
                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.motorPort, port.getId(), port.getLabel());
                saveLabels();

                motorPortListPanel.repaint();
            }

            @Override
            public void valueChanged(final NodeInterface node, final MotorPort port) {
                LOGGER.info("Value has changed for motor port: {}", port);

                SwingUtilities.invokeLater(() -> tableModel.notifyPortStatusChanged(port));
            }

            @Override
            public void configChanged(final NodeInterface node, final MotorPort port) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                SwingUtilities.invokeLater(() -> tableModel.notifyPortConfigChanged(port));
            }

        });

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeWillChange(final NodeInterface node) {
                LOGGER.info("The selected node will change!");
                try {
                    List<NodeInterface> nodes = new LinkedList<>();
                    for (NodeInterface currentNode : testToggleRegistry.keySet()) {
                        nodes.add(currentNode);
                    }
                    LOGGER.info("Found nodes to stop the test toggle task: {}", nodes);
                    for (NodeInterface currentNode : nodes) {
                        stopTestToggleTask(currentNode, null);
                    }
                    LOGGER.info("Stop the test toggle task passed for nodes: {}", nodes);
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop test toggle tasks failed.", ex);
                }
            }

            @Override
            public void nodeChanged(NodeInterface node) {
                super.nodeChanged(node);

                LOGGER.info("The selected node has been changed: {}", node);

                compDisp.dispose();
                compDisp.clear();

                compDisp = new CompositeDisposable();

                if (node != null) {
                    addMotorPortModelListener(node);
                }
            }

        });

        // keep the reference
        this.motorPortListPanel = motorPortListPanel;

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null) {
            addMotorPortModelListener(selectedNode);
        }

        return motorPortListPanel;
    }

    public void addTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Add test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.remove(node);
        ScriptEngine<PortScripting> scriptEngine = null;
        if (nodePortWrapper != null) {
            scriptEngine = nodePortWrapper.removePort(port);
        }

        if (scriptEngine != null) {
            LOGGER.info("Found a node scripting engine in the registry: {}", scriptEngine);
            try {
                scriptEngine.stopScript(Long.valueOf(2000));
            }
            catch (Exception ex) {
                LOGGER.warn("Stop script failed.", ex);
            }
        }

        ApplicationContext context = new DefaultScriptContext();
        context.register(DefaultScriptContext.KEY_SELECTED_NODE, node);
        context.register(DefaultScriptContext.KEY_MAIN_MODEL, mainModel);

        scriptEngine = new ScriptEngine<PortScripting>(this, context);

        List<ScriptCommand<PortScripting>> scriptCommands = new LinkedList<ScriptCommand<PortScripting>>();
        MotorPortCommand<PortScripting> spc = new MotorPortCommand<>(this.consoleService);
        spc.parse(MotorPortCommand.KEY + " " + port.getId() + " 10");
        scriptCommands.add(spc);
        WaitCommand<PortScripting> wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);
        spc = new MotorPortCommand<>(this.consoleService);
        spc.parse(MotorPortCommand.KEY + " " + port.getId() + " 0");
        scriptCommands.add(spc);
        wc = new WaitCommand<>(this.consoleService);
        wc.parse(WaitCommand.KEY + " 2000");
        scriptCommands.add(wc);

        LOGGER.info("Prepared list of commands: {}", scriptCommands);

        scriptEngine.setScriptCommands(scriptCommands);
        // repeating
        scriptEngine.setScriptRepeating(true);

        if (nodePortWrapper == null) {
            LOGGER.info("Create new NodePortWrapper for node: {}", node);
            nodePortWrapper = new NodePortWrapper(node);
        }

        LOGGER.info("Put script engine in registry for node: {}", node);
        nodePortWrapper.addPort(port, scriptEngine);

        testToggleRegistry.put(node, nodePortWrapper);

        scriptEngine.startScript();
    }

    public void stopTestToggleTask(final NodeInterface node, final Port<?> port) {
        LOGGER.info("Stop test toggle task for node: {}, port: {}", node, port);

        NodePortWrapper nodePortWrapper = testToggleRegistry.get(node);

        if (nodePortWrapper != null) {
            Set<Port<?>> toRemove = new HashSet<>();
            if (port != null) {
                toRemove.add(port);
            }
            else {
                toRemove.addAll(nodePortWrapper.getKeySet());
            }

            for (Port<?> removePort : toRemove) {
                ScriptEngine<PortScripting> engine = nodePortWrapper.removePort(removePort);

                if (engine != null) {
                    LOGGER.info("Found a node scripting engine in the registry: {}", engine);
                    try {
                        engine.stopScript(Long.valueOf(2000));
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Stop script failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No scripting engine found for node: {}", node);
                }
            }

            if (nodePortWrapper.isEmpty()) {
                LOGGER.info("No more ports registered for node: {}", node);
                testToggleRegistry.remove(node);
            }
        }
    }

    @Override
    public void sendPortStatusAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, BidibStatus portStatus) {

    }

    @Override
    public void sendPortValueAction(
        final SwitchingNodeInterface node, LcOutputType lcOutputType, int port, int portValue) {
        LOGGER.info("Send motor value on the port: {}, portValue: {}", port, portValue);
        try {
            final SwitchingNodeInterface selectedNode =
                node != null ? node : mainModel.getSelectedNode().getSwitchingNode();

            MotorPort motorPort = new MotorPort();
            motorPort.setId(port);
            motorPort.setValue(portValue);
            switchingNodeService.setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, motorPort);
        }
        catch (Exception ex) {
            LOGGER.warn("Activate motor port failed.", ex);
        }
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.motorPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.motorPortListPanel.refreshView());
        }
    }

    private void addMotorPortModelListener(final NodeInterface selectedNode) {

        LOGGER.info("Add motor port model listener for node: {}", selectedNode);

        final Disposable disp = this.portConfigChangeEventSubject.subscribe(evt -> {
            LOGGER.info("Received event: {}", evt);

            final PortTypeAware port = evt.getPort();

            // update port config
            try {
                final MotorPort motorPort = new MotorPort();
                motorPort.setId(port.getPortNumber());

                LOGGER.info("Prepared motor port: {}", motorPort);

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(), motorPort,
                        null, evt.getPortConfig());
            }
            catch (Exception ex) {
                LOGGER.warn("Set the motorport config failed.", ex);
                selectedNode.setNodeHasError(true);
                selectedNode.setReasonData("Set the motorport config failed.");
            }
        });

        compDisp.add(disp);
    }
}
