package org.bidib.wizard.mvc.main.view.exchange;

import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;

import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.bidib2.BiDiB;
import org.bidib.jbidibc.core.schema.bidib2.ConfigurationVariable;
import org.bidib.jbidibc.core.schema.bidib2.MacroPointInput;
import org.bidib.jbidibc.core.schema.bidib2.MacroPointOutput;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.LookupService;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.exception.UserActionAbortedException;
import org.bidib.wizard.client.common.view.DefaultBusyFrame;
import org.bidib.wizard.client.common.view.cvdef.CvContainer;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeModelRegistry;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.dmx.client.model.DmxSceneryModel;
import org.bidib.wizard.mvc.main.controller.MainController;
import org.bidib.wizard.mvc.main.view.MainNodeListActionListener;
import org.bidib.wizard.mvc.main.view.component.NodeErrorsDialog;
import org.bidib.wizard.mvc.main.view.component.SaveNodeConfigurationDialog;
import org.bidib.wizard.mvc.main.view.panel.listener.NodeListActionListener;
import org.bidib.wizard.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeExchangeHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(NodeExchangeHelper.class);

    public static final String NODE_X_EXTENSION = "nodex";

    private static final String WORKING_DIR_NODE_EXCHANGE_KEY = "nodeExchange";

    public void importNode(
        final DefaultBusyFrame parentComponent, final NodeInterface node, final SettingsService settingsService,
        final NodeService nodeService, final SwitchingNodeService switchingNodeService,
        final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry, final FileFilter nodeFilter,
        final WizardLabelWrapper wizardLabelWrapper, final StatusBar statusBar,
        final NodeListActionListener nodeListActionListener, final LookupService lookupService) {

        final Map<String, Object> importParams = new HashMap<String, Object>();

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY);

        FileDialog dialog = new FileDialog(parentComponent, FileDialog.OPEN, storedWorkingDirectory, null, nodeFilter) {
            private JCheckBox checkRestoreAccessoryContent;

            private JCheckBox checkRestoreMacroContent;

            private JCheckBox checkRestoreNodeString;

            private JCheckBox checkRestoreFeatures;

            private JCheckBox checkRestoreCVs;

            @Override
            public void approve(String fileName) {

                LOGGER.info("Start importing node, fileName: {}", fileName);

                final BiDiB bidib;
                try {
                    LOGGER.info("Use BiDiB data to restore the node.");

                    // load the BiDiB instance from the file
                    final File bidibFile = new File(fileName);
                    bidib = BidibFactory.loadBiDiBFile(bidibFile);

                    // we expect a node in the bidib instance
                    if (bidib == null || bidib.getNodes() == null || bidib.getNodes().getNode().isEmpty()) {
                        LOGGER.warn("The provided BiDiB data is not valid for node import.");

                        throw new IllegalArgumentException("The provided BiDiB data is not valid for node import.");
                    }
                }
                catch (FileNotFoundException ex) {
                    LOGGER.warn("Load the bidib instance from the file failed.", ex);
                    throw new IllegalStateException(ex);
                }

                try {
                    final ExportNodeOptionsDialog exportNodeOptionsDialog =
                        new ExportNodeOptionsDialog(parentComponent,
                            Resources.getString(NodeExchangeHelper.class, "importOptions.title"), node) {

                            private static final long serialVersionUID = 1L;

                            @Override
                            protected Component getAdditionalPanel() {

                                boolean defaultImportCVOnly = false;
                                boolean preventImportCV = false;

                                // check if the current node is the OneControl with GPIO and FW version below 3.x
                                if ((ProductUtils.isOneControl(node.getUniqueId())
                                    || ProductUtils.isOneDriveTurn(node.getUniqueId()))
                                    && SoftwareVersion
                                        .build(3, 0, 0).isHigherThan(node.getNode().getSoftwareVersion())) {
                                    int result =
                                        JOptionPane
                                            .showConfirmDialog(JOptionPane.getFrameForComponent(parentComponent),
                                                Resources
                                                    .getString(MainNodeListActionListener.class,
                                                        "importOneControl.message"),
                                                Resources
                                                    .getString(MainNodeListActionListener.class,
                                                        "importOneControl.title"),
                                                JOptionPane.YES_NO_OPTION, JOptionPane.INFORMATION_MESSAGE);

                                    if (result == JOptionPane.OK_OPTION) {
                                        LOGGER
                                            .info(
                                                "User selected OK to uncheck all options expect the import of CV values.");
                                        defaultImportCVOnly = true;
                                        importParams
                                            .put(SaveNodeConfigurationDialog.DEFAULT_IMPORT_CV_ONLY,
                                                defaultImportCVOnly);
                                    }
                                }
                                else if (ProductUtils.isReadyServoTurn(node.getUniqueId()) && node
                                    .getNode().getSoftwareVersion().isHigherThan(SoftwareVersion.build(2, 3, 2))) {
                                    // ReadyServoTurn 2.04.x must not import CV values from 2.03.02 and below

                                    preventImportCV = true;

                                    String releaseVersion =
                                        org.bidib.wizard.utils.NodeUtils.findFirmwareReleaseOfBackup(bidib);
                                    if (releaseVersion != null) {
                                        SoftwareVersion releaseFwVersion = SoftwareVersion.parse(releaseVersion);
                                        if (releaseFwVersion.isHigherOrEqualThan(SoftwareVersion.build(2, 5, 0))) {
                                            preventImportCV = false;
                                        }
                                    }
                                    LOGGER.info("Current preventImportCV: {}", preventImportCV);
                                }

                                // prepare a panel with checkboxes for loading macro content before export
                                final JPanel additionalPanel = new JPanel();
                                additionalPanel.setLayout(new BoxLayout(additionalPanel, BoxLayout.PAGE_AXIS));

                                checkRestoreNodeString =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreNodeString"),
                                        true);
                                additionalPanel.add(checkRestoreNodeString);

                                if (NodeUtils.hasAccessoryFunctions(node.getUniqueId())) {
                                    // TODO check if this is really necessary ... user can use store permanently
                                    // manually
                                    checkRestoreAccessoryContent =
                                        new JCheckBox(
                                            Resources.getString(MainController.class, "checkRestoreAccessoryContent"),
                                            !defaultImportCVOnly);
                                    additionalPanel.add(checkRestoreAccessoryContent);
                                    checkRestoreMacroContent =
                                        new JCheckBox(
                                            Resources.getString(MainController.class, "checkRestoreMacroContent"),
                                            !defaultImportCVOnly);
                                    additionalPanel.add(checkRestoreMacroContent);
                                    if (defaultImportCVOnly) {
                                        checkRestoreAccessoryContent.setEnabled(false);
                                        checkRestoreMacroContent.setEnabled(false);
                                    }
                                }
                                checkRestoreFeatures =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreFeatures"),
                                        false);
                                additionalPanel.add(checkRestoreFeatures);
                                if (defaultImportCVOnly) {
                                    checkRestoreFeatures.setEnabled(false);
                                }
                                checkRestoreCVs =
                                    new JCheckBox(Resources.getString(MainController.class, "checkRestoreCVs"),
                                        defaultImportCVOnly);
                                additionalPanel.add(checkRestoreCVs);
                                if (preventImportCV) {
                                    checkRestoreCVs.setSelected(false);
                                    checkRestoreCVs.setEnabled(false);
                                }
                                else if (defaultImportCVOnly) {
                                    checkRestoreCVs.setEnabled(false);
                                }

                                return additionalPanel;
                            }
                        };
                    exportNodeOptionsDialog.setVisible(true);

                    if (exportNodeOptionsDialog.getResult() == JOptionPane.OK_OPTION) {
                        LOGGER.info("User confirmed options.");
                    }
                    else {
                        LOGGER.info("User cancelled options.");

                        return;
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Show exportNodeOptionsDialog failed.", ex);
                    return;
                }

                try {
                    parentComponent.setBusy(true);
                    // LOGGER.info("Start importing node, fileName: {}", fileName);
                    //
                    // LOGGER.info("Use BiDiB data to restore the node.");
                    //
                    // // load the BiDiB instance from the file
                    // final File bidibFile = new File(fileName);
                    // final BiDiB bidib = BidibFactory.loadBiDiBFile(bidibFile);
                    //
                    // // we expect a node in the bidib instance
                    // if (bidib == null || bidib.getNodes() == null || bidib.getNodes().getNode().isEmpty()) {
                    // LOGGER.warn("The provided BiDiB data is not valid for node import.");
                    //
                    // throw new IllegalArgumentException("The provided BiDiB data is not valid for node import.");
                    // }

                    // transform the bidib instance to match the target product if possible
                    transformMatchingProduct(parentComponent, node, bidib, lookupService);

                    // only restore the selected parts
                    org.bidib.wizard.utils.NodeUtils
                        .configureFromBiDiB(ConnectionRegistry.CONNECTION_ID_MAIN, nodeService, switchingNodeService,
                            cvDefinitionTreeModelRegistry, node, importParams, bidib, wizardLabelWrapper,
                            (checkRestoreCVs != null ? checkRestoreCVs.isSelected() : false),
                            (checkRestoreFeatures != null ? checkRestoreFeatures.isSelected() : false),
                            (checkRestoreNodeString != null ? checkRestoreNodeString.isSelected() : false),
                            (checkRestoreMacroContent != null ? checkRestoreMacroContent.isSelected() : false),
                            (checkRestoreAccessoryContent != null ? checkRestoreAccessoryContent.isSelected() : false));

                    // save the node labels ....
                    LOGGER.info("Save the node labels.");
                    try {
                        long uniqueId = node.getUniqueId();

                        // save the port labels locally
                        wizardLabelWrapper.saveNodeLabels(uniqueId);
                    }
                    catch (InvalidConfigurationException ex) {
                        LOGGER.warn("Save port labels failed.", ex);

                        String labelPath = ex.getReason();
                        JOptionPane
                            .showMessageDialog(JOptionPane.getFrameForComponent(parentComponent),
                                Resources
                                    .getString(NodeExchangeHelper.class, "labelfileerror.message",
                                        new Object[] { labelPath }),
                                Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                                JOptionPane.ERROR_MESSAGE);
                    }

                    statusBar
                        .setStatusText(
                            String.format(Resources.getString(MainController.class, "importedNodeState"), fileName),
                            StatusBar.DISPLAY_NORMAL);

                    // show a dialog with some checkboxes to allow the user to select all options (port config,
                    // macros, accessories) and move this to saveNode ...
                    SaveNodeConfigurationDialog saveNodeConfigurationDialog =
                        new SaveNodeConfigurationDialog(parentComponent, null, null, node, importParams, true) {
                            private static final long serialVersionUID = 1L;

                            @Override
                            protected void fireContinue(final NodeInterface node) {

                                LOGGER.info("User wants to save the macros to node: {}", node);
                                final Map<String, Object> params = new HashMap<String, Object>();

                                params.put(SAVE_MACROS, isSaveMacros());
                                params.put(SAVE_ACCESSORIES, isSaveAccessories());
                                params.put(SAVE_BACKLIGHTPORTS, isSaveBacklightPorts());
                                params.put(SAVE_LIGHTPORTS, isSaveLightPorts());
                                params.put(SAVE_SERVOPORTS, isSaveServoPorts());
                                params.put(SAVE_SWITCHPORTS, isSaveSwitchPorts());
                                params.put(SAVE_FEATURES, isSaveFeatures());
                                params.put(SAVE_CVS, isSaveConfigurationVariables());

                                try {
                                    parentComponent.setBusy(true);

                                    statusBar
                                        .setStatusText(String
                                            .format(Resources.getString(MainController.class, "writeToNode"), fileName),
                                            StatusBar.DISPLAY_NORMAL);

                                    nodeListActionListener.saveOnNode(node, params);
                                }
                                finally {
                                    statusBar
                                        .setStatusText(String
                                            .format(Resources.getString(MainController.class, "writeToNodeFinished"),
                                                fileName),
                                            StatusBar.DISPLAY_NORMAL);

                                    parentComponent.setBusy(false);
                                }

                                if (params.containsKey(SAVE_ERRORS)) {
                                    LOGGER.warn("The save on node operation has finished with errors!!!");
                                    // show an error dialog with the list of save errors during update node
                                    NodeErrorsDialog nodeErrorsDialog =
                                        new NodeErrorsDialog(parentComponent, null, true);

                                    // set the error information
                                    nodeErrorsDialog
                                        .setErrors(
                                            Resources.getString(MainController.class, "save-values-on-node-failed"),
                                            (List<String>) params.get(SAVE_ERRORS));
                                    nodeErrorsDialog.showDialog();
                                }
                            }

                            @Override
                            protected void fireCancel(final NodeInterface node) {
                                LOGGER.info("User cancelled upload of changes to node: {}", node);
                            }
                        };

                    saveNodeConfigurationDialog.setSaveFeaturesEnabled(checkRestoreFeatures.isSelected());
                    saveNodeConfigurationDialog.setSaveCVsEnabled(checkRestoreCVs.isSelected());

                    saveNodeConfigurationDialog.showDialog();

                    final String workingDir = Paths.get(fileName).getParent().toString();
                    LOGGER.info("Save current workingDir: {}", workingDir);

                    wizardSettings.setWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY, workingDir);
                }
                catch (

                UserActionAbortedException ex) {
                    LOGGER.info("User cancelled the import operation.");
                }
                catch (

                InvalidConfigurationException ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw new RuntimeException(Resources.getString(MainNodeListActionListener.class, ex.getReason()));
                }
                catch (

                IllegalArgumentException ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw ex;
                }
                catch (

                Exception ex) {
                    LOGGER.warn("Load node configuration failed.", ex);

                    // show a message to the user
                    throw new RuntimeException("Load node configuration failed.");
                }
                finally {
                    parentComponent.setBusy(false);
                }
            }
        };
        dialog.showDialog();
    }

    protected void transformMatchingProduct(
        final Component parent, final NodeInterface node, final BiDiB bidib, final LookupService lookupService)
        throws UserActionAbortedException {

        // we expect a node in the bidib instance
        if (bidib == null || bidib.getNodes() == null || bidib.getNodes().getNode().isEmpty()) {
            LOGGER.warn("The provided BiDiB data is not valid for node import.");

            throw new IllegalArgumentException("The provided BiDiB data is not valid for node import.");
        }

        final org.bidib.jbidibc.core.schema.bidib2.Node importNode = bidib.getNodes().getNode().get(0);
        LOGGER.info("Transform the bidib instance, node: {}, schemaNode: {}", node, importNode.getProductName());

        long sourceUniqueId = importNode.getUniqueId();
        long targetUniqueId = node.getUniqueId();

        if (ProductUtils.isProductEqual(targetUniqueId, sourceUniqueId)) {

            // ReadyDMX and OneDMX
            if ((ProductUtils.getVendorId(sourceUniqueId) == 251
                && (ProductUtils.getPid(sourceUniqueId) == 114 || ProductUtils.getPid(sourceUniqueId) == 113))
                || (ProductUtils.getVendorId(sourceUniqueId) == 13 && ProductUtils.getPid(sourceUniqueId) == 118)) {
                // compare the firmware version. Import from version < 2 to node with version >= 2 needs
                // transformation
                final SoftwareVersion importVersion = SoftwareVersion.parse(importNode.getFirmwareRelease());
                if (node.getNode().getSoftwareVersion().isHigherOrEqualThan(SoftwareVersion.build(2, 0, 0))
                    && importVersion.isLowerThan(SoftwareVersion.build(2, 0, 0))) {
                    transformOneDmxToV2(importNode);
                }
            }

            // nothing more to do here
            return;
        }

        if (ProductUtils.getVendorId(sourceUniqueId) == 13 && ProductUtils.getVendorId(targetUniqueId) == 13) {
            // OpenDCC
            if ((ProductUtils.getPid(sourceUniqueId) == 142 && ProductUtils.getPid(targetUniqueId) == 140)
                || (ProductUtils.getPid(sourceUniqueId) == 141 && ProductUtils.getPid(targetUniqueId) == 117)) {
                // OneControl-04-16-48-24 --> OneControl-08-00-48-24
                // OneControl 04-16-32-20 --> OneControl 08-00-32-20
                LOGGER
                    .info(
                        "Found import of OneControl-04-16-48-24 --> OneControl-08-00-48-24 or OneControl 04-16-32-20 --> OneControl 08-00-32-20. Must change the port numbers for all ports with portnumber > 3.");

                int result =
                    JOptionPane
                        .showConfirmDialog(JOptionPane.getFrameForComponent(parent),
                            Resources.getString(NodeExchangeHelper.class, "import-with-transformation.message"),
                            Resources.getString(NodeExchangeHelper.class, "import-with-transformation.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.INFORMATION_MESSAGE);
                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled import operation.");
                    throw new UserActionAbortedException("User cancelled import operation.");
                }

                transformOneControlOneDriveTurn(importNode);
            }
            else if ((ProductUtils.getPid(sourceUniqueId) == 145 && ProductUtils.getPid(targetUniqueId) == 143)
                || (ProductUtils.getPid(sourceUniqueId) == 144 && ProductUtils.getPid(targetUniqueId) == 122)) {
                // OneControl-04-16-48-24 --> OneControl-08-00-48-24
                // OneControl 04-16-32-20 --> OneControl 08-00-32-20
                LOGGER
                    .info(
                        "Found import of OneDriveTurn-04-16-48-24 --> OneDriveTurn-08-00-48-24 or OneDriveTurn 04-16-32-20 --> OneDriveTurn 08-00-32-20. Must change the port numbers for all ports with portnumber > 3.");

                int result =
                    JOptionPane
                        .showConfirmDialog(JOptionPane.getFrameForComponent(parent),
                            Resources.getString(NodeExchangeHelper.class, "import-with-transformation.message"),
                            Resources.getString(NodeExchangeHelper.class, "import-with-transformation.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.INFORMATION_MESSAGE);
                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled import operation.");
                    throw new UserActionAbortedException("User cancelled import operation.");
                }

                transformOneControlOneDriveTurn(importNode);
            }
            else if (ProductUtils.getPid(sourceUniqueId) != ProductUtils.getPid(targetUniqueId)) {

                String sourceProductName = getProductName(lookupService, sourceUniqueId);
                String targetProductName = getProductName(lookupService, targetUniqueId);

                int result =
                    JOptionPane
                        .showConfirmDialog(JOptionPane.getFrameForComponent(parent),
                            Resources
                                .getString(NodeExchangeHelper.class, "import-with-different-pid.message",
                                    sourceProductName, targetProductName),
                            Resources.getString(NodeExchangeHelper.class, "import-with-different-pid.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled import operation.");
                    throw new UserActionAbortedException("User cancelled import operation.");
                }
            }
        }
        else if (ProductUtils.getVendorId(sourceUniqueId) == 251 && ProductUtils.getVendorId(targetUniqueId) == 251) {

            if (ProductUtils.getPid(sourceUniqueId) != ProductUtils.getPid(targetUniqueId)) {

                String sourceProductName = getProductName(lookupService, sourceUniqueId);
                String targetProductName = getProductName(lookupService, targetUniqueId);

                int result =
                    JOptionPane
                        .showConfirmDialog(JOptionPane.getFrameForComponent(parent),
                            Resources
                                .getString(NodeExchangeHelper.class, "import-with-different-pid.message",
                                    sourceProductName, targetProductName),
                            Resources.getString(NodeExchangeHelper.class, "import-with-different-pid.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled import operation.");
                    throw new UserActionAbortedException("User cancelled import operation.");
                }
            }
        }
        else if (ProductUtils.getVendorId(sourceUniqueId) != ProductUtils.getVendorId(targetUniqueId)) {
            if (ProductUtils.getPid(sourceUniqueId) != ProductUtils.getPid(targetUniqueId)) {

                String sourceProductName = getProductName(lookupService, sourceUniqueId);
                String targetProductName = getProductName(lookupService, targetUniqueId);

                int result =
                    JOptionPane
                        .showConfirmDialog(JOptionPane.getFrameForComponent(parent),
                            Resources
                                .getString(NodeExchangeHelper.class, "import-with-different-pid.message",
                                    sourceProductName, targetProductName),
                            Resources.getString(NodeExchangeHelper.class, "import-with-different-pid.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled import operation.");
                    throw new UserActionAbortedException("User cancelled import operation.");
                }
            }
        }
    }

    protected String getProductName(final LookupService lookupService, long sourceUniqueId) {
        String sourceProductName =
            lookupService
                .getProduct(ProductUtils.getVendorId(sourceUniqueId), ProductUtils.getPid(sourceUniqueId))
                .filter(Objects::nonNull).map(pt -> pt.getName())
                .orElse(Resources
                    .getString(NodeExchangeHelper.class, "import-with-different-pid.unknown-product",
                        ProductUtils.getVendorId(sourceUniqueId), ProductUtils.getPid(sourceUniqueId)));
        return sourceProductName;
    }

    protected void transformOneDmxToV2(final org.bidib.jbidibc.core.schema.bidib2.Node importNode) {

        // the overlay mapping needs transformation
        // process the configuration variables
        if (importNode.getConfigurationVariables() != null
            && !importNode.getConfigurationVariables().getConfigurationVariable().isEmpty()) {
            LOGGER.info("Process CV values.");

            Collections.sort(importNode.getConfigurationVariables().getConfigurationVariable(), (cv1, cv2) -> {

                return Integer.compare(Integer.parseInt(cv1.getName()), Integer.parseInt(cv2.getName()));
            });

            LOGGER
                .info("Number of CV values: {}",
                    importNode.getConfigurationVariables().getConfigurationVariable().size());
            if (LOGGER.isDebugEnabled()) {
                importNode
                    .getConfigurationVariables().getConfigurationVariable()
                    .forEach(cv -> LOGGER.debug("Current CV, num: {}, value: {}", cv.getName(), cv.getValue()));
            }

            // move the overlay configuration

            // overlay 0
            // 618 --> 620 transition 0

            // 619 --> 621 DMX channel 1
            // 620 --> 622 brightness 1
            // 621 --> 625 transition 1

            // 622 --> 626 DMX channel 2
            // 623 --> 627 brightness 2
            // 624 --> 630 transition 2

            // 637 --> 651 DMX channel 7
            // 638 --> 652 brightness 7
            // 639 --> 655 transition 7

            // overlay 1
            // 664 --> 656 DMX channel A 0

            // 685 --> 691 DMX channel A 7

            // overlay 2
            // 712 --> 696 DMX channel A 0

            // 733 --> 731 DMX channel A 7

            // overlay 3
            // 760 --> 736 DMX channel A 0

            // 781 --> 771 DMX channel A 7
            // 782 --> 772 brightness A 7
            // ....... 773 --> 0 <-- DMX channel B 7
            // ....... 774 --> 0 <-- brightness B 7
            // 783 --> 775 transition 7

            // ---------------------- 2.02.11 ------------
            // DMX Kanal 1
            // 808 --> 776

            // DMX Kanal 2
            // 834 --> 802

            // DMX Kanal 32
            // 1614 --> 1582
            // 1639 --> 1607

            // Input Mapping
            // 1640 --> 1608
            // ...
            // 1647 --> 1615

            // Internal
            // 1648 --> 1616
            // ...
            // 1657 --> 1625

            // Tools Storage
            // 1710 --> 1678
            // ...
            // 1775 --> 1739

            // we need 2 lists

            final List<ConfigurationVariable> configurationVariables =
                importNode.getConfigurationVariables().getConfigurationVariable();
            final List<ConfigurationVariable> configurationVariablesTarget = new ArrayList<>();

            // transform the overlay reverse --> start from 616
            int startCvSource = 781;
            int startCvTarget = 771;
            for (int overlayIndex = 3; overlayIndex > -1; overlayIndex--) {

                int cvNumberSource = startCvSource;
                int cvNumberTarget = startCvTarget;

                for (int channelIndex = 7; channelIndex > -1; channelIndex--) {
                    LOGGER.info("sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                    // find the source and the target cv and set the value
                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                        (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource + 1,
                        cvNumberTarget + 1, (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource + 2,
                        cvNumberTarget + 4, (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

                    transformCv(configurationVariablesTarget, cvNumberTarget + 2,
                        cvVar -> cvVar.setValue(String.valueOf(0)));

                    transformCv(configurationVariablesTarget, cvNumberTarget + 3,
                        cvVar -> cvVar.setValue(String.valueOf(0)));

                    cvNumberSource -= 3;
                    cvNumberTarget -= 5;
                }

                startCvSource -= 48;
                startCvTarget -= 40;
            }

            // DMX-channel 0..31
            // 808..833 --> 776..801
            // 834..859 --> 802..827
            startCvSource = 808;
            startCvTarget = 776;

            int cvNumberSource = startCvSource;
            int cvNumberTarget = startCvTarget;

            for (int dmxChannelIndex = 0; dmxChannelIndex < 32; dmxChannelIndex++) {

                for (int channelIndex = 0; channelIndex < 26; channelIndex++) {

                    cvNumberSource = startCvSource + channelIndex;
                    cvNumberTarget = startCvTarget + channelIndex;
                    LOGGER.info("DMX-channel, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                    // find the source and the target cv and set the value
                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                        (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));
                }

                startCvSource += 26;
                startCvTarget += 26;
            }

            // input mapping
            startCvSource = 1640;
            startCvTarget = 1608;

            cvNumberSource = startCvSource;
            cvNumberTarget = startCvTarget;

            for (int inputMappingIndex = 0; inputMappingIndex < 8; inputMappingIndex++) {
                cvNumberSource = startCvSource + inputMappingIndex;
                cvNumberTarget = startCvTarget + inputMappingIndex;
                LOGGER.info("input mapping, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                // find the source and the target cv and set the value
                transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                    (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));
            }

            // internal
            startCvSource = 1648;
            startCvTarget = 1616;

            cvNumberSource = startCvSource;
            cvNumberTarget = startCvTarget;

            for (int internalIndex = 0; internalIndex < 10; internalIndex++) {
                cvNumberSource = startCvSource + internalIndex;
                cvNumberTarget = startCvTarget + internalIndex;
                LOGGER.info("internal, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                // find the source and the target cv and set the value
                transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                    (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));
            }

            // Tools Storage
            // 1710 --> 1710
            startCvSource = 1710;
            startCvTarget = 1710;

            cvNumberSource = startCvSource;
            cvNumberTarget = startCvTarget;

            final ConfigurationVariable cvNumberOfConfiguredDimmers = getCv(configurationVariables, cvNumberSource);
            int numberOfConfiguredDimmers = Integer.parseInt(cvNumberOfConfiguredDimmers.getValue());
            LOGGER
                .info("Number of configured dimmers in tools storage: {}, cvNumberSource:{}", numberOfConfiguredDimmers,
                    cvNumberSource);

            LOGGER.info("tools, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

            // find the source and the target cv and set the value
            transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

            cvNumberSource++;
            cvNumberTarget++;

            for (int dmxDimmerIndex = 0; dmxDimmerIndex < DmxSceneryModel.MAX_CONFIGURED_DIMMERS; dmxDimmerIndex++) {

                final ConfigurationVariable cvNumberOfConfiguredChannels =
                    getCv(configurationVariables, cvNumberSource);
                int numberOfConfiguredChannels = Integer.parseInt(cvNumberOfConfiguredChannels.getValue());
                LOGGER
                    .info("Number of configured channels in tools storage: {}, currentChannelCountIndexCv: {}",
                        numberOfConfiguredChannels, cvNumberSource);

                // find the source and the target cv and set the value
                transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                    (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));
                LOGGER.info("tools, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                cvNumberSource++;
                cvNumberTarget++;

                // every channel has 2 CV values
                for (int dmxDimmerChannelIndex =
                    0; dmxDimmerChannelIndex < DmxSceneryModel.MAX_CONFIGURED_CHANNELS; dmxDimmerChannelIndex++) {
                    LOGGER.info("tools, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                    // find the source and the target cv and set the value
                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                        (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

                    cvNumberSource++;
                    cvNumberTarget++;

                    LOGGER.info("tools, sourceCv: {} --> targetCv: {}", cvNumberSource, cvNumberTarget);

                    // find the source and the target cv and set the value
                    transformCv(configurationVariables, configurationVariablesTarget, cvNumberSource, cvNumberTarget,
                        (cvVarSource, cvVarTarget) -> cvVarTarget.setValue(cvVarSource.getValue()));

                    cvNumberSource++;
                    cvNumberTarget++;
                }
            }

            // sort
            Collections.sort(configurationVariablesTarget, (cv1, cv2) -> {
                return Integer.compare(Integer.parseInt(cv1.getName()), Integer.parseInt(cv2.getName()));
            });

            LOGGER.info("configurationVariablesTarget: {}", configurationVariablesTarget);

            // drop all cv values > 615 --> old overlay storage and bigger
            final List<ConfigurationVariable> filteredCVs =
                configurationVariables
                    .stream().filter(cv -> (Integer.parseInt(cv.getName()) < 616)).collect(Collectors.toList());

            LOGGER.info("Number of filtered CV values: {}", filteredCVs.size());

            configurationVariables.clear();
            configurationVariables.addAll(filteredCVs);
            configurationVariables.addAll(configurationVariablesTarget);

            LOGGER.info("Final number of CV values: {}", configurationVariables.size());
        }
    }

    private ConfigurationVariable getCv(final List<ConfigurationVariable> configurationVariables, int cvNumberSource) {
        // find the source and the target cv and set the value
        ConfigurationVariable cvVarSource =
            configurationVariables
                .stream().filter(cv -> cv.getName().equals(String.valueOf(cvNumberSource))).findFirst()
                .orElseThrow(() -> new IllegalArgumentException("No CV available, cvNumber: " + cvNumberSource));

        return cvVarSource;
    }

    private void transformCv(
        final List<ConfigurationVariable> configurationVariables,
        final List<ConfigurationVariable> configurationVariablesTarget, int cvNumberSource, int cvNumberTarget,
        BiConsumer<ConfigurationVariable, ConfigurationVariable> action) {
        // find the source and the target cv and set the value
        ConfigurationVariable cvVarSource =
            configurationVariables
                .stream().filter(cv -> cv.getName().equals(String.valueOf(cvNumberSource))).findFirst()
                .orElseThrow(() -> new IllegalArgumentException("No CV available, cvNumber: " + cvNumberSource));

        // create a clone
        ConfigurationVariable cvVarTarget =
            new ConfigurationVariable()
                .withName(String.valueOf(cvNumberTarget)).withDescription(cvVarSource.getDescription())
                .withDefaultValue(cvVarSource.getDefaultValue()).withValue(cvVarSource.getValue())
                .withTimeout(cvVarSource.isTimeout());
        configurationVariablesTarget.add(cvVarTarget);
        action.accept(cvVarSource, cvVarTarget);
    }

    private void transformCv(
        final List<ConfigurationVariable> configurationVariablesTarget, int cvNumberTarget,
        Consumer<ConfigurationVariable> action) {
        // create a clone
        ConfigurationVariable cvVarTarget = new ConfigurationVariable().withName(String.valueOf(cvNumberTarget));
        configurationVariablesTarget.add(cvVarTarget);

        action.accept(cvVarTarget);
    }

    protected void transformOneControlOneDriveTurn(final org.bidib.jbidibc.core.schema.bidib2.Node importNode) {

        // process the ports
        if (importNode.getPorts() != null) {
            importNode.getPorts().getPort().stream().map(port -> {
                LOGGER.info("Process port: {}", port);
                if (port.getNumber() > 3) {
                    LOGGER.info("Port: change port number from {} to {}", port.getNumber(), port.getNumber() + 4);
                    port.setNumber(port.getNumber() + 4);
                }
                return port;
            }).collect(Collectors.toList());
        }
        else {
            LOGGER.warn("No ports available.");
        }

        // process the macros
        if (importNode.getMacros() != null) {
            importNode.getMacros().getMacro().forEach(macro -> {
                if (macro.getMacroPoints() != null) {
                    macro
                        .getMacroPoints().getMacroPoint().stream()
                        .filter(mp -> mp instanceof MacroPointOutput || mp instanceof MacroPointInput).map(mp -> {
                            if (mp instanceof MacroPointOutput) {
                                MacroPointOutput mpo = (MacroPointOutput) mp;
                                if (mpo.getOutputNumber() > 3) {
                                    LOGGER
                                        .info("Macro: change port number from {} to {}", mpo.getOutputNumber(),
                                            mpo.getOutputNumber() + 4);
                                    mpo.setOutputNumber(mpo.getOutputNumber() + 4);
                                }
                            }
                            else if (mp instanceof MacroPointInput) {
                                MacroPointInput mpi = (MacroPointInput) mp;
                                if (mpi.getInputNumber() > 3) {
                                    LOGGER
                                        .info("Macro: change input port number from {} to {}", mpi.getInputNumber(),
                                            mpi.getInputNumber() + 4);
                                    mpi.setInputNumber(mpi.getInputNumber() + 4);
                                }
                            }
                            return mp;
                        }).collect(Collectors.toList());
                }
            });
        }
        else {
            LOGGER.warn("No macros available.");
        }

        // process the configuration variables
        if (importNode.getConfigurationVariables() != null
            && !importNode.getConfigurationVariables().getConfigurationVariable().isEmpty()) {
            LOGGER.info("Process CV values.");

            Collections.sort(importNode.getConfigurationVariables().getConfigurationVariable(), (cv1, cv2) -> {

                return Integer.compare(Integer.parseInt(cv1.getName()), Integer.parseInt(cv2.getName()));
            });

            LOGGER
                .info("Number of CV values: {}",
                    importNode.getConfigurationVariables().getConfigurationVariable().size());
            if (LOGGER.isDebugEnabled()) {
                importNode
                    .getConfigurationVariables().getConfigurationVariable()
                    .forEach(cv -> LOGGER.debug("Current CV, num: {}, value: {}", cv.getName(), cv.getValue()));
            }

            // drop all cv values >= 10_000
            final List<ConfigurationVariable> filteredCVs =
                importNode
                    .getConfigurationVariables().getConfigurationVariable().stream()
                    .filter(cv -> Integer.parseInt(cv.getName()) < 10_000).collect(Collectors.toList());

            LOGGER.info("Number of filtered CV values: {}", filteredCVs.size());

            importNode.getConfigurationVariables().getConfigurationVariable().clear();
            importNode.getConfigurationVariables().getConfigurationVariable().addAll(filteredCVs);
        }
    }

    public void exportNode(
        final DefaultBusyFrame parentComponent, final NodeInterface node, final SettingsService settingsService,
        final NodeService nodeService, final SwitchingNodeService switchingNodeService,
        final WizardLabelWrapper wizardLabelWrapper, final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry,
        final FileFilter nodeFilter, final StatusBar statusBar, String lang) {

        LOGGER.info("export node: {}", node);

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();

        String fileName = prepareFileName(wizardSettings, node);

        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY);

        // export node data
        FileDialog dialog =
            new FileDialog(parentComponent, FileDialog.SAVE, storedWorkingDirectory, fileName, nodeFilter) {

                private JCheckBox checkLoadCVs;

                @Override
                public void approve(String fileName) {

                    try {
                        final ExportNodeOptionsDialog exportNodeOptionsDialog =
                            new ExportNodeOptionsDialog(parentComponent,
                                Resources.getString(NodeExchangeHelper.class, "exportOptions.title"), node) {

                                private static final long serialVersionUID = 1L;

                                @Override
                                protected Component getAdditionalPanel() {
                                    // prepare a panel with checkboxes for loading macro content before export
                                    JPanel additionalPanel = new JPanel();
                                    additionalPanel.setLayout(new BoxLayout(additionalPanel, BoxLayout.PAGE_AXIS));

                                    checkLoadCVs =
                                        new JCheckBox(Resources.getString(MainController.class, "checkLoadCVs"), true);
                                    additionalPanel.add(checkLoadCVs);
                                    return additionalPanel;
                                }
                            };
                        exportNodeOptionsDialog.setVisible(true);

                        if (exportNodeOptionsDialog.getResult() == JOptionPane.OK_OPTION) {
                            LOGGER.info("User confirmed options.");
                        }
                        else {
                            LOGGER.info("User cancelled options.");

                            return;
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Show exportNodeOptionsDialog failed.", ex);
                        return;
                    }

                    try {
                        parentComponent.setBusy(true);
                        LOGGER.info("Save node state for node: {}, fileName: {}", node, fileName);

                        // must load missing data from node
                        org.bidib.wizard.utils.NodeUtils
                            .loadDataFromNode(ConnectionRegistry.CONNECTION_ID_MAIN, nodeService, switchingNodeService,
                                node, true, checkLoadCVs.isSelected());

                        final BiDiB bidib =
                            prepareBiDiB(node, cvDefinitionTreeModelRegistry, lang, !checkLoadCVs.isSelected(),
                                wizardLabelWrapper);
                        LOGGER.info("Export node, converted node to bidib: {}", bidib);

                        BidibFactory.saveBiDiB(bidib, fileName, false);

                        LOGGER.info("Save node state passed, fileName: {}", fileName);

                        final String workingDir = Paths.get(fileName).getParent().toString();
                        LOGGER.info("Save current workingDir: {}", workingDir);

                        wizardSettings.setWorkingDirectory(WORKING_DIR_NODE_EXCHANGE_KEY, workingDir);

                        statusBar
                            .setStatusText(
                                String.format(Resources.getString(MainController.class, "exportedNodeState"), fileName),
                                StatusBar.DISPLAY_NORMAL);
                    }
                    finally {
                        parentComponent.setBusy(false);
                    }
                }
            };
        dialog.showDialog();
    }

    public BiDiB prepareBiDiB(
        final NodeInterface node, final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry, String lang,
        boolean skipCvValues, final WizardLabelWrapper wizardLabelWrapper) {
        final CvContainer cvContainer = cvDefinitionTreeModelRegistry.getCvContainer(node.getUniqueId());

        final Map<String, CvNode> cvNumberToNodeMap =
            cvContainer != null ? cvContainer.getCvNumberToNodeMap() : new HashMap<>();

        final BiDiB bidib =
            org.bidib.wizard.utils.NodeUtils
                .convertToBiDiB(node, cvNumberToNodeMap, lang, skipCvValues, wizardLabelWrapper);
        LOGGER.debug("prepareBiDiB, converted node to bidib: {}", bidib);
        return bidib;
    }

    public static String prepareFileName(final WizardSettingsInterface wizardSettings, final NodeInterface node) {

        String nodeLabel = org.bidib.wizard.utils.NodeUtils.prepareLabel(node);

        boolean nodeExportAppendDateEnabled = wizardSettings.isNodeExportAppendDateEnabled();

        if (nodeExportAppendDateEnabled) {
            LocalDateTime now = LocalDateTime.now();
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("_yyyyMMdd_HHmmss");
            String currentDateTime = now.format(formatter);

            nodeLabel = nodeLabel + currentDateTime;
        }

        LOGGER.info("export node, prepared nodeLabel as filename: {}", nodeLabel);

        if (!FileUtils.isPathValid(nodeLabel) || !FileUtils.isFilenameValid(nodeLabel)) {
            LOGGER.warn("The default filename is not a valid filename: {}", nodeLabel);
            nodeLabel = FileUtils.escapeInvalidFilenameCharacters(nodeLabel, "_");
            LOGGER.info("export node, prepared escaped filename: {}", nodeLabel);
        }

        return nodeLabel + "." + NodeExchangeHelper.NODE_X_EXTENSION;
    }

}
